;***************************************************************************************
;
;  soft5as8.s 
;     This module duplicates Altair 8" drive operation using a 5.25" drive
;     and soft sectored media. The 5.25" must have at least 77 tracks and
;     be able to rotate at 360rpm. The Teac 55GFR drives work well for this.
;     On the IDC-34 connector, the pin used for the 4th drive select is
;     not standardized. The Teac uses pin 6. On the IDC-34 adapter board,
;     jumper IDC-34 pin 6 to IDC-50 pin 32 to use the 4th drive.
;
;  Version History
;    1.0   01/23/15  M. Douglas
;	- Original
;
;    1.1   02/23/16  M. Douglas 
;	- Support user-selectable stepping rates.
;
;    1.2   04/24/20  M. Douglas	
;	- Fix output compares to work around a problem similar to that described
;	  in the 24FJ errata.
;	- Run all ISRs at the same priority so no ISR nesting occurs.
;	- Do not clear internal head load and interrupt enable status when index
;	  sync is lost.
;
;    1.3   06/19/20  M. Douglas
;	- Remove data written by the 8080 to the output data register even if
;	  a write is not in progress so the EPMP won't lock up.
;	- Change rotation time thresholds to +/-2% instead of 1%. 
;	- Change motor off delay from 1.3s to 5.3s
;	- Protect motor off test with DISI
;	- Redundant de-assert of HEAD_STATUS in motor off removed
;	- Restart the motor time when drive select occurs
;	- Default to 3ms step instead of 10.5ms (in common.s)
;	- Selection of same drive that is already selected no longer does a
;	  de-select before select.
;	- During writes, leave MOVE HEAD status de-asserted until end of trim
;	  erase period like the original FDC.
;	- Clear out all EPMP flags at the end of initialization - immediately
;	  before entering the idle loop - to reset any possible glitches caused
;	  by 8080 CPU start up.
;
;    1.4   06/11/21  M. Douglas
;	- Support drives that suppress index pulses during sync. The "reSync"
;	  loop is entered as soon as a missing index pulse is detected. Normal
;	  sector output is resumed once a full revolution at the right speed
;	  is validated.
;	- The head status bit in the status register is no longer immediately
;	  set with the head load command, but delayed by 50ms as with the normal
;	  8" drive types.
; 	- In the endSector routine, the output compare time for the start of the
;	  next sector is compared to the output compare clock to ensure it is
;	  greater (if it isn't, the OC won't trigger for a full timer wrap). If 
;	  not greater, reSync is entered.
;
;    1.5   06/27/21  M. Douglas
;	Improve drive speed tolerance with the following changes:
;	- Exit to resync if a rotation exceeds +/-100us from average.
;	- Remain in resync loop until a rotation is withing +/-100us of its own
;	  average. While in resync, allow rotations within +/-2% of nominal to
;	  be entered into the running average.
;
;--------------------------------------------------------------------------------------
; 
;  PIC H/W Assignments
;
;     EPMP (PMP) handles port I/O from the 8080
;     UART2 generates the step signal
;     UART3 generates the step direction signal
;     SPI1 generates write data pulses
;     OC1 is the head step timer
;     OC2 is the head settle timer
;     OC4 is sector timer (sector pulse, sector true, read clear, write clear, etc.)
;     IC1 captures read data pulses
;     IC2 captures sector/index pulses 
;     CN68 captures changes in the PINTE line (interrupts enabled) from the 8080
;     TMR4 at 250khz is common sync source for OC4 and IC2, and clock source for OC1-OC3
;
;***************************************************************************************
	.include 	"common.inc" 

; Debug equates. Set to 1 to enable, 0 to disable.

	.equ	actionCodes,0	;displays codes for various actions in this code
	.equ	s3Debug,0		;debug signal output on switch 3 input for scope
 
; The following working registers have permanent variable assignments and cannot
;   be used for any other purpose. 

	.equ	pDriveStatus,w14	;pointer to PMDOUT1L
	.equ	pSectorPos,w13	;pointer to PMDOUT1H
	.equ	pLATB,w12		;pointer to PORTB (LATB)

	.equ	wFFFF,w11		;contains 0xffff
	.equ	wAllDriveOff,w10	;all drive signals off for PORTB
	.equ	wDsInitValue,w9	;drive status initial value
	.equ	wMinDataZero,w8	;minimum time for data to be a zero
	.equ	wPrevHole,w7	;input capture at previous index/sector hole
	.equ	wPrevData,w6	;input capture at previous data pulse

; Map PMDIN1/2 and PMDOUT1/2 to more logical names

	.equ	driveStatus,PMDOUT1L	;drive status IN 08
	.equ	selectCmd,PMDIN1L	;drive select OUT 08
	.equ	sectorPos,PMDOUT1H	;sector position register IN 09
	.equ	driveCmd,PMDIN1H	;drive command OUT 09
	.equ	readData,PMDOUT2L	;read data from disk IN 10
	.equ	writeData,PMDIN2L	;write data to disk OUT 10

	.equ	commandRegs,PMDIN1	;select and drive command registers
	.equ	statusRegs,PMDOUT1	;drive status and sector position as word

; Parallel port (PMP) Status bits

	.equ	pmDRIVE_STATUS,0	;drive status register read
	.equ	pmSECTOR_POS,1	;sector position reigster read
	.equ	pmREAD_DATA,2	;read data register read
	.equ	pmSELECT_CMD,8	;drive select register written
	.equ	pmDRIVE_CMD,9	;drive command register written
	.equ	pmWRITE_DATA,10	;write data register written
	
; Drive Status Register Equates. These are all asserted true as zeros. Bits 3 and 4
;    remain zero when the drive is enabled. We use bit 3 as a drive ready flag.

	.equ	dsENWD,0		;0=enter new write data flag (write)
	.equ	dsMOVE_HEAD,1	;0=OK to move head
	.equ	dsHEAD_STATUS,2	;0=head loaded and settled
	.equ	dsREADY,3		;0=drive ready (not official)
	.equ	dsINTE,5		;0=8080 interrupts enabled
	.equ	dsTRACK0,6		;0=on track zero
	.equ	dsNRDA,7		;0=new read data available (read)
	.equ	dsINIT_VALUE,0xe5	;move head asserted, b4, b3 cleared

; Sector Position Register Equates. Sector True asserts as zero.

	.equ	spSECTOR_TRUE,0	;bit 0 is sector true bit
	.equ	spSECTOR_TRUE_MASK,0xc0	;required bits with sector true asserted
	.equ	spSECTOR_INIT,0xc1	;sets required bits around sector number

; Select Command Register Equates

	.equ	scDRIVE_MASK,0x0f	;mask to get drive number bits alone
	.equ	scDESELECT,7	;bit number of deselect bit

; Drive Command Register Equates

	.equ	dcSTEP_IN,0		;step head towards center
	.equ	dcSTEP_OUT,1	;step head out towards edge
	.equ	dcHEAD_LOAD,2	;load head for read/write
	.equ	dcHEAD_UNLOAD,3	;unload head
	.equ	dcINT_ENABLE,4	;enable interrupts at start of sector
	.equ	dcINT_DISABLE,5	;disable interrupts from the FDC
	.equ	dcHEAD_CURR_SW,6	;no used
	.equ	dcWRITE_ENABLE,7	;initiate write sequence
	.equ	dcHEAD_MASK,0x0f	;get all head step/load bits at once
	.equ	dcMOVE_MASK,0x03	;get both step in/out bits at once

; Drive interface equates. Outputs are all asserted low (not inverted from processor
;   to the drive). Inputs are asserted high (inverted from drive to processor).

	.equ	B0_DRIVE_SELECT1,0
	.equ	B1_DRIVE_SELECT2,1
	.equ	B2_DRIVE_SELECT3,2
	.equ	B3_DRIVE_SELECT4,3
	.equ	B6_WRITE_DATA,6
	.equ	B7_STEP_DIRECTION,7
	.equ	B8_IN_USE,8		;output, not used
	.equ	B9_WRITE_GATE,9
	.equ	B11_MOTOR_ON,11
	.equ	B12_SIDE_SELECT,12	;output, not used
	.equ	B13_WRITE_CURRENT,13	;output, not used
	.equ	F4_STEP,4	
	.equ	D10_READ_DATA,10
	.equ	D0_WRITE_PROTECT,0	;input, not used
	.equ	D1_READY,1
	.equ	D2_TRACK0,2
	.equ	D3_INDEX,3
	.equ	F1_TRUE_READY,1	;input, not used
	.equ	B_DESELECT,0x3bcf	;all drive signals off port B

; Head step and load equates. OC1 and OC2 are used to time head step and head load. The head
;    load time is an equate here. The step time is a user-settable parameter ptStepT58. The
;    timebase used for OC1 and OC2 is 250KHz (4us per tick) and comes from Timer 4. UART2 and
;    UART3 are used to send step and direction pluses.

	.equ	HEAD_SETTLE_TIME,50000/4  ;head settle time, 50ms at 250khz
	.equ	STEP_CHAR,0xfe	;character when sent serially generates a 2us
				;pulse at 1Mb/s (start bit and bit 0 low)
	.equ	DIR_IN_CHAR,0xf0	;character when sent serially generates a 5us
				;pulse at 1Mb/s (start bit and four 0's)

; Sector/index equates. Sector/index pulses are captured by IC2. IC2 and OC4 are
;    sync'd to TMR4 which runs full-wrap at 250khz. The end of sector time (EOS_TIME)
;    is halfway between the end of written data and the start of the next sector.

	.equ	INDEX_TIME,166668/4	;166.667ms full revolution time(68 is right)
	.equ	INDEX_MAX,170000/4	;102% of full revolution time
	.equ	INDEX_MIN,163332/4	;98% of full revolution time
	.equ	MAX_DEV,100/4	;100us max deviation from average rev time

	.equ	SECTOR_TIME,5208/4	;5.208ms sector time
	.equ	SEC_TRUE_TIME,32/4	;32us (extra 2us given)
	.equ	READ_CLEAR_TIME,212/4	;read data ignored for 1st 212us
	.equ	WRITE_CLEAR_TIME,388/4	;write data starts after 388us
	.equ	EOS_TIME,SECTOR_TIME-(SECTOR_TIME-WRITE_CLEAR_TIME-(138*32/4))/2
	.equ	SECTOR_WRAP,0x1f	;mask to wrap sector count
	.equ	TRIM_ERASE_TIME,500/4	;ends at 500us into next sector

; Read and write data equates. Read data pulses are captured by IC1. The timebase for
;    IC1 is 16Mhz. For FM encoding, the bit time is 4us. If no pulse occurs between
;    two 4us pulses, the data bit is a zero. If a pulse is present in between
;    two 4us pulses, the data bit is a one. Writing of data is done with SPI1 at 
;    500ns per SPI bit.

	.equ	MIN_DATA_ZERO,44	;2.75us (44/16Mhz) or more is taken as 4us
	.equ	WRITE_ZEROS,0x7f7f	;two bits of zero

; flags variable

	.equ	fWRITE_ENABLE,0	;1=write enable command from 8080
	.equ	fENTER_INDEX,1	;1=exit cmd interrupt into enterIndex
	.equ	fHEAD_LOADED,2	;1=head loaded due to 8080 command
	.equ	fINTS_ENABLED,3	;1=sector interrupts enabled
	.equ	fSECTOR_ZERO,4	;1=next sector pulse should be sector zero
	.equ	fDATA_ONE,5		;1=next data bit is a one
	.equ	fDATA_SYNC,6	;1=have seen data sync bit
	.equ	fTRIM_ERASE,7	;1=in trim erase period
	.equ	fINDEX_PULSE,8	;1=index pulse received
	.equ	fRESYNC_MASK,(1<<fHEAD_LOADED)+(1<<fINTS_ENABLED)

; timingTable table offsets. A 32 bit sum of full revolution time and a 16 bit
;    average are maintained for each drive. LSB is 4us like TMR4 and OC4.

	.equ	IDX_SUM_MSW,0	;offset of MSW of 32 bit sum
	.equ	IDX_SUM_LSW,2	;offset of LSW of 32 bit sum
	.equ	IDX_AVERAGE,4	;offset of 16 bit average
	.equ	IDX_LEN,6		;6 bytes bytes per drive entry
	.equ	SAMPLE_CNT,4	;averaging 4 samples

; Misc equates

	.equ	NUM_DRIVES,4	;four drives supported. Fourth drive
				;requires jumper on adapter board from
				;IDC-50 p32 to IDC-34 p6
	.equ	PIC_INTS_OFF,0xe0	;write to SR to disable PIC interrupts

; Variables

	 .section *,bss,near
flags:	 .space	2		;see flag bits above
sector:	 .space	2		;current sector number
shiftReg:	 .space	2		;shift register when receiving a byte
bitCount:	 .space	2		;bit counter in byte
sectorStart: .space	2		;value of OC4TMR at start of sector
motorTimer:	 .space	2		;counts sectors until 256 bit set (1.3s)
curDrive:	 .space	2		;current drive number 0-2
sectF7Lsw:	 .space	2		;sector time with 7 bits of fraction
sectF7Msw:	 .space	2
oc4F7Lsw:	 .space	2		;OC4R value with 7 bits of fraction
oc4F7Msw:	 .space	2
timingTable: .space	IDX_LEN*NUM_DRIVES	;index timing sums, averages for all drives

	
;****************************************************************************************
;  
;   Soft5as8 - entry point for 8" drive implemented on soft sectored 5.25" drive
;
;****************************************************************************************
	.text
	.global	Soft5as8

Soft5as8:	mov.w	#STACK_ADDR,w15	;init stack pointer
	mov.w	#PIC_INTS_OFF,w0	;disable all interrupts
	mov.w	w0,SR
;
; Initialize the parallel port to handle the I/O IN and OUT instructions from the 8080. The
;     PMP is run in addressable slave mode where I/O addresses from the 8080 map as shown
;     below.
;
;     I/O address            PIC register
;	 8	PMDOUT1, PMDIN1 low byte
;	 9	PMDOUT1, PMDIN1 high byte
;	10	PMDOUT2, PMDIN2 low byte
;	11	PMDOUT2, PMDIN2 high byte (not used)

	mov.w	#0x0281,w0		;enhanced PSP mode, CS1 on PMA14, interrupts on
	mov.w	w0,PMCON1
	mov.w	#0xc000,w0		;PMWR, PMRD enabled
	mov.w	w0,PMCON3
	mov.w	#0x4003,w0		;CS1(A14), A1, A0 enabled
	mov.w	w0,PMCON4
	mov.w	#0x6000,w0		;CS1 on and active high, RD, WR active low
	mov.w	w0,PMCS1CF
	mov.w	#0x0001,w0		;use TTL buffers
	mov.w	w0,PADCFG1
	mov.w	#0xffff,w0		;set both 8080 status registers to 0xff
	mov.w	w0,statusRegs
	bset.w	PMCON1,#15		;enable EPMP
 
; Map UART2 transmit to the step line and UART3 to the step direction line. Receive
;    data is not used, but if left mapped to the default RP63, receive data is low
;    which looks like a continuous break. It is mapped here to RP18 (Port B, bit 5,
;    ICSP Clock) which initialization sets as an input with internal pullup. This pin
;    remains unused and high in normal operation.

	mov.b	#18,w0		;UART Receive data tied high on RP18
	mov.b	WREG,RPINR19L	;UART 2
	mov.b	WREG,RPINR17H	;UART 3

	mov.b	#5,w0		;UART 2 TX code 5 is on RP10
	mov.b	WREG,RPOR5L
	mov.b	#28,w0		;UART 3 TX code 28 is on RP7
	mov.b	WREG,RPOR3H

; Initialize UART2 and UART3 for 8N1 at 1M baud (1us per bit)

	mov.w	#0x0008,w0		;serial port off, 8N1, BRGH=1
	mov.w	w0,U2MODE
	mov.w	w0,U3MODE
	mov.w	#3,w0		;1M baud
	mov.w	w0,U2BRG
	mov.w	w0,U3BRG
	bset.w	U2MODE,#UARTEN	;UART on
	bset.w	U2STA,#UTXEN	;transmitter on too
	bset.w	U3MODE,#UARTEN	;UART on
	bset.w	U3STA,#UTXEN	;transmitter on too

; Init TMR4 to use a 250khz timebase. This timer is the sync source for the
;   sector/index pulse capture (IC2) and the sector timer (OC4). TMR4 is
;   also the clock source (not sync source) for OC1-OC3. The timer is enabled
;   later after IC2 and OC4 are initialized.

	mov.w	#0x0020,w0		;Fcy/64 for 250khz
	mov.w	w0,T4CON
	mov.w	#0xffff,w0
	mov.w	w0,PR4

; Configure OC1 to time head step delays, OC2 to time the head load delay.
;   TMR4 at 250khz is used as clock source. 

	mov.w	#0x0020,w0		;free running, no output pin
	mov.w	w0,OC1CON2
	mov.w	w0,OC2CON2
	mov.w	#0x0803,w0		;TMR4 and one-shot toggle mode
	mov.w	w0,OC1CON1
	mov.w	w0,OC2CON1

;  Map IC2 to the index line. Configure to capture on rising edge (falling edge
;    from the drive is inverted). IC2 is sync'd to TMR4 running at 250khz.

	mov.b	#22,w0		;IC2 detects index pulses on RP22
	mov.b	WREG,RPINR7H
	mov.w	#0x000e,w0		;IC2 sync'd to the OC4 timer
	mov.w	w0,IC2CON2
	mov.w	#0x0803,w0		;rising edge capture, TMR4 is clk source
	mov.w	w0,IC2CON1	

; Configure OC4 as the sector timer to time sector true, read clear, write clear, etc.
;    Both OC4 and IC2 are sync'd to TMR4.

	mov.w	#0x002e,w0		;no external sync, OC pin output not used
	mov.w	w0,OC4CON2
	mov.w	#0x0803,w0		;TMR4 clk source, continuous compare
	mov.w	w0,OC4CON1		;turn on OC4
	bset.w	T4CON,#TON		;turn on TMR4 now that IC2 and OC4 ready

;  Map IC1 to the read data line. Configure to capture on rising edge using the system clock.

	mov.b	#3,w0		;IC1 detects read data pulses on RP3
	mov.b	WREG,RPINR7L
	clr.w	IC1CON2		;IC1 not sync'd to any other source
	mov.w	#0x1c03,w0		;IC1 capture on rising edge, Fcy clock
	mov.w	w0,IC1CON1	
	
;  SPI1 is used to generate the write data waveform. Initialize for 16 bits at 2Mhz.
;     This will send two bits of drive data per 16 bits of SPI data. The SPI is set
;     to interrupt when the FIFO is empty (16 bit word still in shift register).

	mov.b	#7,w0		;SPI1 Data out is on RP6
	mov.b	WREG,RPOR3L		;assign SPI1 data out to disk write data

	mov.w	#0x0001,w0		;select enhanced mode (8 word buffer)
	mov.w	w0,SPI1CON2
	mov.w	#0x1423,w0		;no clk or data out, 16 bit, master, 2MHz
	mov.w	w0,SPI1CON1
	mov.w	#0x0018,w0		;interrupt when fifo empty
	mov.w	w0,SPI1STAT
	bset.w	SPI1STAT,#SPIEN	;enable SPI
	mov.w	#0xffff,w0		;write all 1's so line is idling high
	mov.w	w0,SPI1BUF

; Set the PINTE signal from the 8080 bus to be an input change interrupts.

	bset.w	CNEN5,#CN68IE	;PINTE signal is CN68

; Put our local ISRs in the interrupt table

	mov.w	#tbloffset(ioPortInt),w0       ;PMP interrupt (8080 I/O)
	mov.w	w0,PMPInterrupt
	mov.w	#tbloffset(headMoveInt),w0     ;OC1 interrupt (head move timer)
	mov.w	w0,OC1Interrupt
	mov.w	#tbloffset(headStatusInt),w0   ;OC2 interrupt (head status timer)
	mov.w	w0,OC2Interrupt
	mov.w	#tbloffset(changeInt),w0       ;CN interrupt (PINTE)
	mov.w	w0,CNInterrupt

; Initialize permanent pointers and values present in some registers

	mov.w	#driveStatus,pDriveStatus    ;permanent pointers
	mov.w	#sectorPos,pSectorPos
	mov.w	#LATB,pLATB

	mov.w	#0xffff,wFFFF	     ;permanent values
	mov.w	#B_DESELECT,wAllDriveOff 
	mov.w	#dsINIT_VALUE,wDsInitValue 
	mov.w	#MIN_DATA_ZERO,wMinDataZero

; Initialize variables

	clr.w	flags		;all flags false
	clr.w	sector		;init sector to zero just to be safe
	clr.w	curDrive		;init current drive number just to be safe

; Initialize the 32 bit sample sum and average of index time (full rotation)

	mov.w	#((SAMPLE_CNT*INDEX_TIME)>>16),w0	   ;w0 = MSW of sum
	mov.w	#((SAMPLE_CNT*INDEX_TIME)&0xffff),w1   ;w1 = LSW of sum
	mov.w	#INDEX_TIME,w2	;w2 = average of 32 bit sum
	mov.w	#timingTable,w3	;w3->index timing table
	mov.w	#NUM_DRIVES,w4	;w4 = loop count

initIdxTbl:	mov.w	w0,[w3+IDX_SUM_MSW]	;init MSW of sum
	mov.w	w1,[w3+IDX_SUM_LSW]	;init LSW of sum
	mov.w	w2,[w3+IDX_AVERAGE]	;init average
	add.w	#IDX_LEN,w3		;move to next table entry
	dec.w	w4,w4
	bra	nz,initIdxTbl
	
    .if (s3Debug)  			;use switch 3 input as an output signal
	mov.w	#0x0000,w0		;port c init
	mov.w	w0,LATC
	mov.w	#0xb000,w0		;port c direction - bit 14 as output
	mov.w	w0,TRISC		;MAKE SURE DRIVE TYPE BIT 3 (SWITCH 4) IS OFF
    .endif

; Sample use of s3 as output for scope

    .if s3Debug
	bclr.w	LATC,#C14_DRV_TYPE3_SW
    .endif

; Enable interrupts and run

	mov.w	PMDIN1,w0		;ensure EPMP is completely reset
	mov.w	PMDIN2,w0
	mov.w	wFFFF,PMDOUT1	;driveStatus and sectorPos to 0xff
	clr.w	PMDOUT2		;data and IO stats zero
	bclr.w	PMSTAT,#IBOV	;make sure errors are cleared
	bclr.w	PMSTAT,#OBUF

	bset.w	IEC2,#PMPIE		;enable parallel port interrupt
	bset.w	IEC1,#CNIE		;enable ready and PINTE line interrupts
				;fall into idle loop

;**********************************************************************************
;
;  Idle loop - do nothing while waiting for the command processing interrupt
;     to select a drive and change the execution point to index verification.
;
;**********************************************************************************

;  enterIdle - reinitialize the stack pointer and reset interrupt level to zero. 
;      This routine is typically entered upon drive selection in the command
;      processing ISR.

enterIdle:	disi	#5		;disable interrupts for next 4 cycles
	mov.w	wFFFF,statusRegs	;driveStatus and sectorPos to 0xff
	mov.w	wAllDriveOff,LATB	;disable all drive signals
	bclr.w	LATD,#D8_INTERRUPT_OUT	;de-assert 8080 interrupt
	clr.w	flags		;set all flags to false

	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	clr.w	SR		;set interrupt level to zero (no interrupt)
     .if actionCodes
	mov.b	#'Z',w1		;Z - Zzzzzz (idle state)
	mov.w	w1,U1TXREG		;transmit the character
     .endif
	
; Idle loop - allow escape to monitor with 'X' or 'x' if monitor switch is on

idleLoop:	btst.w	U1STA,#URXDA	;test for a character
	bra	z,idleLoop
	mov.w	U1RXREG,w0		;get the character
	btst.w	PORTC,#C13_MONITOR_SW	;monitor enabled?
	bra	z,idleLoop		;no
	and.w	#0x5f,w0		;convert lower case to upper
	sub.b	#'X',w0		;exit requested?
	bra	nz,idleLoop		;no
	reset			;get back to boot loader

;**********************************************************************************
;
;  Verify Index Sync loop - This loop is entered when the command processing
;      ISR selects a drive and the drive is ready, or when sector processing
;      determines index sync has been lost. This loop waits for the drive
;      to get up to speed and then enters the main sector loop.
;
;**********************************************************************************

;  enterIndex, reSync - reinitialize the stack pointer and restore interrupt
;      level to zero. This routine is directly entered from either the I/O port
;      ISR (drive selected) or the sector processing loop when it is determined
;      sync has been lost. 

enterIndex:	clr.w	w0		;mask to clear all flags
	bra	1f
reSync:	mov.w	#fRESYNC_MASK,w0	;leave head load and int flags alone
1:	disi	#3		;disable interrupts for next 3 cycles
	and.w	flags		;clear flags based on w0 mask
	mov.b	wFFFF,[pSectorPos]	;sector position register disabled
	bclr.w	LATD,#D8_INTERRUPT_OUT	;de-assert 8080 interrupt
headLoaded:	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	clr.w	SR		;set interrupt level to zero (no interrupt)
     .if actionCodes
	mov.b	#'v',w1		;v - verifying, not verified yet
	mov.w	w1,U1TXREG		;transmit the character
     .endif

; Clean out buffered index sector input-captures, then get a first pulse capture
;    as a starting point for subsequent captures. Init w3 to point to the
;    timingTable entry for the current drive.

clearIC2:	mov.w	IC2BUF,w0		;read a capture value
	btst.w	IC2CON1,#ICBNE	;empty yet?
	bra	nz,clearIC2		;no
	rcall	nextHole		;wait for a pulse
	
	mov.w	curDrive,w2		;w2=current drive number
	mul.uu	w2,#IDX_LEN,w2	;index into timingTable table by drive num
	mov.w	#timingTable,w3	;w3->timingTable
	add.w	w2,w3,w3		;w3->row in timingTable for current drive

     .if actionCodes
	mov.b	#' ',w1		;demark a resync entry
	mov.w	w1,U1TXREG		;transmit the character
     .endif

; Wait for a rotation to be within MAX_DEV of the average, then exit to normal
;   sector processing. While waiting, accept rotation times within +/- 2% of
;   nominal and add them into the running average.

waitIndex:	rcall	nextHole		;get time to next sector/index hole

     .if actionCodes
	mov.b	#'r',w1		;R - resync revolution
	mov.w	w1,U1TXREG		;transmit the character
     .endif

	mov.w	[w3+IDX_AVERAGE],w1	;w1=average rotation time
	sub.w	w0,w1,w1		;w1=deviation from average
	bra	geu,1f		;positive result
	neg.w	w1,w1		;force positive result
1:	mov.w	#MAX_DEV,w2		;within max deviation from average?
	cp.w	w1,w2
	bra	gtu,chkValid	;no, go check if within +/-2%

    .if actionCodes
	mov.b	#'V',w1		;index Verified
	mov.w	w1,U1TXREG		;transmit the character
     .endif
	disi	#5		;protect next five instructions
	btst.w	flags,#fHEAD_LOADED	;is head loaded?
	bra	z,atIndex		;no, don't assert head status
	btst.b	driveStatus,#dsMOVE_HEAD    ;make sure head is not moving
	bra	nz,atIndex		;head not stable, don't assert head status
	bclr.b	driveStatus,#dsHEAD_STATUS  ;assert head status
	bra	atIndex		;in endSector routine

chkValid:	mov.w	#INDEX_MAX,w1	;w1=maximum time to call it full revolution
	cp.w	w0,w1
	bra	gtu,waitIndex	;too long
	mov.w	#INDEX_MIN,w1	;w1=minimum time to call it full revolution
	cp.w	w0,w1
	bra	ltu,waitIndex	;too short
	
	rcall	updateAvg		;put this revolution into the average
	bra	waitIndex
	
;-----------------------------------------------------------------------------
; nextHole- return time between the most recent sector/index hole and the next
;     sector/index hole. 
;  returns: w0 = time between pulses (500ns LSB)
;   	wPrevHole = updated with IC2 capture value for this hole
;	clobbers w1
;-----------------------------------------------------------------------------
nextHole:	btst.w	IC2CON1,#ICBNE	;capture occur?	
	bra	z,nextHole		;no, keep waiting
	mov.w	IC2BUF,w1		;w1=counter at new pulse
	sub.w	w1,wPrevHole,w0	;w0=time between pulses			
	mov.w	w1,wPrevHole	;save new previous IC value
	return

;**********************************************************************************
; Sector Processing Loop
;   1) Assert sector true (if sector position register enabled). Start the
;      30us sector true period.
;
;   2) Start the "read clear" timer. Look for and clear data pulse input
;      captures. Monitor write enable flag in case a write is requested.
;
;  3a) After read clear expires and we're not writing, start the end-of-sector
;      timer (see step 4) and begin shifting data bits into a receive byte. 
;      Once the MSB has a one in it (sync), move the byte to the readData
;      register for the 8080 and assert NRDA in the drive status register. 
;      Now accept every 8 bits and repeat the  process of storing each in
;      readData and asserting NRDA.
;
;  3b) If write data is asserted during the read clear period, assert write
;      enable to the drive and immediately begin writing zeros to the drive.
;      Start the "write clear" timer. When write clear expires, assert ENWD
;      in the drive status register and start the end-of-sector timer (see step 4).
;      As each byte boundary is reached, pull the next byte from the writeData
;      register from the 8080 and assert ENWD again.
;
;   4) When the end-of-sector (EOS) timeout expires, we are about 200us past
;      the end of data on this sector. There is about 200us remaining until
;      the next virtual index pulse is expected. If writing data, idle the 
;      data output and turn off write enable. Look for the real sector pulse
;      to occur on IC2 and when it does, set up OC4 to trigger the next virtual
;      sector pulse. If two IC2 captures occur, then the index pulse occurred
;      along with a sector pulse. Once OC4 is setup for the next virtual sector
;      pulse, jump back to step 1.
;
;-----------------------------------------------------------------------------
; Start of a new sector. Update the sector position register if enabled, assert
;    interrupt to 8080 if enabled.

sectorLoop:	sl.w	sector,WREG		;w0=sector in bits 4-1
	ior.w	#spSECTOR_TRUE_MASK,w0	;set bits 7,6 high, 0 is low

    .if s3Debug
	disi	#6		;protect next 6 instructions
	btss.b	driveStatus,#dsHEAD_STATUS   ;head loaded and settled?
	mov.b	WREG,sectorPos	;yes, update sector position register
	btss.b	driveStatus,#dsHEAD_STATUS   ;head loaded and settled?
	bset.w	LATC,#C14_DRV_TYPE3_SW
    .else
	disi	#4		;protect next 4 instructions
	btss.b	driveStatus,#dsHEAD_STATUS   ;head loaded and settled?
	mov.b	WREG,sectorPos	;yes, update sector position register
    .endif
	btsc.w	flags,#fINTS_ENABLED	;8080 interrupt enabled?
	bset.w	LATD,#D8_INTERRUPT_OUT	;assert 8080 interrupt

; Start a 30us timeout of the sector true period using OC4.
	
	mov.w	sectorStart,w0	;w0=count when sector started
	add.w	#SEC_TRUE_TIME,w0	;w0=count when sector true ends
	mov.w	w0,OC4R		;set that as the next compare
	bclr.w	IFS1,#OC4IF		;make sure flag is false

; wtSecFalse - wait for sector true period to expire

wtSecFalse:	btst.w	IFS1,#OC4IF		;sector true period done?
	bra	z,wtSecFalse	;no

; Sector true period expired. De-assert the sector true signal in the sector
;   position register, then start the read-clear period. 

	bset.b	sectorPos,#spSECTOR_TRUE   ;de-assert sector true flag
	mov.w	sectorStart,w0	;w0=count at which sector started
	add.w	#READ_CLEAR_TIME,w0	;w0=count for read clear to end
	mov.w	w0,OC4R
	bclr.w	IFS1,#OC4IF		;make sure flag is false

   .if s3Debug
	bclr.w	LATC,#C14_DRV_TYPE3_SW
    .endif

;  wtReadClr - waiting for read clear period to end. Pull off data pulse 
;    capture events and update the previous pulse value. While waiting, look
;    for the write enable flag in case a write is requested.

wtReadClr:	btst.w	IFS1,#OC4IF		;read clear timeout done?
	bra	nz,initRead		;yes, enter the read loop
	btst.w	flags,#fWRITE_ENABLE	;write enable requested?
	bra	nz,initWrite	;yes, enter the write loop
	btst.w	IC1CON1,#ICBNE	;data pulse capture occur?	
	bra	z,wtReadClr		;no, keep waiting	
	mov.w	IC1BUF,wPrevData	;pull off the capture time
	bra	wtReadClr

;-------------------------------------------------------------------------------------	
; readLoop - Looks for data pulses from the drive captured by IC1. Once a pulse 
;   is received, decode and shift bits into shiftReg. Assert NRDA to the 8080
;   after each byte is assembled.
;-------------------------------------------------------------------------------------	
; initRead - If trim erase is on, then we're in the first sector not to be written
;    following a write. In this case, set OC4 to time out the 500us trim-erase period
;    after which head movement is allowed again. Otherwise, set OC4 for the end-of-
;    sector period. Initialize for data reception and enter the main read data loop.

initRead:	mov.w	#EOS_TIME,w0	;assume timing until end of sector
	btsc.w	flags,#fTRIM_ERASE	;in trim erase period?
	mov.w	#TRIM_ERASE_TIME,w0	;yes, timing 500us trim erase period
	add.w	sectorStart,WREG	;w0=OC4 count for end-of-sector or...	
	mov.w	w0,OC4R		;  end of trim erase period
	bclr.w	IFS1,#OC4IF		;make sure flag is false
	clr.w	shiftReg		;init received byte to zero
	bclr.w	flags,#fDATA_ONE	;assume data bit is zero
	bclr.w	flags,#fDATA_SYNC	;don't have data sync yet
	btsc.w	IC1CON1,#ICBNE	;make sure we have latest data pulse
	mov.w	IC1BUF,wPrevData	;get most recent pulse (one behind at most)

; readLoop - loop waiting for another data pulse or the OC4 timer to compare. This
;    will be end of sector or end of trim erase.

readLoop:	btst.w	IFS1,#OC4IF		;end of sector time or trim erase?
	bra	nz,chkTrim		;yes, process either
	btst.w	IC1CON1,#ICBNE	;data pulse capture occur?	
	bra	z,readLoop		;no, keep waiting

; Have a data pulse. If it's been about 4us, this pulse marks the end of
;    a bit. If it's been about 2us, this pulse marks a "1" bit.
	
	mov.w	IC1BUF,w1		;pull off the capture time
	sub.w	w1,wPrevData,w0	;w0=time between data pulses
	cp.w	w0,wMinDataZero	;2us (one) or 4us (zero)?
	bra	gtu,readBit		;4us, read the bit
	bset.w	flags,#fDATA_ONE	;2us, bit will be a one
	bra	readLoop

; readBit - end of 4us bit window is done. Decode one or zero and shift the
;   bit into shiftReg. 

readBit:	mov.w	w1,wPrevData	;save new previous count
	sl.w	shiftReg		;make room for new bit
	btsc.w	flags,#fDATA_ONE	;data one or zero?
	bset.w	shiftReg,#0		;data bit was one
	bclr.w	flags,#fDATA_ONE	;reset data one flag
	btst.w	flags,#fDATA_SYNC	;have data sync yet?
	bra	nz,dataBit		;yes, this is a data bit

; Don't have data sync yet, look for first "1" bit to show up in the MSB of shiftReg

	btst.w	shiftReg,#7		;msb one (sync)?
	bra	z,readLoop		;no
	bset.w	flags,#fDATA_SYNC	;now have data sync
	nop			;equalize timing with normal data bit

; postData - eight bit value assembled in shiftReg. Post the read byte to the readData
;    register and assert NRDA. Re-initialize to receive another byte

postData:	mov.w	shiftReg,w0		;w0 = byte just formed
	disi	#2		;protect next two instructions
	mov.b	WREG,readData	;update the data register for the 8080
	bclr.b	driveStatus,#dsNRDA	;assert new read data available
	mov.w	#8,w0		;start bit count over
	mov.w	w0,bitCount
	bra	readLoop

; dataBit - have a data bit, decrement bit counter until a byte is assembled. The
;   odd order of testing for sync above and the databit here is to equalize the
;   time at which NRDA is asserted for the two paths.

dataBit:	dec.w	bitCount
	bra	z,postData		;byte complete, post the data
	bra	readLoop		;otherwise, get next bit

; chkTrim - OC4 triggered during data read loop. This is either due to the
;   end of the trim erase period or the end of sector period. If not for
;   trim erase, jump immediately to endSector. Otherwise, turn off trim erase,
;   set the next OC4 trigger for the end of sector period and re-enter the
;   read loop.

chkTrim:	btst.w	flags,#fTRIM_ERASE	;here to turn off trim erase?
	bra	z,endSector		;no, do end of sector
	bclr.w	flags,#fTRIM_ERASE	;turn off trim erase
	bclr.b	driveStatus,#dsMOVE_HEAD  ;allow head movement again

; Set OC4 to signal the end of sector period and then jump back into the read loop

	mov.w	#EOS_TIME,w0	;w0=timing until end of sector
	add.w	sectorStart,WREG	;w0=OC4 count for end of sector
	mov.w	w0,OC4R
	bclr.w	IFS1,#OC4IF		;make sure flag is false
	bra	readLoop
		
;-----------------------------------------------------------------------------
; Write Data Loop - A write enable command has been received. Start the 
;     write-clear time period during which a stream of zeros is written
;     to the disk. After the write-clear period expires, set the ENWD flag
;     in the drive status register to start the handshake of data with
;     the 8080 that is then written to the disk.
;
;     A data pulse is written every 4us. For a data value of one, an
;     additional pulse is inserted at 2us. For a data value of zero, no
;     additional pulse is present. An SPI port is used to write a bit using
;     8 SPI bits per 8080 data bit.
;-----------------------------------------------------------------------------
; initWrite - initialize for the write data loop. Start the write clear
;    period in OC4 (timing is from sector index). Assert write enable to
;    the disk and begin writing zeros.

initWrite:	bclr.w	LATB,#B9_WRITE_GATE	;assert write gate to the drive
	mov.w	#WRITE_ZEROS,w1	;w1=two bits of zero
	mov.w	w1,SPI1BUF		;send 1st four bits of zeros
	mov.w	w1,SPI1BUF		;write twice so FIFO isn't empty
	bclr.w	IFS0,#SPI1IF	;clear FIFO empty flag
	bclr.w	flags,#fWRITE_ENABLE	;clear the write flag
	bset.w	flags,#fTRIM_ERASE	;indicate trim erase will be on
	bset.b	driveStatus,#dsMOVE_HEAD  ;don't allow step during write
	mov.w	#WRITE_CLEAR_TIME,w0	;w0=write clear time duration
	add.w	sectorStart,WREG	;w0=OC4 count for write clear to end
	mov.w	w0,OC4R
	bclr.w	IFS1,#OC4IF		;make sure flag is false

;  Wait for write clear period to end. Continue writing zeros during this period.

wtWriteClr:	btst.w	IFS1,#OC4IF		;write clear timeout done?
	bra	nz,doWrite		;yes
	btst.w	IFS0,#SPI1IF	;fifo empty?	
	bra	z,wtWriteClr	;no
	mov.w	w1,SPI1BUF		;transmit 2 more bits of zero
	bclr.w	IFS0,#SPI1IF	;clear fifo empty flag
	bra	wtWriteClr

; doWrite - write clear period is done. When fifo empties, set the ENWD flag
;   for the 8080 to see. One full byte of zeros is then written to give
;   the 8080 time to OUT the first data byte. From then on, write data is
;   taken from the writeData register from the 8080. This loop exits when the
;   sector interrupt puts us back at sectorClear.

doWrite:	btst.w	IFS0,#SPI1IF	;fifo empty?	
	bra	z,doWrite		;no
	mov.b	writeData,WREG	;clear PMP status flag
	bclr.b	driveStatus,#dsENWD	;assert ENWD to 8080
 	mov.w	w1,SPI1BUF		;write one byte of zeros
	mov.w	w1,SPI1BUF
	mov.w	w1,SPI1BUF
	mov.w	w1,SPI1BUF
	bclr.w	IFS0,#SPI1IF	;clear fifo empty flag

; start timer in OC4 to signal when it's time for end-of-sector processing

	mov.w	#EOS_TIME,w0	;w0=200us past end of data
	add.w	sectorStart,WREG	;w0=OC4 count for end-of-sector processing	
	mov.w	w0,OC4R
	bclr.w	IFS1,#OC4IF		;make sure flag is false

; wrtLoop - wait for the transmit FIFO to empty, re-assert ENWD to the 8080
;   and write the new byte to the SPI1 FIFO two bits at a time.

wrtLoop:	btst.w	IFS1,#OC4IF		;end of sector time?
	bra	nz,endSector	;yes
	btst.w	IFS0,#SPI1IF	;fifo empty? (8 bits done)	
	bra	z,wrtLoop		;no

	mov.b	writeData,WREG	;w0=byte to write, clear PMP status flag
	bclr.b	driveStatus,#dsENWD	;assert ENWD to 8080

; Bits 7-6

	mov.w	#WRITE_ZEROS,w1	;w1 = 2 bits of zeros
	btsc.w	w0,#7		;start at msb
	bclr.w	w1,#11		;pulse at 2us for a one
	btsc.w	w0,#6
	bclr.w	w1,#3		;pulse at 2us for a one
	mov.w	w1,SPI1BUF		;send bits 7 and 6

; Bits 5-4

	mov.w	#WRITE_ZEROS,w1	;w1 = 2 bits of zeros
	btsc.w	w0,#5		;start at msb
	bclr.w	w1,#11		;pulse at 2us for a one
	btsc.w	w0,#4
	bclr.w	w1,#3		;pulse at 2us for a one
	mov.w	w1,SPI1BUF		;send bits 5 and 4

; Bits 3-2

	mov.w	#WRITE_ZEROS,w1	;w1 = 2 bits of zeros
	btsc.w	w0,#3		;start at msb
	bclr.w	w1,#11		;pulse at 2us for a one
	btsc.w	w0,#2
	bclr.w	w1,#3		;pulse at 2us for a one
	mov.w	w1,SPI1BUF		;send bits 3 and 2

; Bits 1-0

	mov.w	#WRITE_ZEROS,w1	;w1 = 2 bits of zeros
	btsc.w	w0,#1		;start at msb
	bclr.w	w1,#11		;pulse at 2us for a one
	btsc.w	w0,#0
	bclr.w	w1,#3		;pulse at 2us for a one
	mov.w	w1,SPI1BUF		;send bits 1 and 0

; clear the FIFO interrupt and wait for next byte

	bclr.w	IFS0,#SPI1IF	;clear fifo empty flag
	bra	wrtLoop

;-------------------------------------------------------------------------------
;  endSector - end of sector processing. At this point we are about 200us past
;      the end of data on this sector. There is about 200us remaining until
;      the next virtual index pulse is expected. If writing data, idle the
;      data output and turn off write enable. If an index pulse is captured
;      in IC2, then update the rotation time average and compute a new sector
;      time to use for generating virtual sector pulses for the next revolution.
;      Otherwise, increment the sector number, compute and  wait for the next
;      virtual index pulse. Once received, jump back to the sector loop.
;-------------------------------------------------------------------------------
endSector:	bset.w	LATB,#B9_WRITE_GATE	;clear possible write signal
	mov.w	wFFFF,SPI1BUF	;idle write data high
	bset.b	driveStatus,#dsNRDA	;de-assert data handshakes
	bset.b	driveStatus,#dsENWD

; If the motor is off, then continue to generate virtual sector pulses to keep 
;    possible interrupts to the 8080 working properly. Don't look for index pulses
;    from the drive to ensure garbage index pulses as the drive shuts down aren't
;    used in the computation of revolution time.

	btst.w	LATB,#B11_MOTOR_ON	;motor on?
	bra	z,motorIsOn		;yes
 	inc.w	sector,WREG		;else, increment and wrap sector
	and.w	#SECTOR_WRAP,w0
	mov.w	w0,sector
	bra	nextSecOc		;go compute next output compare	

; If the index pulse occured, it's time to update the rotation time average and
;    compute new virtual sector pulse timing. We are also at sector zero.

motorIsOn:	btst.w	IC2CON1,#ICBNE	;index pulse occur?
	bra	z,nextSecInc	;no, move to next sector
	bset.w	flags,#fINDEX_PULSE	;remember index occured
	mov.w	IC2BUF,w1		;w1=counter at the pulse
	sub.w	w1,wPrevHole,w0	;w0=time between pulses
	mov.w	w1,wPrevHole	;save new previous hole time

	mov.w	curDrive,w2		;w2=current drive number
	mul.uu	w2,#IDX_LEN,w2	;index into timingTable table by drive num
	mov.w	#timingTable,w3	;w3->timingTable
	add.w	w2,w3,w3		;w3->row in timingTable for current drive
	
	mov.w	[w3+IDX_AVERAGE],w1	;w1=average rotation time
	sub.w	w0,w1,w1		;w1=deviation from average
	bra	geu,1f		;positive result
	neg.w	w1,w1		;force positive result
1:	mov.w	#MAX_DEV,w2		;beyond max deviation from average?
	cp.w	w1,w2
	bra	gtu,reSync		;yes, force resync

; atIndex - last input capture was index hole. Update running average of rotation
;   time for current drive. On entry, w0 contains time between last two index pulses.

atIndex:	clr.w	sector		;force sector 0
	rcall	updateAvg

; Compute sector zero wakeup as index time + 1/2 sector time. Math is done keeping
;    seven bits of fraction. A copy of OC4R with seven bits of fraction is maintained
;    in oc4F7. Upon entry, w1,w2 = LSW,MSW of 1/2 sector time frac 7, and on exit
;    w3,w4 contain LSW,MSW of index pulse time frac 7.
 
	lsr.w	wPrevHole,#9,w4	;w4=MSW of (IC4 << 7)
	sl.w	wPrevHole,#7,w3	;w3=LSW of (IC4 << 7)
	bra	computeOc4		;go add IC4 and 1/2 sector time

; nextSecInc - increment sector number for the upcoming sector pulse.

nextSecInc:	inc.w	sector,WREG		;increment and wrap sector number
	and.w	#SECTOR_WRAP,w0
	mov.w	w0,sector
	bra	nz,nextSecOc	;go on if not sector 0
	btst.w	flags,#fINDEX_PULSE	;index pulse should have occured
	bra	z,reSync		;it didn't

; nextSecOc - compute output compare for next virtual sector pulse

nextSecOc:	mov.w	oc4F7Lsw,w1		;w1,w2 = previous OC4R w/7 bit fraction
	mov.w	oc4F7Msw,w2
	mov.w	sectF7Lsw,w3	;w3,w4 = sector time w/7 bit fraction
	mov.w	sectF7Msw,w4

; Compute w1,w2 + w3,w4 and store result into ocF7. Shift this result 7 bits to the
;   right and store as the OC4R value for the next wakeup.

computeOc4:	add.w	w1,w3,w1		;w1 = LSWs
	addc.w	w2,w4,w2		;w2 = MSWs
	mov.w	w1,oc4F7Lsw		;save OC4 with w/7 bit fraction
	mov.w	w2,oc4F7Msw	

; Convert 32 bit result with fraction back into 16 bit value for OC4R

	lsr.w	w1,#7,w1		;w1=drop off lower 7 bits of result
	sl.w	w2,#9,w2		;w2=move to upper 7 bits
	ior.w	w1,w2,w0		;w0=w1,w2 >> 7
	mov.w	w0,sectorStart	;save counter at start of nextsector
	mov.w	w0,OC4R
	bclr.w	IFS1,#OC4IF		;make sure flag is false

	subr.w	OC4TMR,WREG		;make sure compare time > current timer
	bra	n,reSync		;already past sector time

; If head is no longer loaded, increment the motor timeout. When the 1024's bit is
;    set (1024 * 5.208ms/sector) = 5.33s, turn off the motor.

	disi	#5
	btsc.w	flags,#fHEAD_LOADED	;check if head loaded
	clr.w	motorTimer		;hold timer in reset if head loaded
	inc.w	motorTimer		
	btsc.w	motorTimer,#10	;test the 1024's bit
	bset.w	LATB,#B11_MOTOR_ON	;turn off  motor

; wtNextSec - look for next virtual index pulse

	bclr.w	flags,#fINDEX_PULSE	;ensure clear for next time
wtNextSec:	btst.w	IFS1,#OC4IF		;next sector match?
	bra	z,wtNextSec		;no, keep waiting
	bra	sectorLoop		;back to start of sector loop	

;-------------------------------------------------------------------------------
; updateAvg - Update the average with the rotation time in w0. On entry
;     w3->timingTable[curDrive]. On exit, w2,w1 contains 1/2 sector time 
;      with 7 bits of fraction.
;-------------------------------------------------------------------------------
updateAvg:	mov.w	[w3+IDX_SUM_LSW],w1	;w1=LSW of sum
	mov.w	[w3+IDX_SUM_MSW],w2	;w2=MSW of sum
	add.w	w1,w0,w1		;w1=LSW of sum + new time (16 bit)
	addc.w	#0,w2		;w2=MSW of sum + new time

	mov.w	[w3+IDX_AVERAGE],w0	;w0=LSW of old average (16 bit)
	sub.w	w1,w0,w1		;w1=LSW of sum - old average
	subb.w	#0,w2		;w2=MSB of sum - old average

	mov.w	w1,[w3+IDX_SUM_LSW]	;save new 32 bit sum
	mov.w	w2,[w3+IDX_SUM_MSW]

; Sector time = sum/4/32 (sum/4 = average full rotation, average/32 = sector time).
;    These two divides represent 7 bits of shifting. Therefore, simply saving the
;    average sum is the same as the sector time with 7 bits of fraction.

	mov.w	w1,sectF7Lsw	;LSW of sector time with 7 bit fraction
	mov.w	w2,sectF7Msw	;MSW of sector time with 7 bit fraction

; Compute and save new average full revolution time = (new sum) / 4.

	lsr.w	w2,w2		;shift MSW (1/2 sector time f7)
	rrc.w	w1,w1		;rotate LSW (1/2 sector time f7)
	lsr.w	w2,w4		;shift MSW into w4 (preserve w2)
	rrc.w	w1,w0		;rotate LSW into w0 (preserve w1)
	mov.w	w0,[w3+IDX_AVERAGE]	;save new average (16 bit)
	return	
	
;****************************************************************************************
;
;  ioPortInt - ISR for handling reads and writes to the Altair FDC control and
;     status registers (done through the PIC EPMP). 
;
;     Some register OUTs from 8080 code are immediately followed by a register IN.
;     Response here needs to be less than 2us (to handle up to a 4MHz processor)
;     between an OUT action that then updates a status register.
;
;     This code makes the assumption that it will never get behind, i.e., it can
;     process the interrupts faster than the 8080 will ever do I/O.
;
;     Note that a drive deselect or new drive select bypasses the normal retfie 
;     interrupt exit and jump directly to the idle state or the verify index state.
;     This means that executing code may not regain control.
;
;****************************************************************************************
;  Check if the drive select register written.

ioPortInt:	bclr.w	IFS2,#PMPIF		;clear the PMP interrupt
	bclr.w	LATD,#D8_INTERRUPT_OUT	;any access removes interrupt to 8080
	btst.w	PMSTAT,#pmSELECT_CMD	;select register written?
	bra	z,chkDriveCmd	;no, go check drive command
	mov.b	selectCmd,WREG	;w0=select command, clear status flag
	btst.z	w0,#scDESELECT	;de-select requested?
	bra	nz,ioDeselect	;yes

; A drive select command was written. Verify the passed drive number and see if it is 
;    already selected. If so, just exit. If invalid, do a drive deselect. Otherwise,
;    do a deselect followed by a select.

	and.w	#scDRIVE_MASK,w0	;get drive number alone
	cp.w	w0,#NUM_DRIVES	;valid drive number?
	bra	geu,ioDeselect	;no, deselect current drive
	mov.w	w0,curDrive		;save current drive number
	btst.z	[pLATB],w0		;same drive already selected?
	bra	z,ioExit		;yes, just exit

; Select the drive, turn on the motor and initialize drive status

	mov.w	wAllDriveOff,LATB	;disable all drive signals (deselects)
	mov.w	wFFFF,statusRegs	;driveStatus and sectorPos to 0xff

	bclr.w	SR,#C		;carry bit to zero
	bsw.c	[pLATB],w0		;assert proper drive select bit
	mov.b	wDsInitValue,[pDriveStatus]   ;init drive status register
	btsc.w	PORTD,#D2_TRACK0	;track zero asserted from drive?
	bclr.b	driveStatus,#dsTRACK0	;yes, assert track zero
	btsc.w	PORTF,#F0_BUS_PINTE	;PINTE signal from bus asserted?
	bclr.b	driveStatus,#dsINTE	;yes, assert INTE in drive status
	bclr.w	LATB,#B11_MOTOR_ON	;turn on motor
	clr.w	motorTimer		;restart motor timer
     .if actionCodes
	mov.b	#'S',w1		;S - select drive command
	mov.w	w1,U1TXREG		;transmit the character
     .endif
	bra	enterIndex		;enter the validate index state 

; ioDeselect - "Exit" by restarting at enterIdle

ioDeselect:
     .if actionCodes
	mov.b	#'s',w1		;s - deselect drive
	mov.w	w1,U1TXREG		;transmit the character
     .endif
	bra	enterIdle		;enter the idle state

;-------------------------------------------------------------------------------------------
; chkDriveCmd - Drive command register written? If so, process it. Simultaneous
;    commands are possible, but some combinations are meaningless. The following
;    logic is implemented:
;
;  if (step in/out or head load/unload) {
;     if (step out)
;        step out
;     else if (step in)
;        step in
;
;     if (head load)
;        load head
;     else if (head unload)
;        unload head
;  }
;  else {
;     if (write enable)
;        set internal write enable flag
;  }
;  if (interrupt disable)
;     disable interrupts
;  else if (interrupt disable)
;     disable interrupts
;   
;-------------------------------------------------------------------------------------------
chkDriveCmd: btst.w	PMSTAT,#pmDRIVE_CMD	;drive command written?
	bra	z,chkReadData	;no, see if read data register read
	mov.b	driveCmd,WREG	;w0=drive command, clear status flag
	btst.b	driveStatus,#dsREADY	;do we show is drive ready?
	bra	nz,ioCmdExit	;no, ignore the command
	push.w	w1		;save w1
	
; Check for any head step or head load/unload commands first. If any of these, immediately
;    set head status to false and disable the sector position register (0xff). This
;    is done because 8080 software may quickly read the status registers after writing 
;    this command. We update the status registers within 19-22 cycles (1.2-1.4us).

	and.b	w0,#dcHEAD_MASK,w1	;any of the head move/load commands?
	bra	z,ioChkWrite	;no, go check write enable
	bset.b	driveStatus,#dsHEAD_STATUS   ;de-assert head status flag
	mov.b	wFFFF,[pSectorPos]	;sector register to 0xff
	and.b	w0,#dcMOVE_MASK,w1	;any of the head move commands?
	bra	z,ioChkHead		;no, must be head load/unload
	bset.b	driveStatus,#dsMOVE_HEAD   ;de-assert move head flag

;=================
; Step Out command
;=================
	btst.z	w0,#dcSTEP_OUT	;step out command?
	bra	z,ioStepIn		;no, must be step in command
	btst.w	PORTD,#D2_TRACK0	;already at track 0?
	bra	nz,ioDoTimers	;yes, don't issue step to drive
	mov.w	#STEP_CHAR,w1	;data byte that when serially transmitted...
	mov.w	w1,U2TXREG		;  generates a 2us step pulse
				;(step direction line already high = out)
     .if actionCodes
	mov.b	#'-',w1		;- (minus) is step out
	mov.w	w1,U1TXREG		;transmit the character
     .endif
	bra	ioDoTimers		;start both timers

;=================
; Step In command 
;=================
ioStepIn:	mov.w	#DIR_IN_CHAR,w1	;data byte that when serially transmitted...
	mov.w	w1,U3TXREG		;  sets step direction low (in) for 5us
	mov.w	#STEP_CHAR,w1	;data byte that when serially transmitted...
	mov.w	w1,U2TXREG		;  generates a 2us step pulse
     .if actionCodes
	mov.b	#'+',w1		;+ (plus) is step in
	mov.w	w1,U1TXREG		;transmit the character
     .endif

; ioDoTimers - start head step timer and 50ms head settle timer on OC1 and OC2

ioDoTimers: push	w0		;preserve w0
	mov.w	ptStepT58,w0	;w0=step time
	add.w	OC1TMR,WREG		;add timeout to current OC1 timer
	mov.w	w0,OC1R		;store as new OC1 compare value
	mov.w	#HEAD_SETTLE_TIME,w0	;w0=50ms head settle timer
	add.w	OC2TMR,WREG		;add timeout to current OC2 timer
	mov.w	w0,OC2R		;store as new OC2 compare value
	pop	w0		;restore w0
	bclr.w	IFS0,#OC1IF		;clear OC1 interrupt flag
	bclr.w	IFS0,#OC2IF		;clear OC2 interrupt flag
	bset.w	IEC0,#OC1IE		;enable OC1 head step interrupt
	bset.w	IEC0,#OC2IE		;enable OC2 head load interrupt

; Check for head load after a move head has already started timers

	btst.z	w0,#dcHEAD_LOAD	;head load command?
	bra	z,ioChkUnload	;no
	BRA	ioLoadHead		;go load the head

	bset.w	flags,#fHEAD_LOADED	;flag that head is loaded
     .if actionCodes
	mov.b	#'L',w1		;L - head load and step at same time
	mov.w	w1,U1TXREG		;transmit the character
     .endif

; If motor is not on, turn it on. Later, after the interrupt enable/disable commands are processed,
;    control will be passed to enterIndex to wait for the disk to spin up and re-sync. 

	btst.w	LATB,#B11_MOTOR_ON	;motor running?
	bra	z,ioChkInt		;yes, motor already on, go on
	bclr.w	LATB,#B11_MOTOR_ON	;turn on motor
	bset.w	flags,#fENTER_INDEX	;control should pass to enterIndex
	bra	ioChkInt		;go check for interrupt enable/disable

;=================
; Head Load command (without a move head command as well). For this drive, the
;   head load command starts the drive motor if not already running.
;=================
ioChkHead:	btst.z	w0,#dcHEAD_LOAD	;head load command?
	bra	z,ioChkUnload	;no
	push	w0		;preserve w0
	mov.w	#HEAD_SETTLE_TIME,w0	;w0=50ms head load timer
	add.w	OC2TMR,WREG		;add timeout to current OC2 timer
	mov.w	w0,OC2R		;store as new OC2 compare value
	pop	w0		;restore w0
	bclr.w	IFS0,#OC2IF		;clear OC2 interrupt flag
	bset.w	IEC0,#OC2IE		;enable OC2 head load interrupt

ioLoadHead:	bset.w	flags,#fHEAD_LOADED	;flag that head is loaded
     .if actionCodes
	mov.b	#'H',w1		;H - head load
	mov.w	w1,U1TXREG		;transmit the character
     .endif 
	btst.w	LATB,#B11_MOTOR_ON	;motor running (head loaded)?
	bra	z,ioChkInt		;yes, motor already on
	bclr.w	LATB,#B11_MOTOR_ON	;else, turn on the motor
	bset.w	flags,#fENTER_INDEX	;control should pass to enterIndex
	bra	ioChkInt		;go check for interrupt enable/disable

;=================
; Head Unload command. Clear the head loaded flag.
;=================
ioChkUnload: btst.z	w0,#dcHEAD_UNLOAD	;head unload command?
	bra	z,ioChkInt		;no, go check for interrupt enable/disable
	bclr.w	flags,#fHEAD_LOADED	;flag that head not loaded
    .if actionCodes
	mov.b	#'h',w1		;h - head unload
	mov.w	w1,U1TXREG		;transmit the character
     .endif
	bra	ioChkInt		;go check for interrupt enable/disable

;=================
; Write enable command. Set the internal write enable flag. Non-interrupt codes looks at
;    this flag early in the sector to determine whether or not to enable a write operation
;    to the drive.
;=================
ioChkWrite:	btsc.w	w0,#dcWRITE_ENABLE	;write enable command?
	bset.w	flags,#fWRITE_ENABLE	;yes, set write enable flag

;=================
; Interrupt enable/disable. These commands set or clear the flag looked at by sector
;    index processing. When disabling, the output signal is immediately turned off
;    as well.
;=================
ioChkInt:	pop.w	w1		;restore w1
	btst.z	w0,#dcINT_DISABLE	;disable interrupt command?
	bra	z,ioChkEnable	;no, go check enable
	bclr.w	flags,#fINTS_ENABLED	;clear interrupts enabled flag
	bclr.w	LATD,#D8_INTERRUPT_OUT	;de-assert 8080 interrupt
	bra	ioCmdExit		;exit (disable overrides an enable command)

; Check for interrupt enable command 

ioChkEnable: btsc.w	w0,#dcINT_ENABLE	;enable interrupt command?
	bset.w	flags,#fINTS_ENABLED	;set interrupts enabled flag
ioCmdExit:	btst.w	flags,#fENTER_INDEX	;exit into enterIndex?
	bclr.w	flags,#fENTER_INDEX
	bra	nz,headLoaded	;yes, headLoaded is in enterIndex
ioExit:	pop	w0		;otherwise, normal exit
	retfie

;-------------------------------------------------------------------------------------------
; chkReadData - See if the data byte has been read from the controller. If so, clear
;    the NRDA (new read data available) flag.
;-------------------------------------------------------------------------------------------
chkReadData: btsc.w	PMSTAT,#pmREAD_DATA	;read data register read?
	bset.b	driveStatus,#dsNRDA	;yes, de-assert data available bit
				;fall through to check write data
;-------------------------------------------------------------------------------------------
; chkWriteData - See if a data byte has been written for the drive to write. If so,
;    de-assert the ENWD (enter new write data) flag and read the byte from the PMP
;    to prevent an overflow status which blocks further updates.
;-------------------------------------------------------------------------------------------
chkWriteData: btst.w	PMSTAT,#pmWRITE_DATA	;write data register written?
	bra	z,1f
	mov.w	writeData,w0	;yes, clear status flag
	bset.b	driveStatus,#dsENWD	;de-assert ENWD
1:	pop	w0		;restore and exit
	retfie

;****************************************************************************************
;
;  headMoveInt - ISR to process expiration of OC1. This interrupt occurs 10.5ms after
;      a step in/out command is issued. Update the track0 bit in the status register
;      and assert the move head status bit in the drive status register.
;
;****************************************************************************************
headMoveInt: bclr.w	IEC0,#OC1IE		;disable the OC1 interrupt
     .if actionCodes
	mov.b	#'M',w0		;M - OK to move head
	mov.w	w0,U1TXREG		;transmit the character
     .endif
	btst.b	driveStatus,#dsREADY	;drive still ready?
	bra	nz,1f		;no, just exit
	btsc.w	PORTD,#D2_TRACK0	;copy track 0 bit without a glitch...
	bclr.b	driveStatus,#dsTRACK0	;   in track 0 flag in register
	btss.w	PORTD,#D2_TRACK0
	bset.b	driveStatus,#dsTRACK0
	bclr.b	driveStatus,#dsMOVE_HEAD  ;assert move head signal
1:	pop.w	w0
	retfie

;****************************************************************************************
;
;  headStatusInt - ISR to process expiration of OC2. This interrupt occurs 50ms after
;      a step in/out or head load command is issued. Assert the head loaded status bit 
;      in the drive status register. Drive still ready is qualified by verification
;      that the head loaded bit is asserted.
;
;****************************************************************************************
headStatusInt:
	bclr.w	IEC0,#OC2IE		;disable the OC2 interrupt
     .if actionCodes
	mov.b	#'T',w0		;T - head seTTled
	mov.w	w0,U1TXREG		;transmit the character
    .endif
	btsc.w	flags,#fHEAD_LOADED	;head still loaded?
	bclr.b	driveStatus,#dsHEAD_STATUS  ;yes, assert head status signal
	pop.w	w0
	retfie

;****************************************************************************************
;
;  changeInt - ISR to process port change interrupts. 
;     The PINTE line from the 8080 (interrupt enabled signal) generates an interrupt
;     so the INTE status bit in the drive status register can be updated.
;
;****************************************************************************************
changeInt:	bclr.w	IFS1,#CNIF		;clear the port change interrupt flag
	btst.w	PORTF,#F0_BUS_PINTE	;PINTE signal from bus asserted?
	bra	z,pInteOff		;no	
	btss.b	driveStatus,#dsREADY	;skip next inst if drive not ready
	bclr.b	driveStatus,#dsINTE	;assert INTE in drive status
	pop.w	w0		;restore w0 and exit
	retfie

; pInteOff - PINTE not asserted. De-assert INTE in the drive status register

pInteOff:	bset.b	driveStatus,#dsINTE	;de-assert INTE in drive status
	pop.w	w0		;restore w0 and exit
	retfie

	.end
