;***************************************************************************************
;
;  serialDrive.s 
;     This module duplicates Altair 8" and minidisk drive operation over the serial
;     port built into the FDC+. Communication is done at 403.2K baud so performance
;     is similar to the real drive. A file server must run on the host PC.
;
;  Version History
;    1.0   02/25/15  M. Douglas
;	- Original
;    1.1   04/24/20  M. Douglas	
;	- Fix output compares to work around a problem similar to that described
;	  in the 24FJ errata.
;	- Run all ISRs at the same priority so no ISR nesting occurs.
;	- Do not clear internal head load and interrupt enable status when index
;	  sync is lost. 
;	- Add support for an 8Mb drive that is an Altair 8" drive but with 2048
;	  tracks. This primarily required using a word for track numbers and
;	  logic instead of a byte.
;    1.2   06/19/20  M. Douglas	
;	- Clear out all EPMP flags at the end of initialization - immediately
;	  before entering the idle loop - to reset any possible glitches caused
;	  by 8080 CPU start up.
;
;-------------------------------------------------------------------------------
;   
;  PIC H/W Assignments
;
;     EPMP (PMP) handles port I/O from the 8080
;     OC1 is the head step timer
;     CN68 captures changes in the PINTE line (interrupts enabled) from the 8080
;     UART1 is the serial port used to communicate with the PC host
;     TMR1 at 2MHz is used to generate real-time sector interrupts
;     TMR3 at 62.5KHz is used to time server response and timeouts
;     TMR4 at 250khz is clock source for OC1 (head step timer)
;
;-------------------------------------------------------------------------------
;
;  Communication with the server is over a serial port at 403.2K Baud, 8N1.
;  All transactions are initiated by the FDC. The second choice for baud rate
;  is 460.8K. Finally, 230.4K is the most likely supported baud rate on the PC
;  if 403.2K and 460.8K aren't avaialable.
;
;  403.2K is the preferred rate as it allows full-speed operation and is the
;  most accurate of the three baud rate choices on the FDC. 460.8K also allows
;  full speed operation, but the baud rate is off by about 3.5%. This works, but
;  is borderline. 230.4K is available on most all serial ports, is within 2% of
;  the FDC baud rate, but runs at 80%-90% of real disk speed.
;
;  FDC TO SERVER COMMANDS
;    Commands from the FDC to the server are fixed length, ten byte messages. The 
;    first four bytes are a command in ASCII, the remaining six bytes are grouped
;    as three 16 bit words (little endian). The checksum is the 16 bit sum of the
;    first eight bytes of the message.
;
;    Bytes 0-3   Bytes 4-5 as Word   Bytes 6-7 as Word   Bytes 8-9 as Word
;    ---------   -----------------   -----------------   -----------------
;     Command       Parameter 1         Parameter 2           Checksum
;
;    Commands:
;      STAT - Provide and request drive status. The FDC sends the selected drive
;             number and head load status in Parameter 1 and the current track 
;             number in Parameter 2. The Server responds with drive mount status
;             (see below). The LSB of Parameter 1 contains the currently selected
;             drive number or 0xff is no drive is selected. The MSB of parameter 1
;             is non-zero if the head is loaded, zero if not loaded.
;
;             The FDC issues the STAT command about ten times per second so that
;             head status and track number information is updated quickly. The 
;             server should also assume the drive is selected, the head is loaded,
;             and update the track info whenever a READ or WRIT command is received.
;
;      READ - Read specified track. Parameter 1 contains the drive number in the
;             MSNibble. The lower 12 bits contain the track number. Transfer length
;             length is in Parameter 2 and must be the track length. Also see
;             "Transfer of Track Data" below.
;
;      WRIT - Write specified track. Parameter 1 contains the drive number in the
;             MSNibble. The lower 12 bits contain the track number. Transfer length
;             must be track length. Server responds with WRIT response when ready
;             for the FDC to send the track of data. See "Transfer of Track Data" below.
;
;
;  SERVER TO FDC 
;    Reponses from the server to the FDC are fixed length, ten byte messages. The 
;    first four bytes are a response command in ASCII, the remaining six bytes are
;    grouped as three 16 bit words (little endian). The checksum is the 16 bit sum
;    of the first eight bytes of the message.
;
;    Bytes 0-3   Bytes 4-5 as Word   Bytes 6-7 as Word   Bytes 8-9 as Word
;    ---------   -----------------   -----------------   -----------------
;     Command      Response Code        Reponse Data          Checksum
;
;    Commands:
;      STAT - Returns drive status in Response Data with one bit per drive. "1" means a
;             drive image is mounted, "0" means not mounted. Bits 15-0 correspond to
;             drive numbers 15-0. Response code is ignored by the FDC.
;
;      WRIT - Issued in repsonse to a WRIT command from the FDC. This response is
;             used to tell the FDC that the server is ready to accept continuous transfer
;             of a full track of data (response code word set to "OK." If the request
;             can't be fulfilled (e.g., specified drive not mounted), the reponse code
;             is set to NOT READY. The Response Data word is don't care.
;
;      WSTA - Final status of the write command after receiving the track data is returned
;             in the repsonse code field. The Response Data word is don't care.
;
;    Reponse Code:
;      0x0000 - OK
;      0x0001 - Not Ready (e.g., write request to unmounted drive)
;      0x0002 - Checksum error (e.g., on the block of write data)
;      0x0003 - Write error (e.g., write to disk failed)
;
;
;  TRANSFER OF TRACK DATA
;    Track data is sent as a sequence of bytes followed by a 16 bit, little endian 
;    checksum. Note the Transfer Length field does NOT include the two bytes of
;    the checksum. The following notes apply to both the FDC and the server.
;
;  ERROR RECOVERY
;    The FDC uses a timeout of one second after the last byte of a message or data block
;        is sent to determine if a transmission was ignored.
;
;    The server should ignore commands with an invalid checksum. The FDC may retry the
;        command if no response is received. An invalid checksum on a block of write
;        data should not be ignored, instead, the WRIT response should have the
;        Reponse Code field set to 0x002, checksum error.
;
;    The FDC ignores responses with an invalid checksum. The FDC may retry the command
;        that generated the response by sending the command again.
;
;***************************************************************************************
	.include 	"common.inc" 
 
; Debug equates. Set to 1 to enable, 0 to disable.

	.equ	s3Debug,0		;debug signal output on switch 3 input for scope

; Drive Equates

	.equ	NUM_DRIVES,8	;up to 8 drives supported
	.equ	NUM_TRACKS8,77	;tracks on 8" drive
	.equ	NUM_TRACKS5,35	;tracks on minidisk
	.equ	NUM_SECTORS8,32	;sectors per track on 8" drive
	.equ	NUM_SECTORS5,16	;sectors per track on minidisk
	.equ	BYTES_PER_SECTOR,137
	.equ	TRACK_LENGTH8,(NUM_SECTORS8*BYTES_PER_SECTOR)
	.equ	TRACK_LENGTH5,(NUM_SECTORS5*BYTES_PER_SECTOR)

; The following working registers have permanent variable assignments and cannot
;   be used for any other purpose. 

	.equ	pDriveStatus,w14	;pointer to PMDOUT1L
	.equ	pSectorPos,w13	;pointer to PMDOUT1H
	.equ	pReadData,w12	;pointer to PMDOUT2L
	.equ	pWriteData,w11	;pointer to PMD2INL
	.equ	pTrackTable,w10	;pointer to trackTable
	.equ	pSector,w9		;pointer into current sector in trackBuf
	.equ	wDrivesReady,w8	;drive per bit, 1=ready, 0=not ready
	.equ	wCurDrivex2,w7	;current drive number x 2
	.equ	wFFFF,w6		;contains 0xffff
	.equ	wDsInitValue,w5	;drive status initial value

; Map PMDIN1/2 and PMDOUT1/2 to more logical names

	.equ	driveStatus,PMDOUT1L	;drive status IN 08
	.equ	selectCmd,PMDIN1L	;drive select OUT 08
	.equ	sectorPos,PMDOUT1H	;sector position register IN 09
	.equ	driveCmd,PMDIN1H	;drive command OUT 09
	.equ	readData,PMDOUT2L	;read data from disk IN 10
	.equ	writeData,PMDIN2L	;write data to disk OUT 10

	.equ	commandRegs,PMDIN1	;select and drive command registers
	.equ	statusRegs,PMDOUT1	;drive status and sector position as word

; Parallel port (PMP) Status bits

	.equ	pmDRIVE_STATUS,0	;drive status register read
	.equ	pmSECTOR_POS,1	;sector position reigster read
	.equ	pmREAD_DATA,2	;read data register read
	.equ	pmSELECT_CMD,8	;drive select register written
	.equ	pmDRIVE_CMD,9	;drive command register written
	.equ	pmWRITE_DATA,10	;write data register written
	
; Drive Status Register Equates. These are all asserted true as zeros. Bits 3 and 4
;    remain zero when the drive is enabled. We use bit 3 as a drive ready flag.

	.equ	dsENWD,0		;0=enter new write data flag (write)
	.equ	dsMOVE_HEAD,1	;0=OK to move head
	.equ	dsHEAD_STATUS,2	;0=head loaded and settled
	.equ	dsREADY,3		;0=drive ready (not official)
	.equ	dsINTE,5		;0=8080 interrupts enabled
	.equ	dsTRACK0,6		;0=on track zero
	.equ	dsNRDA,7		;0=new read data available (read)
	.equ	dsINIT_VALUE,0xe5	;move head asserted, b4, b3 cleared

; Sector Position Register Equates. Sector True asserts as zero.

	.equ	spSECTOR_TRUE,0	;bit 0 is sector true bit
	.equ	spSECTOR_TRUE_MASK,0xc0;  ;unused bits set, sector true left alone
	.equ	spSECTOR_INIT,0xc1	;unused bits set, sector true de-asserted

; Select Command Register Equates

	.equ	scDRIVE_MASK,0x07	;mask to get drive number bits alone
	.equ	scDESELECT,7	;bit number of deselect bit

; Drive Command Register Equates

	.equ	dcSTEP_IN,0		;step head towards center
	.equ	dcSTEP_OUT,1	;step head out towards edge
	.equ	dcHEAD_LOAD,2	;load head for read/write
	.equ	dcHEAD_UNLOAD,3	;unload head
	.equ	dcINT_ENABLE,4	;enable interrupts at start of sector
	.equ	dcINT_DISABLE,5	;disable interrupts from the FDC
	.equ	dcHEAD_CURR_SW,6	;reduce head current
	.equ	dcWRITE_ENABLE,7	;initiate write sequence
	.equ	dcHEAD_MASK,0x0f	;get all head step/load bits at once
	.equ	dcMOVE_MASK,0x03	;get both step in/out bits at once

; Sector/index equates

	.equ	SECTOR_TIME8,5208*2	;5.208ms sector time at 2MHz (8" drive)
	.equ	SECTOR_TIME5,12500*2	;12.5ms sector time at 2MHz (minidisk)
	.equ	SECTOR_WRAP8,0x1f	;mask to wrap sector count (8" drive)
	.equ	SECTOR_WRAP5,0x0f	;mask to wrap sector count (minidisk)

; Head step equates. OC1 is used to time head step. The timebase used for OC1 is 250KHz
;    (4us per tick) and comes from Timer 4. 

	.equ	STEP_TIMER8,10500/4	;head step time, 10.5ms at 250khz (8" drive)
	.equ	STEP_TIMER5,50000/4	;head step time, 50ms at 250khz (minidisk)

; flags variable equates

	.equ	fTRACK_DIRTY,0	;1=trackBuf needs to be written
	.equ	fTRACK_READ,1	;1=request track read from server
	.equ	fINTS_ENABLED,2	;1=FDC interrupts to 8080 enabled
	.equ	fSECTOR_START,3	;1=start of sector PIC interrupt has occured
	.equ	fHEAD_LOADED,4	;1=head is loaded

; server command strings formed as two, two byte word values

	.equ	ASCII_ST,'S'+('T'<<8)	;STAT
	.equ	ASCII_AT,'A'+('T'<<8)

	.equ	ASCII_RE,'R'+('E'<<8)	;READ
	.equ	ASCII_AD,'A'+('D'<<8)

	.equ	ASCII_WR,'W'+('R'<<8)	;WRIT
	.equ	ASCII_IT,'I'+('T'<<8)

	.equ	ASCII_WS,'W'+('S'<<8)	;WSTA
	.equ	ASCII_TA,'T'+('A'<<8)

; motorTimer equates

	.equ	mtMINI64,9		;512's bit for 6.4s minidisk timeout
	.equ	mtMINI16,7		;128's bit for 1.6s minidisk idle timeout
	.equ	mtEIGHT13,8		;256's bit for 1.3s 8" idle timeout
	
; Misc Equates

	.equ	WRITE_TRIES,3	;server write attempts
	.equ	RESPONSE_TIMEOUT,62499	;server response timeout, one second at 62.5khz
	.equ	STAT_RATE,6249	;STAT command every 0.1s at 62.5khz
	.equ	PIC_INTS_OFF,0xe0	;write to SR to disable PIC interrupts

; Data storage

	 .section *,bss,near
flags:	 .space	2		;see flag bits above
byteCount:	 .space	2		;bytes read/written in current sector
driveTrack:	 .space	2		;drive(msb), track(lsb) of data in trackBuf
secsBuffed:	 .space	2		;number of sectors buffered so far
retryCount:	 .space	2		;server i/o retry counter
motorTimer:	 .space	2		;counts sectors until 512 bit set (6.4s)
trackTable:	 .space	NUM_DRIVES*2	;track number each drive is on
stepTime:	 .space	2		;current step time value for OC1

; Server command transmit buffer

	 .equ	COMMAND_LEN,8	;checksum sent separately
cmdBuf:
cmdCommand:	 .space	4		;four byte ASCII command
cmdParam1:   .space	2		;drive:4 and track:12 as 16 bit word
cmdParam2:	 .space	2		;transfer length as word

; Server response receive buffer

	 .equ	RESPONSE_LEN,8	;checksum received separately
rspBuf:
rspCommand:	 .space	4		;four byte ASCII response "command"
rspCode:	 .space	2		;response code as 16 bit word
rspData:	 .space	2		;response data as 16 bit word

; Track buffer

	 .bss
trackBuf:	 .space	TRACK_LENGTH8
	
;****************************************************************************************
;  
;   SerialDrive - entry point for 8 inch and minidisk serial drives
;
;****************************************************************************************
	.text
	.global	SerialDrive

SerialDrive: mov.w	#STACK_ADDR,w15	;init stack pointer
	mov.w	#PIC_INTS_OFF,w0	;disable all interrupts
	mov.w	w0,SR
;
; Initialize the parallel port to handle the I/O IN and OUT instructions from the 8080. The
;     PMP is run in addressable slave mode where I/O addresses from the 8080 map as shown
;     below.
;
;     I/O address            PIC register
;	 8	PMDOUT1, PMDIN1 low byte
;	 9	PMDOUT1, PMDIN1 high byte
;	10	PMDOUT2, PMDIN2 low byte
;	11	PMDOUT2, PMDIN2 high byte (not used)

	mov.w	#0x0281,w0		;enhanced PSP mode, CS1 on PMA14, interrupts on
	mov.w	w0,PMCON1
	mov.w	#0xc000,w0		;PMWR, PMRD enabled
	mov.w	w0,PMCON3
	mov.w	#0x4003,w0		;CS1(A14), A1, A0 enabled
	mov.w	w0,PMCON4
	mov.w	#0x6000,w0		;CS1 on and active high, RD, WR active low
	mov.w	w0,PMCS1CF
	mov.w	#0x0001,w0		;use TTL buffers
	mov.w	w0,PADCFG1
	mov.w	#0xffff,w0		;set both 8080 status registers to 0xff
	mov.w	w0,statusRegs
	bset.w	PMCON1,#15		;enable EPMP
 
; Init TMR1 for 2Mhz. Set the period register to generate 5.208ms or 12.5ms sector
;     interrupts based on the drive type.

	mov.w	#0x0010,w0		;divide 16Mhz by 8 = 2 Mhz
	mov.w	w0,T1CON
	mov.w	#SECTOR_TIME8,w0	;assume 8" drive sector time
	btss.w	DriveType,#0	;8" drive?
	mov.w	#SECTOR_TIME5,w0	;no, use minidisk sector time
	mov.w	w0,PR1
	clr.w	TMR1		;init timer to zero
	bset.w	T1CON,#TON		;turn on timer

; Init TMR3 to use a 62.5KHz timebase. Used primarily to time the server message
;   response timeout and the STAT command interval.

	mov.w	#0x0030,w0		;Fcy/256 for 62.5khz
	mov.w	w0,T3CON
	mov.w	#RESPONSE_TIMEOUT,w0	;timeout in period register
	mov.w	w0,PR3
	bset.w	T3CON,#TON		;turn on the timer

; Init TMR4 to use a 250khz timebase. This timer is the clock source for OC1.

	mov.w	#0x0020,w0		;Fcy/64 for 250khz
	mov.w	w0,T4CON
	mov.w	#0xffff,w0
	mov.w	w0,PR4
	bset.w	T4CON,#TON		;turn on TMR4

; Configure OC1 to time head step delays. TMR4 at 250khz is used as clock source. 

	mov.w	#0x0020,w0		;free running, no output pin
	mov.w	w0,OC1CON2
	mov.w	#0x0803,w0		;TMR4 and one-shot toggle mode
	mov.w	w0,OC1CON1

; Initialize UART1 for 8N1 at the baud rate in ptBaudRate

	mov.w	#0x0008,w0		;serial port off, 8N1, BRGH=1
	mov.w	w0,U1MODE
	mov.w	ptBaudRate,w0	;w0=BRG value to use
	mov.w	w0,U1BRG
	bset.w	U1MODE,#UARTEN	;UART on
	bset.w	U1STA,#UTXEN	;transmitter on too
	
; Set the ready line from the drive and the PINTE signal from the 8080 bus to be
;    input change interrupts.

	bset.w	CNEN4,#CN50IE	;ready line is CN50
	bset.w	CNEN5,#CN68IE	;PINTE signal is CN68

; Put our local ISRs in the interrupt table

	mov.w	#tbloffset(ioPortInt),w0       ;PMP interrupt (8080 I/O)
	mov.w	w0,PMPInterrupt
	mov.w	#tbloffset(headMoveInt),w0     ;OC1 interrupt (head move timer)
	mov.w	w0,OC1Interrupt
	mov.w	#tbloffset(changeInt),w0       ;CN interrupt (drive ready and PINTE)
	mov.w	w0,CNInterrupt
	mov.w	#tbloffset(sectorInt),w0       ;Timer 1 interrupt (sector timing)
	mov.w	w0,T1Interrupt

; Initialize permanent pointers and values present in some registers

	mov.w	#driveStatus,pDriveStatus    ;permanent pointers
	mov.w	#sectorPos,pSectorPos
	mov.w	#readData,pReadData
	mov.w	#writeData,pWriteData
	mov.w	#trackTable,pTrackTable
	mov.w	#trackBuf,pSector	

	mov.w	#0xffff,wFFFF	;easy to access 0xffff
	clr.w	wDrivesReady	;assume all drives not ready 
	mov.w	#dsINIT_VALUE,wDsInitValue  ;drive status initial value
	clr.w	wCurDrivex2		;default to drive zero

; Initialize variables

	clr.w	flags		;all flags false
	clr.w	byteCount		;byte count in sector
	clr.w	motorTimer		;reset minidisk motor timeout
	mov.w	#STEP_TIMER8,w0	;init step time based on drive type
	btss.w	DriveType,#0	;8" drive?
	mov.w	#STEP_TIMER5,w0	;no, use minidisk stepping rate
	mov.w	w0,stepTime
	bset.w	driveTrack,#15	;force drive:track buffer compare to fail
	mov.w	pTrackTable,w0	;w0->track table
	repeat	#NUM_DRIVES-1	;init all track numbers to zero
	clr.w	[w0++]

    .if (s3Debug)  			;use switch 3 input as an output signal
	mov.w	#0x0000,w0		;port c init
	mov.w	w0,LATC
	mov.w	#0xb000,w0		;port c direction - bit 14 as output
	mov.w	w0,TRISC		;MAKE SURE DRIVE TYPE BIT 3 (SWITCH 4) IS OFF
    .endif

; Sample use of s3 as output for scope. BE SURE S3 is off (rocker down towards
;  top of board).

;    .if s3Debug
;	bset.w	LATC,#C14_DRV_TYPE3_SW
;    .endif

; Enable interrupts and run

	mov.w	PMDIN1,w0		;ensure EPMP is completely reset
	mov.w	PMDIN2,w0
	mov.w	wFFFF,PMDOUT1	;driveStatus and sectorPos to 0xff
	clr.w	PMDOUT2		;data and IO stats zero
	bclr.w	PMSTAT,#IBOV	;make sure errors are cleared
	bclr.w	PMSTAT,#OBUF

	bset.w	IEC2,#PMPIE		;enable parallel port interrupt
	bset.w	IEC0,#T1IE		;enable PIC sector interrupts
	bset.w	IEC1,#CNIE		;enable PINTE line interrupts
	clr.w	SR		;enable PIC interrupts
				;fall into idle loop

;**********************************************************************************
;
;  Idle loop - look for a read track command, the STAT command timer, or a 
;     no activity timeout to occur.
;
;**********************************************************************************
idleLoop:	btsc.w	flags,#fTRACK_READ	;read track requested?
	rcall	readTrack		;yes, read a track
	btst.w	IFS0,#T3IF		;time for stat command?
	bra	z,idleLoop		;no

; Send/receive status with the server. Then check if it's time to flush the write 
;    buffer and invalidate the track buffer (1-2 sec). Invalidating the track buffer
;    forces a server fetch if the same track is requested. This makes user perceived 
;    performance for typed commands more accurate and properly handles a disk swap
;    on the server.

	rcall	sendStat		;send/receive status
	btst.w	DriveType,#0	;8" or minidisk?
	bra	nz,idle8		;8" drive

; Minidisk in use. If the motorTimer has reached 128 (1.6 sec), then invalidate the
;   trackBuffer (forces a server fetch next time) and flush the write buffer if dirty.
;   If motorTimer reaches 512 (6.4 seconds), then de-select the drive.

	disi	#3		;protect next 3 instructions
	btst.w	motorTimer,#mtMINI16	;128's bit set (1.6s)?
	bra	z,chkMotor		;no
	bset.w	driveTrack,#15	;force drive:track buffer compare to fail
	btsc.w	flags,#fTRACK_DIRTY	;track need to be written?
	rcall	writeTrack		;yes
	bra	idleLoop			

; chkMotor - if motorTimer reaches 512, de-select the minidisk

chkMotor:	disi	#7		;protect next 7 instructions
	btst.w	motorTimer,#mtMINI64	;512's bit set (6.4s)?
	bra	z,idleLoop		;no
	clr.w	motorTimer		;reset motor timer (reduces entries here)
	mov.w	wFFFF,statusRegs	;driveStatus and sectorPos to 0xff
	bclr.w	flags,#fINTS_ENABLED	;FDC interrupts to 8080 off
	bclr.w	LATD,#D8_INTERRUPT_OUT	;remove interrupt to 8080
	bclr.w	flags,#fHEAD_LOADED	;head is not loaded
	bra	idleLoop

; idle8 - 8" disk in use. If the motorTimer reaches 256 (1.3 sec) and the head is not
;   loaded, then invalidate the trackBuffer (forces a server fetch next time) and flush
;   the write buffer if dirty. 

idle8:	btst.w	flags,#fHEAD_LOADED	;head loaded?
	bra	nz,idleLoop		;yes, don't do anything
	disi	#4		;protect next 3 instructions
	btst.w	motorTimer,#mtEIGHT13	;256's bit set (1.3s)?
	bra	z,idleLoop		;no
	clr.w	motorTimer		;reset motor timer (reduces entries here)
	bset.w	driveTrack,#15	;force drive:track buffer compare to fail
	btsc.w	flags,#fTRACK_DIRTY	;track buffer dirty?
	rcall	writeTrack		;yes, write the track
	bra	idleLoop

;--------------------------------------------------------------------------------
; sendStat - send drive status request to server
;--------------------------------------------------------------------------------
sendStat:	mov.w	#ASCII_ST,w0	;form "STAT" command in cmdBuf
	mov.w	w0,cmdCommand
	mov.w	#ASCII_AT,w0
	mov.w	w0,cmdCommand+2

	mov.w	#0xff,w0		;w0=head not loaded, no drive selected
	btst.b	driveStatus,#dsREADY    ;is a drive selected?
	bra	nz,1f		;no, w0 is correct
	lsr.w	wCurDrivex2,w0	;w0 has drive number in lsb
	btsc.w	flags,#fHEAD_LOADED	;head loaded?
	sub.w	#0x100,w0		;yes, force MSB to 0xff = loaded
1:	mov.w	w0,cmdParam1
	mov.w	[pTrackTable+wCurDrivex2],w0   ;w0=track number
	mov.w	w0,cmdParam2	;pass track number in param2

	cp0.b	stepTime+1		;already switched to fast rate?
	bra	z,1f		;yes
	sub.w	#NUM_TRACKS8,w0	;past end of a normal 8" drive?
	bra	ltu,1f		;no
	clr.b	stepTime+1		;make step rate fast	

1:	mov.w	#cmdBuf,w1		;w1->command buffer to transmit
	mov.w	#COMMAND_LEN,w2	;w2=length of buffer to transmit
	rcall	sendBuffer		;transmit the command

; Read response to STAT

	mov.w	#rspBuf,w1		;w1->response buffer
	mov.w	#RESPONSE_LEN,w2	;w2->bytes to receive
	rcall	rcvBuffer		;receive the response
	bra	nz,statExit		;error, ignore response
	mov.w	#ASCII_ST,w0	;verify "STAT" response received
	cp.w	rspCommand
	bra	nz,statExit		;not STAT, exit
	mov.w	#ASCII_AT,w0
	cp.w	rspCommand+2
	bra	nz,statExit		;not STAT, exit
	mov.w	rspData,wDrivesReady	;update the drive status bit array

; Verify the server still has the currently selected drive "ready." If not, set
;    drive status and sector position registers to 0xff.

	disi	#6		;protect next 5 instructions
	btst.b	driveStatus,#dsREADY    ;is a drive selected?
	bra	nz,statExit		;no, exit
	lsr.w	wCurDrivex2,w0	;w0=drive number
	btst.z	wDrivesReady,w0	;drive still ready?
	bra	nz,statExit		;yes, just exit
	mov.w	wFFFF,statusRegs	;status and sector regs to 0xff

; statExit - restore STAT command interval timer

statExit:	clr.w	TMR3		;restart timeout
	mov.w	#STAT_RATE,w0	;w0=time per STAT command
	mov.w	w0,PR3
	bclr.w	IFS0,#T3IF		;clear match flag
	return

;--------------------------------------------------------------------------------
; readTrack - read a track from the server
;--------------------------------------------------------------------------------
;  First, see if the existing track buffer must be written before we read a
;  new track over the top of it.

readTrack:	btsc.w	flags,#fTRACK_DIRTY	;track buffer dirty?
	rcall	writeTrack		;yes, write the track

; Update drive:track with the new drive and track that will be buffered. secsBuffed
;    is forced to zero to indicate that no sectors have been received yet.

	clr.w	secsBuffed		;no sectors buffered yet
	sl.w	wCurDrivex2,#11,w1	;drive number in msn of w1
	mov.w	[pTrackTable+wCurDrivex2],w0   ;track number in lsb of w0
	ior.w	w0,w1,w0		;w0=drive in MSN, track in lower 12 bits
	mov.w	w0,driveTrack	;drive and track in the track buffer
	mov.w	w0,cmdParam1	;is also parameter 1 for the command

; Create and send the read command

	mov.w	#ASCII_RE,w0	;form "READ" command in cmdBuf
	mov.w	w0,cmdCommand
	mov.w	#ASCII_AD,w0
	mov.w	w0,cmdCommand+2

	mov.w	#TRACK_LENGTH8,w0	;w0=bytes to read (8" track length)
	btss.w	DriveType,#0	;8" drive?
	mov.w	#TRACK_LENGTH5,w0	;no, used mindisk bytes per track
	mov.w	w0,cmdParam2

	mov.w	#cmdBuf,w1		;w1->command buffer to transmit
	mov.w	#COMMAND_LEN,w2	;w2=length of buffer to transmit
	rcall	sendBuffer		;transmit the command

; Receive the track into trackBuf

	mov.w	#trackBuf,w1	;w1->receiver buffer
	mov.w	#TRACK_LENGTH8,w2	;w2=bytes to read (8" track length)
	btss.w	DriveType,#0	;8" drive?
	mov.w	#TRACK_LENGTH5,w2	;no, used mindisk bytes per track
	rcall	rcvBuffer		;receive the track
	bra	z,1f		;everything ok, exit
	bset.w	driveTrack,#15	;force drive:track to not match
1:	bclr.w	flags,#fTRACK_READ	;read request completed

; Restore STAT command interval timer and exit

	clr.w	TMR3		;restart timeout
	mov.w	#STAT_RATE,w0	;w0=time per STAT command
	mov.w	w0,PR3
	bclr.w	IFS0,#T3IF		;clear match flag
	return		
	
;--------------------------------------------------------------------------------
; writeTrack - write trackBuf to the server
;--------------------------------------------------------------------------------
writeTrack:	mov.w	#WRITE_TRIES,w0	;initialize retry counter
	mov.w	w0,retryCount
	bclr.w	flags,#fTRACK_DIRTY	;indicate trackBuf has been saved

writeLoop:	mov.w	#ASCII_WR,w0	;form "WRIT" command in cmdBuf
	mov.w	w0,cmdCommand
	mov.w	#ASCII_IT,w0
	mov.w	w0,cmdCommand+2

	mov.w	driveTrack,w0	;w0=drive and track combined
	mov.w	w0,cmdParam1	;store drive:track parameter for the command
	bclr.w	cmdParam1,#15	;make sure flag bit is zero

	mov.w	#TRACK_LENGTH8,w0	;store bytes to write (8" track length)
	btss.w	DriveType,#0	;8" drive?
	mov.w	#TRACK_LENGTH5,w0	;no, used mindisk bytes per track
	mov.w	w0,cmdParam2

	mov.w	#cmdBuf,w1		;w1->command buffer to transmit
	mov.w	#COMMAND_LEN,w2	;w2=length of buffer to transmit
	rcall	sendBuffer		;transmit the command

; Wait for server to respond with WRIT command and ready status

	mov.w	#rspBuf,w1		;w1->response buffer
	mov.w	#RESPONSE_LEN,w2	;w2->bytes to receive
	rcall	rcvBuffer		;receive the response
	bra	nz,wrtRetry		;response error, retry
	mov.w	#ASCII_WR,w0	;verify "WRIT" response received
	cp.w	rspCommand
	bra	nz,wrtRetry		;not WRIT, retry
	mov.w	#ASCII_IT,w0
	cp.w	rspCommand+2
	bra	nz,wrtRetry		;not WRIT, retry
	cp0.w	rspCode		;response code "OK" ?
	bra	nz,wrtRetry		;no, retry
	
; Transmit the track buffer to the server

	mov.w	#trackBuf,w1	;w1->transmit buffer
	mov.w	#TRACK_LENGTH8,w2	;w2=bytes to write (8" track length)
	btss.w	DriveType,#0	;8" drive?
	mov.w	#TRACK_LENGTH5,w2	;no, used mindisk bytes per track
	rcall	sendBuffer		;send the track

; Wait for server to respond with WSTA after the buffer is sent

	mov.w	#rspBuf,w1		;w1->response buffer
	mov.w	#RESPONSE_LEN,w2	;w2->bytes to receive
	rcall	rcvBuffer		;receive the response
	bra	nz,wrtRetry		;response error, retry
	mov.w	#ASCII_WS,w0	;verify "WSTA" response received
	cp.w	rspCommand
	bra	nz,wrtRetry		;not WSTA, retry
	mov.w	#ASCII_TA,w0
	cp.w	rspCommand+2
	bra	nz,wrtRetry		;not WSTA, retry
	cp0.w	rspCode		;response code "OK" ?
	bra	nz,wrtRetry		;no, retry

; wrtExit - restore STAT command interval timer and exit

wrtExit:	clr.w	TMR3		;restart timeout
	mov.w	#STAT_RATE,w0	;w0=time per STAT command
	mov.w	w0,PR3
	bclr.w	IFS0,#T3IF		;clear match flag
	return

; wrtRetry - retry the write request until retryCount reaches zero

wrtRetry:	dec.w	retryCount
	bra	nz,writeLoop	;no zero, try again
	bra	wrtExit		;give up and exit

;------------------------------------------------------------------------------------
; sendBuffer - transmit buffer pointed to by w1 of length w2. Sixteen bit checksum
;    is computed and transmitted following the buffer. Clobbers w0-w3
;------------------------------------------------------------------------------------
sendBuffer:	clr.w	w3		;init checksum
sndBufLoop:	mov.b	[w1++],w0		;w0=next byte to send
	ze.w	w0,w0		;make full word from the byte
	add.w	w0,w3,w3		;update the checksum in w3
1:	btst.w	U1STA,#UTXBF	;wait for transmit ready
	bra	nz,1b
	mov.w	w0,U1TXREG		;transmit the character	
	dec.w	w2,w2		;decrement byte count
	bra	nz,sndBufLoop

; send the checksum bytes

1:	btst.w	U1STA,#UTXBF	;wait for transmit ready
	bra	nz,1b
	mov.w	w3,U1TXREG		;send lsb of checksum
	swap.w	w3		;put msb into lsb
1:	btst.w	U1STA,#UTXBF	;wait for transmit ready
	bra	nz,1b
	mov.w	w3,U1TXREG		;send msb of checksum	
	return	

;------------------------------------------------------------------------------------
; rcvBuffer - receive buffer pointed to by w1 of length w2. Length does NOT include
;    the two byte checksum expected at the end of the buffer. Returns zero status
;    if buffer received without error. Otherwise, non-zero if there was a checksum 
;    error or a timeout (RESPONSE_TIMEOUT) occurs. Clobbers w0-w4.
;
;    While receiving the buffer, secsBuffed is incremented by one for every 137
;    bytes received. This will only occur when receiving a track buffer from the
;    server and allows 8080 reads to occur while track data is being received.
;------------------------------------------------------------------------------------
; Clear the receive FIFO and possible over-run error bit (OERR inhibits reception)

rcvBuffer:	mov.w	U1RXREG,w0		;flush UART receive FIFO
	mov.w	U1RXREG,w0
	mov.w	U1RXREG,w0
	mov.w	U1RXREG,w0
	mov.w	U1RXREG,w0		;one extra to be safe
	bclr.w	U1STA,#OERR		;clear possible over-run bit	

; Start timeout (RESPONSE_TIMEOUT) using TMR3. Initialize checksum in w3, bytes
;    received per sector in w4.

	clr.w	TMR3		;restart timeout
	mov.w	#RESPONSE_TIMEOUT,w0	;w0=command response timeout
	mov.w	w0,PR3
	bclr.w	IFS0,#T3IF		;and clear match flag
	clr.w	w3		;initialize checksum
	mov.w	#BYTES_PER_SECTOR,w4	;bytes to receive per sector

; rcvBufLoop - loop receiving bytes until the specified number of bytes is received
;   or a timeout occurs.

rcvBufLoop:	btst.w	U1STA,#URXDA	;byte received?
	bra	nz,rcvByte		;yes
	btst.w	IFS0,#T3IF		;timeout yet?
	bra	z,rcvBufLoop	;no, keep waiting
	return			;return with timeout error (not zero)

; rcvByte - a new byte is available. Move it to the receive buffer and update checksum.

rcvByte:	mov.w	U1RXREG,w0		;w0=new byte
	ze.w	w0,w0		;make sure msb is zero
	add.w	w0,w3,w3		;update checksum in w3
	mov.b	w0,[w1++]		;store the new byte
	dec.w	w4,w4		;count bytes per sector
	bra	nz,1f		;sector not done yet
	mov.w	#BYTES_PER_SECTOR,w4	;restart bytes in sector counter
	inc.w	secsBuffed		;increment # of sectors buffered
1:	dec.w	w2,w2		;decrement byte counter
	bra	nz,rcvBufLoop	;not done yet

; all bytes received. Now receive and verify the checksum

1:	btst.w	U1STA,#URXDA	;byte received?
	bra	nz,rcvChkSum1	;yes, have 1st checksum byte
	btst.w	IFS0,#T3IF		;timeout yet?
	bra	z,1b		;no, keep waiting
	return			;return with timeout error (not zero)	

rcvChkSum1:	mov.w	U1RXREG,w1		;w1=lsb of checksum
	nop			;must wait before checking URXDA again
1:	btst.w	U1STA,#URXDA	;byte received?
	bra	nz,rcvChkSum2	;yes, have 2nd checksum byte
	btst.w	IFS0,#T3IF		;timeout yet?
	bra	z,1b		;no, keep waiting
	return			;return with timeout error (not zero)	

rcvChkSum2:	mov.w	U1RXREG,w0		;w0=msb of checksum
	swap.w	w0		;put msb into the msb
	mov.b	w1,w0		;w0=16 bit checksum
	cp.w	w0,w3		;match computed?
	return			;return with zero status if good

;****************************************************************************************
;
;  ioPortInt - ISR for handling reads and writes to the Altair FDC control and
;     status registers (done through the PIC EPMP). 
;
;     Some register OUTs from 8080 code are immediately followed by a register IN.
;     Response here needs to be less than 2us (to handle up to a 4MHz processor)
;     between an OUT action that then updates a status register.
;
;     This code makes the assumption that it will never get behind, i.e., it can
;     process the interrupts faster than the 8080 will ever do I/O.
;
;****************************************************************************************
;  Check if the drive select register written.

ioPortInt:	bclr.w	IFS2,#PMPIF		;clear the PMP interrupt
	bclr.w	LATD,#D8_INTERRUPT_OUT	;any access removes interrupt to 8080
	btst.w	PMSTAT,#pmSELECT_CMD	;select register written?
	bra	z,chkDriveCmd	;no, go check drive command
	mov.w	wFFFF,statusRegs	;driveStatus and sectorPos to 0xff
	mov.b	selectCmd,WREG	;w0=select command, clear status flag
	bclr.w	flags,#fINTS_ENABLED	;FDC interrupts to 8080 off
	bclr.w	flags,#fHEAD_LOADED	;head is not loaded
	btst.z	w0,#scDESELECT	;de-select requested?
	bra	nz,ioSelExit	;yes, de-select already done, just exit

; Drive select requested

	and.w	#scDRIVE_MASK,w0	;get drive number alone
	sl.w	w0,wCurDrivex2	;save new drive number*2
	btst.z	wDrivesReady,w0	;drive ready?
	bra	z,ioSelExit		;no, just exit

; Update the drive status and sector position registers.

	mov.b	wDsInitValue,[pDriveStatus]   ;init drive status register
	mov.w	[pTrackTable+wCurDrivex2],w0    ;w0=track number for current drive
	cp0.w	w0		;on track zero?
	bra	nz,1f		;no
	bclr.b	driveStatus,#dsTRACK0	;otherwise, assert track zero
1:	btsc.w	PORTF,#F0_BUS_PINTE	;PINTE signal from bus asserted?
	bclr.b	driveStatus,#dsINTE	;yes, assert INTE in drive status
	clr.w	motorTimer		;restart the power down/activity timer
	btst.w	DriveType,#0	;8" drive?
	bra	nz,ioSelExit	;yes, 8" drive so we're done
	bclr.b	driveStatus,#dsHEAD_STATUS  ;otherwise, minidisk has head loaded
	bset.w	flags,#fHEAD_LOADED	;internal flag that head is loaded
ioSelExit:	pop.w	w0
	retfie

;-------------------------------------------------------------------------------------------
; chkDriveCmd - Drive command register written? If so, process it. Simultaneous
;    commands are possible, but some combinations are meaningless. The following
;    logic is implemented:
;
;  if (step in/out or head load/unload) {
;     if (step out)
;        step out
;     else if (step in)
;        step in
;     if (head unload)
;        unload head
;     if (head load)
;        load head
;  }
;  if (write enable)
;     set internal write enable flag
;  if (interrupt disable)
;     disable interrupts
;  else if (interrupt disable)
;     disable interrupts
;   
;-------------------------------------------------------------------------------------------
chkDriveCmd: btst.w	PMSTAT,#pmDRIVE_CMD	;drive command written?
	bra	z,chkReadData	;no, see if read data register read
	mov.b	driveCmd,WREG	;w0=drive command, clear status flag
	btst.b	driveStatus,#dsREADY	;do we show is drive ready?
	bra	nz,ioCmdExit	;no, ignore the command
	push.w	w1		;save w1
	
; Check for any head step or head load/unload commands first. If any of these, immediately
;    disable the sector position register (0xff). This is done because 8080 software may 
;    quickly read the status registers after writing this command.

	and.b	w0,#dcHEAD_MASK,w1	;any of the head move/load commands?
	bra	z,ioChkWrite	;no, go check write enable
	btsc.w	DriveType,#0	;8" drive?
	mov.b	wFFFF,[pSectorPos]	;yes, force sector register to 0xff
	and.b	w0,#dcMOVE_MASK,w1	;any of the head move commands?
	bra	z,ioChkHead		;no, must be head load/unload
	mov.b	wFFFF,[pSectorPos]	;sector register to 0xff for minidisk too	
	bset.b	driveStatus,#dsMOVE_HEAD     ;de-assert move head flag
	bset.b	driveStatus,#dsHEAD_STATUS   ;de-assert head status flag
	clr.w	motorTimer		;restart motor timeout with a step
	mov.w	[pTrackTable+wCurDrivex2],w1   ;w1=track number of current drive

;=================
; Step Out command
;=================
	btst.z	w0,#dcSTEP_OUT	;step out command?
	bra	z,ioStepIn		;no, must be step in command
	cp0.w	w1		;already at track 0?
	bra	z,ioDoTimer		;yes, don't decrement track number
	dec.w	w1,w1		;decrement track number
	bra	nz,1f		;reach track zero?
	bclr.b	driveStatus,#dsTRACK0	;yes, assert track zero flag
1:	mov.w	w1,[pTrackTable+wCurDrivex2]   ;save updated track
	bra	ioDoTimer		;go set head step timer

;=================
; Step In command. Maximum track validation was removed with addition of the 8Mb drive.
;=================
ioStepIn:	bset.b	driveStatus,#dsTRACK0	;no longer on track zero
	inc.w	w1,w1		;increment track number
	mov.w	w1,[pTrackTable+wCurDrivex2]   ;save updated track

ioDoTimer:	push	w0		;preserve w0
	mov.w	stepTime,w0		;w0=step delay time in use
	add.w	OC1TMR,WREG		;add timeout to current OC1 timer
	mov.w	w0,OC1R		;store as new OC1 compare value
	pop	w0		;restore w0
	bclr.w	IFS0,#OC1IF		;clear OC1 interrupt flag
	bset.w	IEC0,#OC1IE		;enable OC1 head step interrupt
    .if (s3Debug)
	bset.w	LATC,#C14_DRV_TYPE3_SW
    .endif

;=================
; Head Unload command. De-assert the head status flag for the 8" drive. For the mindisk,
;    do nothing.
;=================
ioChkHead:	btst.z	w0,#dcHEAD_UNLOAD	;head unload command?
	bra	z,ioChkLoad		;no
	btst.w	DriveType,#0	;8" drive?
	bra	z,ioChkLoad		;no, skip next statements
	bclr.w	flags,#fHEAD_LOADED	;internal flag that head no longer loaded
	bset.b	driveStatus,#dsHEAD_STATUS   ;de-assert head status signal

;=================
; Head Load command. Assert the head status flag for the 8" drive. For minidisk,
;    restart the drive timeout instead.
;=================
ioChkLoad:	btst.z	w0,#dcHEAD_LOAD	;head load command?
	bra	z,ioChkWrite	;no
	clr.w	motorTimer		;restart the power down timer
	btst.w	DriveType,#0	;8" drive?
	bra	z,ioChkWrite	;no
	bset.w	flags,#fHEAD_LOADED	;internal flag that head is loaded
	bclr.b	driveStatus,#dsHEAD_STATUS  ;assert head status

;=================
; Write enable command. The sector pointer and byte count should have already been
;    set up by the 8080 program reading the sector position register and seeing
;    sector true. We just assert ENWD here which will start the write process with
;    the chkWriteData entry processing each byte written
;=================
ioChkWrite:	btst.z	w0,#dcWRITE_ENABLE	;write enable command?
	bra	z,ioChkInt		;no, go on
	bset.w	flags,#fTRACK_DIRTY	;flag track has been written to
	bclr.b	driveStatus,#dsENWD	;assert enter new write data
	
;=================
; Interrupt enable/disable. These commands set or clear the flag looked at by sector
;    index processing. When disabling, the output signal is immediately turned off
;    as well.
;=================
ioChkInt:	pop.w	w1		;restore w1
	btst.z	w0,#dcINT_DISABLE	;disable interrupt command?
	bra	z,ioChkEnable	;no, go check enable
	bclr.w	flags,#fINTS_ENABLED	;FDC interrupts to 8080 disabled
	bclr.w	LATD,#D8_INTERRUPT_OUT	;de-assert 8080 interrupt
	bra	ioCmdExit		;exit (disable overrides an enable command)

; Check for interrupt enable command 

ioChkEnable: btsc.w	w0,#dcINT_ENABLE	;enable interrupt command?
	bset.w	flags,#fINTS_ENABLED	;yes, enable FDC interrupts to 8080
ioCmdExit:	pop	w0
	retfie

;-------------------------------------------------------------------------------------------
; chkReadData - See if the data byte has been read from the controller. If so, move
;   in the next byte. Storing a new byte clears the pmREAD_DATA flag. The logic
;   below de-asserts NRDA after 137th byte is read but continues to update the
;   readData register so pmREAD_DATA is always cleared.
;-------------------------------------------------------------------------------------------
chkReadData: btst.w	PMSTAT,#pmREAD_DATA	;read data register read?
	bra	z,chkWriteData	;no, go check for write data
	inc.w	byteCount		;increment byte counter
	mov.w	#BYTES_PER_SECTOR,w0	;w0=maximum bytes in sector
	cp.w	byteCount		;reach max bytes for sector?
	bra	ltu,1f		;not max yet
	bset.b	driveStatus,#dsNRDA	;de-assert data available if past end
1:	mov.b	[++pSector],[pReadData]	;store new byte for 8080 to read
	pop	w0		;restore and exit
	retfie

;-------------------------------------------------------------------------------------------
; chkWriteData - See if a data byte has been written for the drive to write. If so,
;    move it to the trackBuffer as pointed to by pSector. Reading the register
;    clears the pmWRITE_DATA flag. ENWD is left low so bytes past 137 are accepted,
;    but they are not stored.
;-------------------------------------------------------------------------------------------
chkWriteData: btst.w	PMSTAT,#pmWRITE_DATA	;write data register written?
	bra	z,chkSector		;no, check sector position register
	inc.w	byteCount		;increment byte counter
	mov.w	#BYTES_PER_SECTOR,w0	;w0=maximum bytes in sector
	cp.w	byteCount		;reach max bytes for sector?
	bra	gtu,writeSkip	;past max bytes, don't update
	mov.b	[pWriteData],[pSector++]  ;store next byte
	pop	w0
	retfie	

; writeSkip - past 137th byte, skip the write, but load the writeData register
;   to clear the pmWriteData flag

writeSkip:	mov.b	writeData,WREG	;clear pmWRITE flag
	pop	w0		;restore and exit
	retfie

;-------------------------------------------------------------------------------------------
; chkSector - See if the sector position register has been read.
;     8080 code loops reading the sector position register to find the desired
;     sector number as it comes under the head. When required, this is where we
;     stall the 8080 code to read a new track of data from the server.
;
;     IMPORTANT: This routine is entered AFTER the sector position is read
;     Processing of sector number, buffer pointer, etc. must be done based
;     on the sector number the 8080 just read from the sector position register.
;-------------------------------------------------------------------------------------------
chkSector:	btst.w	PMSTAT,#pmSECTOR_POS	;sector position register read?
	bra	z,secPosExit	;no, exit
	bset.b	driveStatus,#dsENWD	;make sure new write data flag is off
	btsc.w	DriveType,#0	;8" drive?
	clr.w	motorTimer		;yes, use this access as an activity indicator	

; If sector true was asserted in the value just read by the 8080 from the sector
;    register, then initialize pointers to the proper sector in trackBuf, put the
;    first byte in the read data register and assert new read data available.

	btst.b	sectorPos,#spSECTOR_TRUE   ;was sector true asserted?
	bra	nz,nextSPValue	   ;no, go determine next SP value
	lsr.b	sectorPos,WREG	;get sector number in w0
	and.w	#SECTOR_WRAP8,w0	;wrap to 32 sectors for 8" drive
	btss.w	DriveType,#0	;lsb is 1 for 8" drive
	and.w	#SECTOR_WRAP5,w0	;wrap to 16 sectors for mindisk
	clr.w	byteCount		;restart byte count at zero
	push	w1		;save w1
	mov.w	#BYTES_PER_SECTOR,w1	;w1=bytes per sector
	mul.uu	w0,w1,w0		;w0=sector num * sector length
	mov.w	#trackBuf,w1	;w1->track buffer
	add.w	w0,w1,pSector	;pSector->start of sector in trackBuf
	pop	w1		;restore w1
	mov.b	[pSector],[pReadData]	;put out 1st byte for 8080 to read
	bclr.b	driveStatus,#dsNRDA	;assert read data available

;  nextSPValue - Determine next sector position register value. If the current drive:track is
;     not in trackBuf, then request a read of the current track from the server, force sector
;     number to max so it will roll over to zero when data is ready. Sector True is de-asserted.

nextSPValue: push	w1		;save w1
	sl.w	wCurDrivex2,#11,w1	;put drive number in msn of w1
	mov.w	[pTrackTable+wCurDrivex2],w0   ;w0=track number
	ior.w	w0,w1,w0		;w0=drive:track
	pop	w1		;restore w1
	cp.w	driveTrack		;match buffered drive:track?
	bra	z,haveTrack		;yes, proper track buffered or buffering
	bset.w	flags,#fTRACK_READ	;otherwise, request read of new track	
	mov.b	wFFFF,[pSectorPos]	;sector true not asserted
	pop	w0		;restore and exit
	retfie

;  haveTrack - The proper track is buffered or buffering in trackBuf. If time for a new
;     sector based on the SECTOR_START flag, increment the sector number internally and 
;     see if the new sector has been buffered yet.

haveTrack:	btst.w	flags,#fSECTOR_START	;time for a new sector?
	bra	z,sameSector	;no, stay on same sector
	lsr.b	sectorPos,WREG	;get current sector number in W0
	inc.w	w0,w0		;increment to next sector
	and.w	#SECTOR_WRAP8,w0	;wrap to 32 sectors for 8" drive
	btss.w	DriveType,#0	;lsb is 1 for 8" drive
	and.w	#SECTOR_WRAP5,w0	;wrap to 16 sectors for mindisk
	cp.w	secsBuffed		;current sector buffered yet?
	bra	leu,sameSector	;no, stay on same sector

;  The new sector has been buffered, update the sector number and assert sector true	
			
	sl.w	w0,w0		;put sector number in bits 5-1
	ior.b	#spSECTOR_TRUE_MASK,w0	;set unused bits, leave sector true asserted
	mov.b	WREG,sectorPos	;store new sector position register
	bclr.w	flags,#fSECTOR_START	;clear the new sector flag
secPosExit:	pop	w0		;restore and exit
	retfie

sameSector:	bset.b	sectorPos,#spSECTOR_TRUE   ;de-assert sector true
	pop	w0
	retfie

;****************************************************************************************
;
;  headMoveInt - ISR to process expiration of OC1. This interrupt occurs stepTime ms
;      after a step in/out command is issued. Assert the move head bit in the drive
;      status register if the drive is still ready.
;
;****************************************************************************************
headMoveInt: bclr.w	IEC0,#OC1IE		    ;disable further interrupts
 	btss.b	driveStatus,#dsREADY	    ;drive still ready?
	bclr.b	driveStatus,#dsMOVE_HEAD    ;yes, assert move head signal
	btsc.w	flags,#fHEAD_LOADED	    ;head still loaded?
	bclr.b	driveStatus,#dsHEAD_STATUS  ;yes, assert head status signal
    .if (s3Debug)
	bclr.w	LATC,#C14_DRV_TYPE3_SW
    .endif
	pop.w	w0
	retfie

;****************************************************************************************
;
;  changeInt - ISR to process port change interrupts. 
;     The PINTE line from the 8080 (interrupt enabled signal) generates an interrupt
;     so the INTE status bit in the drive status register can be updated.
;
;****************************************************************************************
changeInt:	bclr.w	IFS1,#CNIF		;clear the port change interrupt flag
	btst.w	PORTF,#F0_BUS_PINTE	;PINTE signal from bus asserted?
	bra	z,pInteOff		;no	
	btss.b	driveStatus,#dsREADY	;skip next inst if drive not ready
	bclr.b	driveStatus,#dsINTE	;assert INTE in drive status
	pop.w	w0		;restore w0 and exit
	retfie

; pInteOff - PINTE not asserted. De-assert INTE in the drive status register

pInteOff:	bset.b	driveStatus,#dsINTE	;de-assert INTE in drive status
	pop.w	w0		;restore w0 and exit
	retfie

;****************************************************************************************
;
;  sectorInt - ISR to process 5.2ms (8" drive) or 12.5ms (mindisk) sector interrupt 
;     This interrupt is used to generate realistic sector timing by setting
;     the SECTOR_START flag (looked at by sector position register processing) to true.
;     This interrupt also generates sector interrupts to the 8080 when enabled.
;
;****************************************************************************************
sectorInt:	bclr.w	IFS0,#T1IF		;clear the timer 1 interrupt
	btst.b	driveStatus,#dsREADY	;drive ready?
	bra	nz,1f		;no, just exit
	btsc.w	flags,#fINTS_ENABLED	;FDC interrupts to 8080 enabled?
	bset.w	LATD,#D8_INTERRUPT_OUT	;yes, assert the 8080 interrupt
	bset.w	flags,#fSECTOR_START	;assert sector start
	inc.w	motorTimer		;increment mindisk motor timeout
1:	pop	w0
	retfie

	.end
