;****************************************************************************************
;
;  MONITOR.S
;      This module contains the code that allows the user to modify some properties
;      of the FDC+
;
;  Version History
;    1.0	03/05/15  M. Douglas
;	- Original		
;    1.1    02/27/16  M. Douglas
;	- Support user-selectable stepping rates and index offset. Add restore
;	  defaults menu option.
;    1.2    06/19/20  M. Douglas
;	- Add parameter support for the new 5.25 high density drive option
;    1.3    01/17/21  M. Douglas
;	- The zero index timing offset (INDEX_0) in common.inc was changed to
;	  8us instead of zero. A non-zero value is required because processing
;	  time must be allowed from when the end of sector routine computes the
;	  next sector wakeup until the sector hunt routine is entered to wait for
;	  that wakeup. The menu text for the zero offset option was also updated
;	  to indicate it is for Pertec drives (e.g., the FD514). These changes
;	  only affect operation of the Shugart-800 disk type option.
;    1.4	09/05/21  M. Douglas
;	- Add support for iCOM FD3712 and FD3812.
;    1.5  	01/13/26  M. Douglas
;	- The MonitorMode entry point in this module sets the SPLIM register
;	  (stack limit) to ensure SPLIM agrees with the memory map of the 
;	  currently programmed version of the application. Previously, SPLIM
;	  was set only by the bootloader, which in turn, is not updated when
;	  a user does a firmware update.
;
;****************************************************************************************
	.include 	"common.inc"   

; equates
	.equ	MAX_IN_LEN,19	;maximum number of characters to read
	.equ	CR,13		;ascii carriage return
	.equ	LF,10		;ascii line feed
	.equ	BS,8		;ascii back space character

; flag word equates

	.equ	fUPDATE,0		;1=update parameter flash on exit

; variables

	.section *,bss,near
flags:	.space	2		;flags per bit (see equates above)
inbuf:	.space	MAX_IN_LEN+1	;buffer for terminal read operations

	.text
;---------------------------------------------------------------------------
; Main Menu
;---------------------------------------------------------------------------
sMainMenu1:	.ascii	"\r\n\n===== Altair FDC+ Monitor =====\r\n\n"
	.asciz	"Drive Type: "

sMainMenu2:	.ascii	"\r\n\n1) Drive Timing Parameters\r\n"
	.ascii	"2) Serial Drive Baud Rate\r\n"
	.ascii	"3) Firmware Version Information\r\n"
	.ascii	"4) Restore Defaults\r\n"
	.ascii	"5) Update Firmware\r\n"
	.ascii	"x) Exit Monitor and Start FDC+\r\n\n"
	.asciz	"Enter Choice: "

;---------------------------------------------------------------------------
; Serial drive baud rate menu
;---------------------------------------------------------------------------
sBaudRate1:	.ascii	"\r\n===== Serial Drive Baud Rate =====\r\n\n"
	.asciz	"Current Setting: "

sBaudRate2:	.ascii	"1) 403.2K (preferred)\r\n"
	.ascii	"2) 460.8K (2nd choice)\r\n"
	.ascii	"3) 230.4K (compatibility fall-back)\r\n"
	.ascii	"x) Exit this Menu\r\n\n"
	.asciz	"Enter Choice: "

s403200:	.asciz	"403.2K Baud\r\n\n"
s460800:	.asciz	"460.8K Baud\r\n\n"
s230400:	.asciz	"230.4K Baud\r\n\n"

;---------------------------------------------------------------------------
; Module version display
;---------------------------------------------------------------------------
sVersions:	.ascii	"\r\n\n===== Module Version Information =====\r\n\n"
	.ascii	"Module          Version     Date\r\n"
	.asciz	"---------------  ------  ----------\r\n" 

;---------------------------------------------------------------------------
; Drive timing parameters
;---------------------------------------------------------------------------
sDtMenu:	.ascii	"\r\n\n===== Drive Timing Parameters =====\r\n\n"
	.ascii	"1) Step timing\r\n"
	.ascii	"2) Index timing\r\n"
	.ascii	"x) Exit this Menu\r\n\n"
	.asciz	"Enter Choice: "

sDtFixed:	.ascii	"Drive timing is fixed for the current drive type\r\n\n"
	.asciz	"Press RETURN to Continue: "

;---------------------------------------------------------------------------
; Drive index timing
;---------------------------------------------------------------------------
sIndexHead:	.ascii	"\r\n\n===== Drive Index Timing =====\r\n\n"
	.asciz	"Current index offset: "

sIndexMenu:	.ascii	"\r\n\n"
	.ascii	"1) 460us\r\n"
	.ascii	"2) 360us (default)\r\n"
	.ascii	"3) 260us\r\n"
	.ascii	"4) 160us\r\n"
	.ascii	"5) 60us\r\n"
	.ascii	"6) 0us (for Pertec drives)\r\n"
	.ascii	"x) Exit this Menu\r\n\n"
	.asciz	"Enter Choice: "

; index offset string table, indexed by menu number

sIndexTbl:	.word	s460us,s360us,s260us,s160us,s60us,s0us,0

s460us:	.asciz	"460us"
s360us:	.asciz	"360us"
s260us:	.asciz	"260us"
s160us:	.asciz	"160us"
s60us:	.asciz	"60us"
s0us:	.asciz	"0us"

; index offset table, indexed by menu number

indexTable:	.word	INDEX_460,INDEX_360,INDEX_260
	.word	INDEX_160,INDEX_60,INDEX_0

;---------------------------------------------------------------------------
; Drive step timing
;---------------------------------------------------------------------------
sStepHead:	.ascii	"\r\n\n===== Drive Step Timing =====\r\n\n"
	.asciz	"Current step time: "

sStepMenu:	.ascii	"\r\n\n"
	.ascii	"1) 50ms (Altair Mini Disk)\r\n"
	.ascii	"2) 25ms\r\n"
	.ascii	"3) 10.5ms (Altair 8 inch)\r\n"
	.ascii	"4) 8ms\r\n"
	.ascii	"5) 6ms\r\n"
	.ascii	"6) 3ms\r\n"
	.ascii	"x) Exit this Menu\r\n\n"
	.asciz	"Enter Choice: "

; step-time string table, indexed by menu number

sStepTbl:	.word	s50ms,s25ms,s10ms,s8ms,s6ms,s3ms,0

s50ms:	.asciz	"50ms"
s25ms:	.asciz	"25ms"
s10ms:	.asciz	"10.5ms"
s8ms:	.asciz	"8ms"
s6ms:	.asciz	"6ms"
s3ms:	.asciz	"3ms"

; step-time table, indexed by menu number

stepTable:	.word	STEP_TIME48, STEP_TIME25, STEP_TIME10
	.word	STEP_TIME8, STEP_TIME6, STEP_TIME3

;---------------------------------------------------------------------------
; Other Messages
;---------------------------------------------------------------------------
sFdcRunning: .asciz	"FDC+ Running..."

sRestore:	.ascii	"\r\nThis will reset all parameters to their default values"
	.asciz	"\r\n\nAre you sure? "

sResDone:	.asciz	"Default values restored and saved\r\n"

sIndexCal:	.asciz	"\r\nIndex Calibration Time in hex us: "
sContinue:	.asciz	"\r\nPress RETURN to Continue: "
sCrLf:	.asciz	"\r\n"

;---------------------------------------------------------------------------
; Tables indexed by drive type
;---------------------------------------------------------------------------

; drv2Name - 16 pointers to drive type strings indexed by DriveType

drv2Name:	.word	sAltair8,sShugart8,sAltairMini,sShugart5
	.word	sSoft5as8,sHdFloppy,sSerial5,sSerial8
	.word	sFD3712,sFD3812,sAltairMini,sShugart5
	.word	sSoft5as8,sHdFloppy,sSerial5,sSerial8

sAltair8:	.asciz	"Original Altair 8\" Drive"
sShugart8:	.asciz	"Shugart Compatible 8\" Drive as Altair"
sAltairMini: .asciz	"Original Altair Minidisk"
sShugart5:	.asciz	"5.25\" Drive as Altair Minidisk"
sSoft5as8:	.asciz	"5.25\" Drive as Altair 8\" Drive"
sHdFloppy:	.asciz	"1.5Mb Floppy Drive"
sSerial8:	.asciz	"Altair 8\" Serial Drive"
sSerial5:	.asciz	"Altair Mindisk Serial Drive"
sFD3712:	.asciz	"iCOM FD3712 (Single Density)"
sFD3812:	.asciz	"iCOM FD3812 (Double Density)"

; drv2Step - pointers to the proper step-time variable for each
;   DriveType. A zero pointer indicates the a step-time parameter
;   is not used.

drv2Step:	.word	0,ptStepT8,0,ptStepT5
	.word	ptStepT58,ptStepTHD,0,0
	.word	ptStep3712,ptStep3812,0,ptStepT5
	.word	ptStepT58,ptStepTHD,0,0

;************************************************************************************
;
;   MonitorEntry
;        Display and proess the main menu.
;
;************************************************************************************
	.global	MonitorEntry
MonitorEntry: 
	mov.w	#STACK_ADDR,w15	;init stack pointer
	mov.w	#STACK_END,w0	;stack pointer limit
	mov.w	w0,SPLIM	
	clr.w	flags

;  Main menu loop 

mainMenu:	mov.w	#tbloffset(sMainMenu1),w0  ;display the main menu title
	rcall	DispString
	rcall	dispDrive		   ;display drive type
	mov.w	#tbloffset(sMainMenu2),w0  ;display remainder of main menu
	rcall	DispString

	mov.w	#inbuf,w0
	rcall	inputString

	mov.b	inbuf,WREG		;w0 = menu response
	sub.b	#'0',w0		;force ascii 1-9 to binary 1-9
	cp.b	w0,#1
	bra	z,driveParams	;set drive timing parameters
	cp.b	w0,#2
	bra	z,baudRate		;serial drive baud rate
	cp.b	w0,#3
	bra	z,dispVers		;display version information
	cp.b	w0,#4
	bra	z,restoreDef	;restore default parameters
	cp.b	w0,#5
	bra	z,updateFirm	;update firmware

; Force specific entry of "x" or "X" to exit monitor

	mov.b	inbuf,WREG		;w0=operator's entry
	ior.b	#0x20,w0		;force lower case
	sub.b	#'x',w0		;"X" exit requested?
	bra	z,enterFDC		;exit monitor, enter FDC mode

	bra	mainMenu		;invalid choice

;---------------------------------------------------------------------------
;  enterFDC - display "running" message and enter normal FDC processing
;---------------------------------------------------------------------------
enterFDC:	mov.w	#tbloffset(sFdcRunning),w0
	rcall	DispString		;display "FDC+ Running" message
1:	btst.w	U1STA,#TRMT		;wait till transmission finished
	bra	z,1b		;   before we exit
	bra	NormalMode

;---------------------------------------------------------------------------
;  driveParams - update drive timing parameters
;---------------------------------------------------------------------------
; If the drive type doesn't have adjustable parameters, display a message
;    and exit.

driveParams: mov.w	#tbloffset(drv2Step),w4	     ;w4->drv2Step variable table
	sl.w	DriveType,WREG	;w0=DriveType*2	
	add.w	w0,w4,w0		;w0->drv2Step[DriveType]
	tblrdl.w	[w0],w4		;w4->step variable for drive
	cp0.w	w4		;if zero, no timing option
	bra	z,dpNone

; If the drive isn't the 8" shugart option, there is not an index timing
;    option, so jump directly to display the step timing menu

	mov.w	#ptStepT8,w0	;w0->8" shugart step variable
	cp.w	w0,w4		;is this the one we're using?
	bra	nz,dpStep		;no, go straight to step menu

; Display and process the main drive timing menu

	mov.w	#tbloffset(sDtMenu),w0   ;display menu
	rcall	DispString
	
	mov.w	#inbuf,w0		;read user response
	rcall	inputString

	mov.b	inbuf,WREG		;w0 = menu response
	sub.b	#'0',w0		;force ascii 1-9 to binary 1-9
	cp.b	w0,#1		;step timing?
	bra	z,dpStep
	cp.b	w0,#2		;index timing?
	bra	z,dpIndex

	bra	mainMenu		;other choice, exit

; dtExit - exiting to main menu, write flash parameter memory if a change has been made

dtExit:	btsc.w	flags,#fUPDATE	;should parameter flash be udpated?
	rcall	WriteParams		;yes, update parameter flash
	bclr.w	flags,#fUPDATE	;clear the flag
	bra	mainMenu		;return to main menu

; dpNone - drive parameters not available for current drive type, display message, wait
;    for user to press return and exit.

dpNone:	mov.w	#tbloffset(sDtFixed),w0 
	rcall	DispString		;"drive timing can't be changed"

	mov.w	#inbuf,w0		;wait for user to press return
	call	inputString
	bra	mainMenu		;return to main menu

;---------------------------------------------------------------------------
; dpIndex - adjust the index offset parameter
;---------------------------------------------------------------------------
dpIndex:	mov.w	#tbloffset(sIndexHead),w0  ;display heading and 
	rcall	DispString		   ;    "Current index offset: "

	rcall	iTime2Str		;w0->index time string from index time
	rcall	DispString		;display the current setting

	mov.w	#tbloffset(sIndexMenu),w0
	rcall	DispString		;display the index offset menu
	
; Read new index offset selection

	mov.w	#inbuf,w0		;read user's response into inbuf
	call	inputString

	mov.b	inbuf,WREG		;w0 = menu response
	sub.b	#'1',w0		;force ascii 1-9 to binary 0-8
	cp.b	w0,#5		;valid selection (0-5)?
	bra	gtu,dtExit		;no, exit

	mov.w	#tbloffset(indexTable),w1   ;w1->table of index times by menu num
	ze.w	w0,w0		;make word out of menu selection
	sl.w	w0,w0		;w0=word offset
	add.w	w0,w1,w0		;w0->indexTable[menu selection]
	tblrdl.w	[w0],w0		;w0=new index time from table
	mov.w	w0,ptIndexCal	;save new index time
	bset.w	flags,#fUPDATE	;request param flash update on exit
	bra	dpIndex		;re-display menu
	
;---------------------------------------------------------------------------
; dpStep - adjust the step time parameter
;---------------------------------------------------------------------------
dpStep:	mov.w	#tbloffset(sStepHead),w0  ;display heading and 
	rcall	DispString		  ;    "Current step time: "

	rcall	sTime2Str		;get w0->step time string from step rate
	rcall	DispString		;display the current setting

	mov.w	#tbloffset(sStepMenu),w0
	rcall	DispString		;display the step time menu
	
; Read new step time selection

	mov.w	#inbuf,w0		;read user's response into inbuf
	call	inputString

	mov.b	inbuf,WREG		;w0 = menu response
	sub.b	#'1',w0		;force ascii 1-9 to binary 0-8
	cp.b	w0,#5		;valid selection (0-5)?
	bra	gtu,dtExit		;no, exit

	mov.w	#tbloffset(stepTable),w1  ;w1->table of step times by menu number
	ze.w	w0,w0		;make word out of menu selection
	sl.w	w0,w0		;w0=word offset
	add.w	w0,w1,w0		;w0->stepTable[menu selection]
	tblrdl.w	[w0],w0		;w0=new step time from table
	mov.w	w0,[w4]		;save new step time in ptStep varible
	bset.w	flags,#fUPDATE	;request param flash update on exit
	bra	dpStep		;re-display menu

;---------------------------------------------------------------------------
; iTime2Str - convert index time in ptIndexCal into string pointer in w0.
;    If index time does not match a valid time, it is forced to INDEX_360.
; On exit:
;    w0->string of index time
;    Clobbers w0-w4
;---------------------------------------------------------------------------
iTime2Str:	mov.w	ptIndexCal,w4	;w4=current index timing
	mov.w	#tbloffset(sIndexTbl),w2    ;w2->string table
	mov.w	#tbloffset(indexTable),w3   ;w3->time value table

i2sLoop:	tblrdl.w	[w2++],w0		;w0->time string
	cp0.w	w0		;end of table?
	bra	z,i2sErr		;yes, no match found
	
	tblrdl.w	[w3++],w1		;w1=corresponding index time
	cp.w	w1,w4		;index time match?
	bra	nz,i2sLoop		;no, check next

	return			;match found, w0->string

; i2sErr - no match found, force ptIndexCal to default of 360us

i2sErr:	mov.w	#tbloffset(s360us),w0	;w0->360us string
	mov.w	#INDEX_360,w1	;force index time to 360us
	mov.w	w1,ptIndexCal	;update the ptStep variable
	bset.w	flags,#fUPDATE	;request param flash update on exit
	return	

;---------------------------------------------------------------------------
; sTime2Str - convert step time in (w4) into string pointer in w0. If
;    step time does not match a valid time, it is forced to STEP_TIME10
;    for 8" drives and to STEP_TIME48 for 5.25" drives.
; On Entry:
;    w4->step time variable for current drive
; On exit:
;    w0->string of step time
;    (w4) updated with valid time if previously invalid
;    Clobbers w0-w3
;---------------------------------------------------------------------------
sTime2Str:	mov.w	#tbloffset(sStepTbl),w2    ;w2->string table
	mov.w	#tbloffset(stepTable),w3   ;w3->time value table

s2sLoop:	tblrdl.w	[w2++],w0		;w0->time string
	cp0.w	w0		;end of table?
	bra	z,s2sErr		;yes, no match found
	
	tblrdl.w	[w3++],w1		;w1=corresponding step time
	cp.w	w1,[w4]		;step time match?
	bra	nz,s2sLoop		;no, check next

	return			;match found, w0->string

; s2sErr - no match found, force (w4) to a valid time based on the drive type	

s2sErr:	mov.w	#ptStepT5,w0	;see if w4->ptStepT5 (5.25" drive)
	cp.w	w0,w4
	bra	z,s2sDrive5		;yes

; Use 10.5ms for an 8" drive

	mov.w	#tbloffset(s10ms),w0	;w0->10.5ms string
	mov.w	#STEP_TIME10,w1	;force step time to 10.5ms
	bra	s2sExit		;update and exit

; Force 50ms for a 5.25" drive

s2sDrive5:	mov.w	#tbloffset(s50ms),w0	;w0->10.5ms string
	mov.w	#STEP_TIME48,w1	;force step time to 50ms

s2sExit:	mov.w	w1,[w4]		;update the ptStep variable
	bset.w	flags,#fUPDATE	;request param flash update on exit
	return	

;-----------------------------------------------------------------------------------
;  baudRate - set serial drive baud rate
;-----------------------------------------------------------------------------------
baudRate:	mov.w	#tbloffset(sBaudRate1),w0
	rcall	DispString		;display baud rate title
	mov.w	ptBaudRate,w1	;w1=BRG value for current baud rate
	mov.w	#tbloffset(s403200),w0	;w0->403.2K baud message	
	cp.w	w1,#BAUD_403200	;match?
	bra	z,dispBaud		;yes
	mov.w	#tbloffset(s460800),w0	;w0->460.8K baud message	
	cp.w	w1,#BAUD_460800	;match?
	bra	z,dispBaud		;yes
	mov.w	#tbloffset(s230400),w0	;w0->230.4K baud message	
	cp.w	w1,#BAUD_230400	;match?
	bra	z,dispBaud		;yes

; Invalid baud rate detected, force it to 403.2K

	mov.w	#BAUD_403200,w1
	mov.w	w1,ptBaudRate
	mov.w	#tbloffset(s403200),w0	;w0->403.2K baud message	

; dispBaud - display the baud rate message pointed to by w0, then display the
;   rest of the baud rate menu.

dispBaud:	rcall	DispString		;display current baud rate
	mov.w	#tbloffset(sBaudRate2),w0
	rcall	DispString		;display rest of baud rate menu

; Read new baud rate selection

	mov.w	#inbuf,w0
	call	inputString
	mov.b	inbuf,WREG		;w0 = menu response
	sub.b	#'0',w0		;force ascii 1-9 to binary 1-9
	mov.w	#BAUD_403200,w1
	cp.b	w0,#1		;403.2K?
	bra	z,haveBaud		;yes
	mov.w	#BAUD_460800,w1
	cp.b	w0,#2		;460.8K?
	bra	z,haveBaud		;yes
	mov.w	#BAUD_230400,w1
	cp.b	w0,#3		;230.4K?
	bra	z,haveBaud		;yes

; Exiting to main menu. Write flash parameter memory if a change has been made

	btsc.w	flags,#fUPDATE	;should parameter flash be udpated?
	rcall	WriteParams		;yes, update parameter flash
	bclr.w	flags,#fUPDATE	;clear the flag
	bra	mainMenu		;return to main menu

; haveBaud - new baud rate is in w1.

haveBaud:	mov.w	w1,ptBaudRate	;save new baud rate
	bset.w	flags,#fUPDATE	;request param flash update on exit
	bra	baudRate		;redisplay baud rate menu

;---------------------------------------------------------------------------
;  dispVers - display module version information
;---------------------------------------------------------------------------
dispVers:	mov.w	#tbloffset(sVersions),w0
	rcall	DispString

; Loop through and display version data for each module in the module table

	mov.w	#tbloffset(ModuleTable),w2    ;w2=module table address
verLoop:	tblrdl.w	[w2++],w0		;w0->module name string
	cp0.w	w0		;end of table?
	bra	z,verWait		;yes, prompt and wait for user
	rcall	DispString		;display module name
	tblrdl.w	[w2++],w0		;w0->module version string
	call	DispString
	tblrdl.w	[w2++],w0		;w0->module date string
	rcall	DispString
	mov.w	#tbloffset(sCrLf),w0	;trailing CR/LF
	rcall	DispString
	bra	verLoop

; verWait - Prompt for user to press return

verWait:	mov.w	#tbloffset(sContinue),w0
	rcall	DispString
	mov.w	#inbuf,w0
	call	inputString
	bra	mainMenu	

;---------------------------------------------------------------------------
;  restoreDef - restore default parameters and save after prompting
;     user for confirmation
;---------------------------------------------------------------------------
restoreDef: mov.w	#tbloffset(sRestore),w0	  ;display confirmation message
	rcall	DispString
	mov.w	#inbuf,w0		;read confirmation response
	rcall	inputString

	mov.b	inbuf,WREG		;w0=response
	ior.b	#0x20,w0		;convert 'Y' to 'y'
	sub.b	#'y',w0		;compare to 'y'
	bra	nz,mainMenu		;don't do restore

; restore defaults

	rcall	ResetParams		;reset to default in RAM
	rcall	WriteParams		;write to EEPROM

 	mov.w	#tbloffset(sResDone),w0   ;restore "done" message
	rcall	DispString
	bra	mainMenu

;---------------------------------------------------------------------------
;  updateFirm - update firmware by calling the update firmware routine
;     in the bootloader module
;---------------------------------------------------------------------------
updateFirm: call	FirmUpdate		;in the bootloader module
	bra	mainMenu

;---------------------------------------------------------------------------
;  dispDrive - display drive type string
;---------------------------------------------------------------------------
dispDrive:	mov.w	#tbloffset(drv2Name),w1	     ;w1->drv2Name table
	sl.w	DriveType,WREG	     ;w0=DriveType * 2	
	add.w	w0,w1,w0		     ;w0->drv2Name[DriveType]
	tblrdl.w	[w0],w0		     ;w0->drive type string
	rcall	DispString		     ;display the drive type string
	return

;---------------------------------------------------------------------------
;  DispString - display a string from PROGRAM MEMORY pointed to by w0.
;      Clobbers w0, w1
;---------------------------------------------------------------------------
	.global	DispString
DispString: tblrdl.b	[w0++],w1		;get the next byte of the message
	cp0.b	w1		;zero is end of string - exit
	bra	nz,1f		
	return
1:	btst.w	U1STA,#UTXBF	;loop until room for a character
	bra	nz,1b
	mov.w	w1,U1TXREG		;transmit the character
	bra	DispString

;---------------------------------------------------------------------------
;  dispRString - display a string from RAM pointed to by w0.
;     Clobbers w0, w1
;---------------------------------------------------------------------------
dispRString: cp0.b	[w0]		;zero is end of string - exit
	bra	nz,1f		
	return
1:	btst.w	U1STA,#UTXBF	;loop until room for a character
	bra	nz,1b
	mov.b	[w0++],w1		;retrieve and transmit the character
	mov.w	w1,U1TXREG
	bra	dispRString

;---------------------------------------------------------------------------
;  inputString - input a string to the buffer pointed to by w0. Handles
;     backspace. Maximum of MAX_IN_LEN characters read. String is null terminated.
;     Clobbers w1-w3
;---------------------------------------------------------------------------
inputString: push.w	w0		;save the start of buffer address
	clr.w	w2		;received (valid) character count
	bclr.w	U1STA,#OERR		;clear possible overrun error
waitChar:	btst.w	U1STA,#URXDA	;wait for a character
	bra	z,waitChar

; Process the character just typed in

	mov.w	U1RXREG,w1		;get the character
	and.w	#0x7f,w1		
	cp.b	w1,#CR		;C/R?
	bra	z,inputDone
	cp.b	w1,#BS		;back space?
	bra	z,backspace
	mov.w	w1,w3		;save the characgter
	sub.b	#' ',w1		;make sure it's a printable character
	bra	ltu,waitChar	;not, ignore it
	cp.w	w2,#MAX_IN_LEN	;room for more?
	bra	geu,waitChar	;no, ignore it
	inc.w	w2,w2		;otherwise, count the character
	mov.w	w3,U1TXREG		;echo the character	
	mov.b	w3,[w0++]		;and store the character
	bra	waitChar

; Process a backspace

backspace:	cp0.w	w2		;already at zero characters?
	bra	z,waitChar		;yes, do nothing
	dec.w	w2,w2		;otherwise, decrement the count
	dec.w	w0,w0		;and the buffer pointer
	mov.w	w1,U1TXREG		;echo backspace, space, backspace to do a delete
	mov.w	#' ',w1
	mov.w	w1,U1TXREG
	mov.w	#BS,w1
	mov.w	w1,U1TXREG	
	bra	waitChar

; Process C/R (line done)

inputDone:	clr.b	[w0]		;store null terminator
	mov.w	#0x0d,w1		;echo CR and two LFs
	mov.w	w1,U1TXREG
	mov.w	#0x0a,w1	
	mov.w	w1,U1TXREG
	mov.w	w1,U1TXREG	
	pop.w	w0		;restore original buffer pointer
	return

	.end

