;***************************************************************************************
;
;  hdfloppy.s
;     This module uses a 96tpi high density (HD) floppy drive like the Teac 55-GFR
;     to provide 1.5Mb of formatted storage. The disk layout is substantially different
;     than the Altair formats. A single sector of 10240 bytes is used. Any I/O is
;     essentially a full track. The controller takes care of track number verification
;     and a 16 bit checksum verification. The 8080 software sees only the formatted
;     data (80 * 128 byte CP/M sector size). The read clear and write clear periods
;     have been doubled to 400us/800us from the original Altair timing to allow for
;     greater index alignment variance. This disk layout allows for a max 1.5% speed
;     variation.
;
;     On the IDC-34 connector, the pin numer used for the 4th drive select is not
;     standardized. The Teac uses pin 6. On the IDC-34 adapter board, jumper IDC-34
;     pin 6 to IDC-50 pin 32 to use the 4th drive.
;
;     This code also works for an 8" DSDD drive. The motor-on signal toggled in this
;     code is the head load line on the 8" drive. The track 43 low current signal
;     for 8" drives is provided.
;
;     For better reliability with aging media, the last 8 tracks on the top side
;     of the disk are not used (the top head is offset inward by 8 tracks versus the
;     bottom head). Track 76 on the bottom is the last track used for compatibility
;     with 8" drives. The track layout is a follows:
;
;        Cylinder    0   1   2   3       70  71  72  73  74  75  76  77  78  79
;	        --- --- --- --- ... --- --- --- --- --- --- --- --- --- ---
;        Track Top   1   3   5   7      141 143  -   -   -   -   -   -   -   -
;              Bot   0   2   4   6      140 142 144 145 146 147 148  -   -   -
;
;     125ns write precomp begins at cylinder 60 (52 top), 250ns precomp begins at
;     cylinder 77 (69 top). With the above disk layout, only cylinders 69-71 on the
;     top have 250ns precomp applied. This is optimized for 5.25" HD drives more than
;     for 8" DSDD drives.
;
;     The disk format and data interchange between the 8080 and the FDC+ is completely
;     different for the HD floppy drive option than for standard Altair drives. This 
;     means a different boot PROM is required to boot the loader found at the start of
;     track zero on an HD floppy drive. As a way to get around this requirement, the
;     main sector loop returns a standard Altair 137 byte sector (found at the label
;     bootSector) whenever the index pulse is reached (i.e., new sector) AND an HD
;     floppy read is not in progress. This means an Altair boot PROM will receive
;     the fabricated boot sector and jump to it as if it came from a standard Altair
;     disk. The code in the boot sector, in turn, contains the code to load track 
;     zero from an HD floppy drive to continue the boot process. 
;
;  Version History
;    1.0   06/19/20  M. Douglas
;    1.1   07/01/20  M. Douglas
;	         Add Write Protect status bit in the drive status register. Use 
;	         only tracks 0-71 on the top and 0-76 on the bottom for better
;	         reliability with old media and for compatibility with 8" drives
;	         (77 tracks).
;
;--------------------------------------------------------------------------------------
; 
;  PIC H/W Assignments
;
;     EPMP (PMP) handles port I/O from the 8080
;     UART2 generates the step signal
;     UART3 generates the step direction signal
;     SPI1 generates write data pulses
;     OC1 is the head step timer
;     OC2 is the head settle timer
;     OC4 is sector timer (sector true, read clear, write clear, etc.)
;     IC1 captures read data pulses
;     IC2 captures sector/index pulses 
;     CN68 captures changes in the PINTE line (interrupts enabled) from the 8080
;     TMR4 at 250khz is common sync source for OC4 and IC2, and clock source for OC1-OC3
;
;***************************************************************************************

	.include 	"common.inc" 

; Debug equates. Set to 1 to enable, 0 to disable.

	.equ	actionCodes,0	;displays codes for various actions in this code
	.equ	s3Debug,0		;debug signal output on switch 3 input for scope
	.equ	debugPrecomp,0	;use "P" to cycle through forced precomp settings
 
; The following working registers have permanent variable assignments and cannot
;   be used for any other purpose. 

	.equ	pSectorPos,w14	;pointer to PMDOUT1H
	.equ	pLATB,w13		;pointer to PORTB (LATB)
	.equ	pReadData,w12	;pointer to PMDOUT2L
	.equ	pWriteData,w11	;pointer to PMDIN2L
	.equ	wFFFF,w10		;contains 0xffff
	.equ	wPeriod2,w9		;maximum time for 2us pulse
	.equ	wPeriod3,w8		;maximum time for 3us pulse
	.equ	wPrevHole,w7	;input capture at previous index/sector hole
	.equ	wPrevData,w6	;input capture at previous data pulse
	.equ	wShiftReg,w5	;shift register for xmit
	.equ	wInsertPtr,w4	;buffer insert pointer
	.equ	wRemovePtr,w3	;buffer remove pointer
	.equ	wReadJump,w2	;jump vector into read bit routines

; Map PMDIN1/2 and PMDOUT1/2 to more logical names

	.equ	driveStatus,PMDOUT1L	;drive status IN 08
	.equ	selectCmd,PMDIN1L	;drive select OUT 08
	.equ	sectorPos,PMDOUT1H	;sector position register IN 09
	.equ	driveCmd,PMDIN1H	;drive command OUT 09
	.equ	readData,PMDOUT2L	;read data from disk IN 10
	.equ	writeData,PMDIN2L	;write data to disk OUT 10
	.equ	ioStatus,PMDOUT2H	;I/O status results IN 11
	.equ	trackNum,PMDIN2H	;track number from 8080, OUT 11
	.equ	commandRegs,PMDIN1	;select and drive command registers
	.equ	statusRegs,PMDOUT1	;drive status and sector position as word

; Parallel port (PMP) Status bits

	.equ	pmDRIVE_STATUS,0	;drive status register read
	.equ	pmSECTOR_POS,1	;sector position reigster read
	.equ	pmREAD_DATA,2	;read data register read
	.equ	pmSELECT_CMD,8	;drive select register written
	.equ	pmDRIVE_CMD,9	;drive command register written
	.equ	pmWRITE_DATA,10	;write data register written
	.equ	pmTRACK_NUM,11	;track number register written
	
; Drive Status Register Equates. These are all asserted true as zeros. Bit 4 has been
;    re-purposed to indicate completion of the transfer so the 8080 software does
;    not have to count bytes.

	.equ	dsENWD,0		;0=enter new write data flag (write)
	.equ	dsMOVE_HEAD,1	;0=OK to move head
	.equ	dsHEAD_STATUS,2	;0=head loaded and settled
	.equ	dsREADY,3		;0=drive ready (not official)
	.equ	dsWRITE_PROTECT,4	;0=write protected disk
	.equ	dsINTE,5		;0=8080 interrupts enabled
	.equ	dsTRACK0,6		;0=on track zero
	.equ	dsNRDA,7		;0=new read data available (read)
	.equ	dsINIT_VALUE,0xf5	;move head asserted, b3 cleared

; Sector Position Register Equates. Sector True asserts as zero.

	.equ	spSECTOR_TRUE,0	;bit 0 is sector true bit
	.equ	spSECTOR_TRUE_MASK,0xc0	;required bits with sector true asserted
	.equ	spSECTOR_INIT,0xc1	;sets required bits around sector number

; Select Command Register Equates

	.equ	scDRIVE_MASK,0x0f	;mask to get drive number bits alone
	.equ	scDESELECT,7	;bit number of deselect bit

; Drive Command Register Equates

	.equ	dcSTEP_IN,0		;step head towards center
	.equ	dcSTEP_OUT,1	;step head out towards edge
	.equ	dcHEAD_LOAD,2	;load head for read/write
	.equ	dcHEAD_UNLOAD,3	;unload head
	.equ	dcREAD_ENABLE,4	;initiate read sequence
	.equ	dcHEAD_CURR_SW,6	;low write current
	.equ	dcWRITE_ENABLE,7	;initiate write sequence
	.equ	dcHEAD_MASK,0x0f	;get all head step/load bits at once
	.equ	dcMOVE_MASK,0x03	;get both step in/out bits at once

; I/O Status Register Equates

	.equ	iosCHKSUM_ERR,0	;1=checksum error
	.equ	iosTRACK_ERR,1	;1=track number doesn't match
	.equ	iosEOS_ERR,2	;1=unexpected end of sector
	.equ	iosIO_COMPLETE,7	;1=physical disk I/O completed

; Drive interface equates. Outputs are all asserted low (not inverted from processor
;   to the drive). Inputs are asserted high (inverted from drive to processor).

	.equ	B0_DRIVE_SELECT1,0
	.equ	B1_DRIVE_SELECT2,1
	.equ	B2_DRIVE_SELECT3,2
	.equ	B3_DRIVE_SELECT4,3
	.equ	B6_WRITE_DATA,6
	.equ	B7_STEP_DIRECTION,7
	.equ	B8_IN_USE,8		;output, not used
	.equ	B9_WRITE_GATE,9
	.equ	B11_MOTOR_ON,11
	.equ	B12_SIDE_SELECT,12
	.equ	B13_WRITE_CURRENT,13
	.equ	F4_STEP,4	
	.equ	D10_READ_DATA,10
	.equ	D0_WRITE_PROTECT,0
	.equ	D1_READY,1
	.equ	D2_TRACK0,2
	.equ	D3_INDEX,3
	.equ	F1_TRUE_READY,1	;input, not used
	.equ	B_DESELECT,0x3bcf	;all drive signals off port B

; Head step and load equates. OC1 and OC2 are used to time head step and head load. The head
;    load time is an equate here. The step time is a user-settable parameter ptStepTHD. The
;    timebase used for OC1 and OC2 is 250KHz (4us per tick) and comes from Timer 4. UART2 and
;    UART3 are used to send step and direction pluses.

	.equ	HEAD_SETTLE_TIME,18000/4  ;head settle time, 18ms at 250khz
	.equ	STEP_CHAR,0xfe	;character when sent serially generates a 2us
				;pulse at 1Mb/s (start bit and bit 0 low)
	.equ	DIR_IN_CHAR,0xf0	;character when sent serially generates a 5us
				;pulse at 1Mb/s (start bit and four 0's)

; Sector/index equates. Sector/index pulses are captured by IC2. IC2 and OC4 are
;    sync'd to TMR4 which runs full-wrap at 250khz.

	.equ	INDEX_TIME,166668/4	;166.667ms full revolution time(68 is right)
	.equ	INDEX_MAX,169168/4	;101.5% of full revolution time
	.equ	INDEX_MIN,164164/4	;98.5% of full revolution time
	.equ	SEC_TRUE_TIME,32/4	;32us (extra 2us given)
	.equ	READ_CLEAR_TIME,400/4	;read data ignored for 1st 400us
	.equ	WRITE_CLEAR_TIME,800/4	;write data starts after 800us
	.equ	READ_CLEAR_DELTA,READ_CLEAR_TIME-SEC_TRUE_TIME
	.equ	WRITE_CLEAR_DELTA,WRITE_CLEAR_TIME-READ_CLEAR_TIME

; Read and write data equates. Read data pulses are captured by IC1. The timebase for
;    IC1 is 16Mhz. For FM encoding, the bit time is 4us. If no pulse occurs between
;    two 4us pulses, the data bit is a zero. If a pulse is present in between
;    two 4us pulses, the data bit is a one. Writing of data is done with SPI1 at 
;    500ns per SPI bit.

	.equ	XFER_LENGTH,10240	;10240 data bytes per sector (track)
	.equ	ZERO_BIT,0xe7ff	;SPI data for a zero MFM bit
	.equ	PERIOD2,40		;2.5us max for 2us period (16mhz)
	.equ	PERIOD3,56		;3.5us max for 3us period (16mhz)
	.equ	PRECOMP125_TRACK,60*2	;start 125ns precomp on cylinder 60 (52)
	.equ	PRECOMP250_TRACK,77*2	;start 250ns precomp on cylinder 77 (69)
	.equ	END_TOP_TRACK,72*2	;no more top tracks from cylinder 72

; flags variable

	.equ	fWRITE_ENABLE,0	;1=write enable command from 8080
	.equ	fREAD_ENABLE,1	;1=read enable command from 8080
	.equ	fLOW_WRITE_CURRENT,2	;1=assert low write current signal to drive
	.equ	fWAIT_MOTOR,3	;1=exit cmd interrupt to waitMotor
	.equ	fHEAD_LOADED,4	;1=head loaded due to 8080 command

; Misc equates

	.equ	NUM_DRIVES,4	;four drives supported. Fourth drive
				;requires jumper on adapter board from
				;IDC-50 p32 to IDC-34 p6
	.equ	PIC_INTS_OFF,0xe0	;write to SR to disable PIC interrupts

; Variables

	.section *,bss,near
flags:	.space	2		;see flag bits above
bitCount:	.space	2		;bit counter in byte
motorTimer:	.space	2		;counts sectors until 256 bit set (1.3s)
checkSum:	.space	2		;checksum of sector data
xferCount:	.space	2		;transfer counter
curTrack:	.space	2		;current track number from 8080
sector:	.space	2		;current sector number (0 or 15)

preCompTbl:	.space	2		;used for precomp debug only
preCompNum:	.space	2		;same


; trackBuf - This buffer is used to pass track data back and forth with the 8080.
;    The buffer is forced into higher memory so that all other .bss data can be
;    in the first 8K for direct addressing.

	.bss
bootImage:	.space	256		;holds an Altair boot sector
trackBuf:	.space	XFER_LENGTH+256	;track length plus some padding


;****************************************************************************************
;  
;   HdFloppy - entry point for 1560K drive
;
;****************************************************************************************
	.text
	.global	HdFloppy

HdFloppy:	mov.w	#STACK_ADDR,w15	;init stack pointer
	mov.w	#PIC_INTS_OFF,w0	;disable all interrupts
	mov.w	w0,SR
;
; Initialize the parallel port to handle the I/O IN and OUT instructions from the 8080. The
;     PMP is run in addressable slave mode where I/O addresses from the 8080 map as shown
;     below.
;
;     I/O address            PIC register
;	 8	PMDOUT1, PMDIN1 low byte
;	 9	PMDOUT1, PMDIN1 high byte
;	10	PMDOUT2, PMDIN2 low byte
;	11	PMDOUT2, PMDIN2 high byte

	mov.w	#0x0281,w0		;enhanced PSP mode, CS1 on PMA14, interrupts on
	mov.w	w0,PMCON1
	mov.w	#0xc000,w0		;PMWR, PMRD enabled
	mov.w	w0,PMCON3
	mov.w	#0x4003,w0		;CS1(A14), A1, A0 enabled
	mov.w	w0,PMCON4
	mov.w	#0x6000,w0		;CS1 on and active high, RD, WR active low
	mov.w	w0,PMCS1CF
	mov.w	#0x0001,w0		;use TTL buffers
	mov.w	w0,PADCFG1
	mov.w	#0xffff,w0		;set both 8080 status registers to 0xff
	mov.w	w0,statusRegs
	bset.w	PMCON1,#15		;enable EPMP
 
; Map UART2 transmit to the step line and UART3 to the step direction line. Receive
;    data is not used, but if left mapped to the default RP63, receive data is low
;    which looks like a continuous break. It is mapped here to RP18 (Port B, bit 5,
;    ICSP Clock) which initialization sets as an input with internal pullup. This pin
;    remains unused and high in normal operation.

	mov.b	#18,w0		;UART Receive data tied high on RP18
	mov.b	WREG,RPINR19L	;UART 2
	mov.b	WREG,RPINR17H	;UART 3

	mov.b	#5,w0		;UART 2 TX code 5 is on RP10
	mov.b	WREG,RPOR5L
	mov.b	#28,w0		;UART 3 TX code 28 is on RP7
	mov.b	WREG,RPOR3H

; Initialize UART2 and UART3 for 8N1 at 1M baud (1us per bit)

	mov.w	#0x0008,w0		;serial port off, 8N1, BRGH=1
	mov.w	w0,U2MODE
	mov.w	w0,U3MODE
	mov.w	#3,w0		;1M baud
	mov.w	w0,U2BRG
	mov.w	w0,U3BRG
	bset.w	U2MODE,#UARTEN	;UART on
	bset.w	U2STA,#UTXEN	;transmitter on too
	bset.w	U3MODE,#UARTEN	;UART on
	bset.w	U3STA,#UTXEN	;transmitter on too

; Init TMR4 to use a 250khz timebase. This timer is the sync source for the
;   sector/index pulse capture (IC2) and the sector timer (OC4). TMR4 is
;   also the clock source (not sync source) for OC1-OC3. The timer is enabled
;   later after IC2 and OC4 are initialized.

	mov.w	#0x0020,w0		;Fcy/64 for 250khz
	mov.w	w0,T4CON
	mov.w	#0xffff,w0
	mov.w	w0,PR4

; Configure OC1 to time head step delays, OC2 to time the head load delay.
;   TMR4 at 250khz is used as clock source. 

	mov.w	#0x0020,w0		;free running, no output pin
	mov.w	w0,OC1CON2
	mov.w	w0,OC2CON2
	mov.w	#0x0803,w0		;TMR4 and one-shot toggle mode
	mov.w	w0,OC1CON1
	mov.w	w0,OC2CON1

;  Map IC2 to the index line. Configure to capture on rising edge (falling edge
;    from the drive is inverted). IC2 is sync'd to TMR4 running at 250khz.

	mov.b	#22,w0		;IC2 detects index pulses on RP22
	mov.b	WREG,RPINR7H
	mov.w	#0x000e,w0		;IC2 sync'd to the OC4 timer
	mov.w	w0,IC2CON2
	mov.w	#0x0803,w0		;rising edge capture, TMR4 is clk source
	mov.w	w0,IC2CON1	

; Configure OC4 as the sector timer to time sector true, read clear, write clear, etc.
;    Both OC4 and IC2 are sync'd to TMR4.

	mov.w	#0x002e,w0		;no external sync, OC pin output not used
	mov.w	w0,OC4CON2
	mov.w	#0x0803,w0		;TMR4 clk source, continuous compare
	mov.w	w0,OC4CON1		;turn on OC4
	bset.w	T4CON,#TON		;turn on TMR4 now that IC2 and OC4 ready

;  Map IC1 to the read data line. Configure to capture on rising edge using the system clock.

	mov.b	#3,w0		;IC1 detects read data pulses on RP3
	mov.b	WREG,RPINR7L
	clr.w	IC1CON2		;IC1 not sync'd to any other source
	mov.w	#0x1c03,w0		;IC1 capture on rising edge, Fcy clock
	mov.w	w0,IC1CON1	
	
;  SPI1 is used to generate the write data waveform. Initialize for 16 bits at 2Mhz.
;     This will send two bits of drive data per 16 bits of SPI data. The SPI is set
;     to interrupt when the FIFO is empty (16 bit word still in shift register).

	mov.b	#7,w0		;SPI1 Data out is on RP6
	mov.b	WREG,RPOR3L		;assign SPI1 data out to disk write data

	mov.w	#0x0001,w0		;select enhanced mode (8 word buffer)
	mov.w	w0,SPI1CON2
	mov.w	#0x143b,w0		;no clk or data out, 16 bit, master, 8MHz
	mov.w	w0,SPI1CON1
	mov.w	#0x0010,w0		;interrupt when fifo has a spot
	mov.w	w0,SPI1STAT
	bset.w	SPI1STAT,#SPIEN	;enable SPI
	mov.w	#0xffff,w0		;write all 1's so line is idling high
	mov.w	w0,SPI1BUF

; Set the PINTE signal from the 8080 bus to be an input change interrupts.

	bset.w	CNEN5,#CN68IE	;PINTE signal is CN68

; Put our local ISRs in the interrupt table

	mov.w	#tbloffset(ioPortInt),w0       ;PMP interrupt (8080 I/O)
	mov.w	w0,PMPInterrupt
	mov.w	#tbloffset(headMoveInt),w0     ;OC1 interrupt (head move timer)
	mov.w	w0,OC1Interrupt
	mov.w	#tbloffset(headStatusInt),w0   ;OC2 interrupt (head status timer)
	mov.w	w0,OC2Interrupt
	mov.w	#tbloffset(changeInt),w0       ;CN interrupt (PINTE)
	mov.w	w0,CNInterrupt

; Copy Altair compatible boot sector from flash into RAM in bootImage

	mov.w	#tbloffset(bootSector),w1      ;w1->sector data in flash
	mov.w	#bootImage,w2	       ;w2->where to put it in RAM
	mov.w	#BOOT_SECTOR_LENGTH/2,w3       ;w3=loop counter
	
1:	tblrdl.w	[w1],w0		;w0=next two bytes
	add.w	#2,w1		;increment source pointer
	mov.w	w0,[w2++]		;store in RAM
	dec.w	w3,w3		;loop until moved
	bra	nz,1b

; Initialize permanent pointers and values present in some registers

	mov.w	#sectorPos,pSectorPos	;permanent pointers
	mov.w	#readData,pReadData
	mov.w	#writeData,pWriteData
	mov.w	#LATB,pLATB

	mov.w	#0xffff,wFFFF	;permanent values
	mov.w	#PERIOD2,wPeriod2
	mov.w	#PERIOD3,wPeriod3
	mov.w	#trackBuf,wInsertPtr
	mov.w	#trackBuf,wRemovePtr

	mov.w	#tbloffset(preComp000),w0  ;default no precomp (for debug only)
	mov.w	w0,preCompTbl
	clr.w	preCompNum

; S3 debug initialization

    .if (s3Debug)  			;use switch 3 input as an output signal
	mov.w	#0x0000,w0		;port c init
	mov.w	w0,LATC
	mov.w	#0xb000,w0		;port c direction - bit 14 as output
	mov.w	w0,TRISC		;MAKE SURE DRIVE TYPE BIT 3 (SWITCH 4) IS OFF
    .endif

; Sample use of s3 as output for scope

    .if s3Debug
	bclr.w	LATC,#C14_DRV_TYPE3_SW
    .endif
	
; Enable interrupts and run

	mov.w	PMDIN1,w0		;ensure EPMP is completely reset
	mov.w	PMDIN2,w0
	mov.w	wFFFF,PMDOUT1	;driveStatus and sectorPos to 0xff
	clr.w	PMDOUT2		;data and IO stats zero
	bclr.w	PMSTAT,#IBOV	;make sure errors are cleared
	bclr.w	PMSTAT,#OBUF
	
	bset.w	IEC2,#PMPIE		;enable parallel port interrupt
	bset.w	IEC1,#CNIE		;enable ready and PINTE line interrupts
				;fall into idle loop

;**********************************************************************************
;
;  Idle loop - do nothing while waiting for the command processing interrupt
;     to select a drive and change the execution point to waitMotor.
;
;**********************************************************************************
enterIdle:	disi	#5		;disable interrupts for next 4 cycles
	mov.w	wFFFF,statusRegs	;driveStatus and sectorPos to 0xff
	clr.b	ioStatus		;clear I/O status flags	
	mov.w	#B_DESELECT,w0	;disable all drive signals
	mov.w	w0,LATB
	clr.w	flags		;set all flags to false

	mov.w	#spSECTOR_TRUE_MASK,w0	;init sector register value
	mov.w	w0,sector		;sector in bits 5-1 is zero

	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	clr.w	SR		;enable interrupts
	
; Idle loop - allow escape to monitor with 'X' or 'x' if monitor switch is on

idleLoop:	btst.w	U1STA,#URXDA	;test for a character
	bra	z,idleLoop
	
	mov.w	U1RXREG,w0		;get the character
	btst.w	PORTC,#C13_MONITOR_SW	;monitor enabled?
	bra	z,idleLoop		;no
	
	and.w	#0x5f,w0		;convert lower case to upper

  .if debugPrecomp

	sub.b	#'P',w0		;precomp change?
	bra	nz,idleLoop		;no

	mov.w	preCompNum,w0	;on 0 (no precomp)?
	cp.w	w0,#0
	bra	nz,1f		;no

	mov.w	#tbloffset(preComp125),w1  ;move to 1 (precomp125)
	mov.w	w1,preCompTbl
	inc.w	preCompNum		;now 1
	mov.b	#'1',w1		;precomp 1
	mov.w	w1,U1TXREG		;transmit the character
	bra	idleLoop
	
1:	cp.w	w0,#1		;on 1 (precomp 125)
	bra	nz,1f		;no

	mov.w	#tbloffset(preComp250),w1  ;move to 2 (precomp250)
	mov.w	w1,preCompTbl
	inc.w	preCompNum		;now 2
	mov.b	#'2',w1		;precomp 2
	mov.w	w1,U1TXREG		;transmit the character
	bra	idleLoop

1:	cp.w	w0,#2		;on 2 (precomp 250)?
	bra	nz,1f		;no

	mov.w	#tbloffset(preComp375),w1  ;move to 3 (precomp375)
	mov.w	w1,preCompTbl
	inc.w	preCompNum		;now 3
	mov.b	#'3',w1		;precomp 3
	mov.w	w1,U1TXREG		;transmit the character
	bra	idleLoop

1:	mov.w	#tbloffset(preComp000),w1  ;move to 0 (no precomp)
	mov.w	w1,preCompTbl
	clr.w	preCompNum		;now 0
	mov.b	#'0',w1		;precomp 1
	mov.w	w1,U1TXREG		;transmit the character
	bra	idleLoop
  .endif
	
	sub.b	#'X',w0		;exit requested?
	bra	nz,idleLoop		;no
	
	reset			;go back to boot loader

;**********************************************************************************
; Sector Processing Loop
;   1) Wait for next index hole, verify rotation speed is good.
;
;   2) Assert sector true (if sector position register enabled) and start
;      the 30us sector true period.
;
;   3) Start the "read clear" timer. Look for and clear data pulse input
;      captures. Monitor write enable flag in case a write is requested.
;
;  4a) After read clear expires and if a read enable request was received,
;      begin shifting data bits into a receive byte. Once the MSB has a
;      one in it (sync), begin assembling and moving bytes to the track
;      buffer. Once four bytes have been queued, assert NRDA to the 8080
;      so it can start retrieving the bytes.
;
;  4b) If write data is asserted during the read clear period, assert write
;      enable to the drive and immediately begin writing zeros to the drive.
;      ENWD is asserted to the 8080 so it will begin sending the data to
;      write which is queued in the track buffer. Finally, start the "write
;      clear timer. When write clear expires, data is pulled from the track
;      buffer and written to the disk.
;
;   5) When XFER_LENGTH bytes have been read/written, send or compare the
;      checksum word and update the I/O status register with the result.
;      Then wait for the index pulse to start all over at step 1.
;
;-----------------------------------------------------------------------------

; waitMotor - Entered when motor started to properly enter the sector
;    processing loop.

waitMotor:	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	clr.w	SR		;set interrupt level zero

clearIC2:	mov.w	IC2BUF,w0		;clean out buffered index captures
	btst.w	IC2CON1,#ICBNE	;empty yet?
	bra	nz,clearIC2		;no

	rcall	nextHole		;get a first pulse

;-----------------------------------------------------------------------------
; sectorLoop - Sector processing loop
;-----------------------------------------------------------------------------
sectorLoop:	rcall	nextHole		;wait for next index pulse

; If an HD read command has not been issued, then point the remove pointer to the
;    boot sector image so that a standard Altair boot loader will get a valid
;    boot sector.

	btst.w	flags,#fREAD_ENABLE	;was an HD read in progress?
	bclr.w	flags,#fREAD_ENABLE	;clear flag now in case it was
	bra	nz,1f		;HD read still finishing
	mov.w	#bootImage,wRemovePtr	;point to the boot sector image
	mov.b	[wRemovePtr++],w0	;init data register with 1st byte
	mov.b	WREG,readData
	bclr.b	driveStatus,#dsNRDA	;ensure NRDA is asserted

; If head is no longer loaded, increment the motor timeout. When the 32's bit is
;    set (32 * 166.67ms/rev) = 5.33s, turn off the motor.

1:	disi	#5		;protect for 5 cycles
	btsc.w	flags,#fHEAD_LOADED	;check if head loaded
	clr.w	motorTimer		;hold timer in reset if head loaded
	inc.w	motorTimer	
	btsc.w	motorTimer,#5	;test the 32's bit
	bset.w	LATB,#B11_MOTOR_ON	;turn off the motor if 32 reached	

; Verify rotation time is within +/-1.5%

	mov.w	#INDEX_MAX,w0	;w0=maximum time for full revolution
	cp.w	w1,w0		;w1 has rotation time
	bra	gtu,sectorLoop	;slower than maximum time

	mov.w	#INDEX_MIN,w0	;w0=minimum time for full revolution
	cp.w	w1,w0
	bra	ltu,sectorLoop	;faster than minimum time

; Enable and update the sector register if the head is loaded and settled. The sector
;    number toggles between 0 and 15. Normal HD floppy code does not care about the
;    sector number, but the original Altair boot pROM looks for sector zero and the
;    Combined Disk Boot Load (CDBL) looks for sectors 0 and 15.

	mov.w	#0x01e,w0		;toggle sector between 0 and 15
	xor.w	sector
	mov.w	sector,w0		;w0=sector register
	disi	#2		;protect next 2 instructions
	btss.b	driveStatus,#dsHEAD_STATUS   ;head loaded and settled?
	mov.b	WREG,sectorPos	;yes, update sector position register

; Start a 30us timeout of the sector true period using OC4 and then
;    wait for the sector true period to expire.
	
	mov.w	OC4TMR,w0		;w0=sector start time (now)
	add.w	#SEC_TRUE_TIME,w0	;w0=count when sector true ends
	mov.w	w0,OC4R		;set that as the next compare
	bclr.w	IFS1,#OC4IF		;make sure flag is false
	
wtSecFalse:	btst.w	IFS1,#OC4IF		;sector true period done?
	bra	z,wtSecFalse	;no

; Sector true period expired. De-assert the sector true signal in the sector
;   position register, then start the read-clear period. 

	bset.b	sectorPos,#spSECTOR_TRUE   ;de-assert sector true flag
	mov.w	#READ_CLEAR_DELTA,w0	;time until read clear expires
	add.w	OC4R		;set new compare time
	bclr.w	IFS1,#OC4IF		;make sure flag is false

; wtReadClr - waiting for read clear period to end. Pull off data pulse 
;    capture events and update the previous pulse value. While waiting, look
;    for the write enable flag in case a write is requested.

wtReadClr:	btst.w	IFS1,#OC4IF		;read clear timeout done?
	bra	nz,initRead		;yes, enter the read loop
	btst.w	flags,#fWRITE_ENABLE	;write enable requested?
	bra	nz,initWrite	;yes, enter the write loop
	btst.w	IC1CON1,#ICBNE	;data pulse capture occur?	
	bra	z,wtReadClr		;no, keep waiting	
	mov.w	IC1BUF,wPrevData	;pull off the capture time
	bra	wtReadClr

;-------------------------------------------------------------------------------------	
; Read Data Loop - Look for data pulses from the drive captured by IC1. Once a  
;   pulse is received, decode and shift bits into shiftReg and store in the
;   track buffer.
;-------------------------------------------------------------------------------------	
initRead:	btst.w	flags,#fREAD_ENABLE	;read requested?
	bra	z,sectorLoop	;no, go wait for next sector

	disi	#2		;protect next two cycles
	mov.w	#trackBuf,wInsertPtr	;init buffer pointers
	mov.w	wInsertPtr,wRemovePtr
	mov.w	#XFER_LENGTH-1,w0	;init transfer length
	mov.w	w0,xferCount
	clr.b	ioStatus		;clear I/O status flags to 8080
	clr.w	checkSum		;init checksum
	clr.w	wShiftReg		;zero the data shift register
	clr.w	bitCount		;zero the bit counter
	btsc.w	IC1CON1,#ICBNE	;make sure we have latest data pulse
	mov.w	IC1BUF,wPrevData	;get most recent pulse (one behind at most)
	mov.w	#tbloffset(prevWasClk),wReadJump  ;read bit entry point

; waitSync - Wait for a byte with the sync bit (MSBit) set. Once found, this byte
;     also contains the cylinder number in the lower 7 bits. Verify it is correct.

waitSync:	call	wReadJump		;update shiftReg with next bit
	btst.z	wShiftReg,#7	;msb one (sync)?
	bra	z,waitSync		;no, keep looking

	mov.b	curTrack,WREG	;W0=track number expected
	sub.b	w0,wShiftReg,w0	;subtract and convert to cylinder
	and.b	#0x7f,w0
	bra	z,readLoop		;match, no error
	bset.b	ioStatus,#iosTRACK_ERR	;set error flag for the 8080					;fall into read loop

; readLoop - Read and transfer the sector until we reach the checksum bytes

readLoop:	bset.w	bitCount,#3		;reset bit count to 8

bitLoop:	call	wReadJump		;update shiftReg with next bit
	dec.w	bitCount		;count 8 bits of the byte
	bra	nz,bitLoop		;loop until byte assembled

	mov.w	wShiftReg,w0	;w0 = byte just formed
	mov.b	w0,[wInsertPtr++]	;save new byte in track buffer
	and.w	#0xff,w0		;clear MSByte
	add.w	checkSum		;add byte to the checksum
	btss.w	xferCount,#2	;done at least 4 bytes yet?
	bclr.b	driveStatus,#dsNRDA	;assert NRDA so 8080 will start reading
	dec.w	xferCount		;decrement bytes remaining
	bra	geu,readLoop

; doCheckSum - Read the two checksum bytes and compare to computed value

	bset.w	bitCount,#3		;set bitCount to 8
doCheckSum:	call	wReadJump		;update shiftReg with next bit
	dec.w	bitCount		;count 8 bits of the byte
	bra	nz,doCheckSum	;loop until complete
	mov.b	wShiftReg,[wInsertPtr]	;save LSB of checksum from disk

	bset.w	bitCount,#3		;set bitCount to 8
checkSum2:	call	wReadJump		;update shiftReg with next bit
	dec.w	bitCount		;count 8 bits of the byte
	bra	nz,checkSum2	;loop until complete
	
	bset.b	ioStatus,#iosIO_COMPLETE  ;disk I/O is complete
	sl.w	wShiftReg,#8,w0	;w0=checksum MSB in MSB
	mov.b	[wInsertPtr],w0	;w0=16 bit checksum from disk
	cp.w	checkSum		;match computed checksum?
	bra	z,sectorLoop	;yes, no error, all done
	bset.b	ioStatus,#iosCHKSUM_ERR	;else, set error
	bra	sectorLoop		;all done

;-------------------------------------------------------------------------------------	
; readBit - Decode pulses to read the next data bit and insert into shiftReg.
;     Look for an index/sector pulse while waiting. This aborts back to start
;     a new sector/track.
;-------------------------------------------------------------------------------------	

; Previous pulse was data. Data 1 path is 15 cycles including return

prevWasData: btst.w	IC2CON1,#ICBNE	;hit the next sector/index pulse?	
	bra	nz,readAbort	;yes, abort this sector
	btst.w	IC1CON1,#ICBNE	;data pulse capture occur?	
	bra	z,prevWasData	;no, keep waiting

	mov.w	IC1BUF,w1		;pull off the capture time
	sub.w	w1,wPrevData,w0	;w0=time between data pulses
	mov.w	w1,wPrevData	;save new previous count

	cp.w	w0,wPeriod2		;2us between pulses?
	bra	gtu,check3		;no, check for 3us
				;fall into data 1 (saves 1 cycle)
data1:	sl.w	wShiftReg,wShiftReg	;shift in a zero bit
	bset.w	wShiftReg,#0	;set new bit to one
	mov.w	#tbloffset(prevWasData),wReadJump
	return

check3:	cp.w	w0,wPeriod3		;3us between pulses?
	bra	gtu,data01		;no, must be 4us, so data is 01
				;3us, data is 00
data00:	sl.w	wShiftReg,wShiftReg	;shift in a zero bit
	mov.w	#tbloffset(data0),wReadJump
	return

data01:	sl.w	wShiftReg,wShiftReg	;shift in a zero bit
	mov.w	#tbloffset(data1),wReadJump
	return

; Previous pulse was clock. Data 0 path is 14 cycles including return

prevWasClk:	btst.w	IC2CON1,#ICBNE	;hit the next sector/index pulse?	
	bra	nz,readAbort	;yes, abort this sector
	btst.w	IC1CON1,#ICBNE	;data pulse capture occur?	
	bra	z,prevWasClk	;no, keep waiting

	mov.w	IC1BUF,w1		;pull off the capture time
	sub.w	w1,wPrevData,w0	;w0=time between data pulses
	mov.w	w1,wPrevData	;save new previous count

	cp.w	w0,wPeriod2		;2us between pulses?
	bra	gtu,data1		;no, must be 3us, so data is 1
				;2us so data is zero
data0:	sl.w	wShiftReg,wShiftReg	;shift in a zero bit
	mov.w	#tbloffset(prevWasClk),wReadJump
	return

; readAbort - A new sector/index pulse was received before expected. Set error
;    flags for the 8080 and also assert NRDA in case the 8080 is still stuck
;    in a loop waiting for it. Return to start of sector loop

readAbort:	mov.w	IC2BUF,w0		;w0=counter at new pulse
	mov.w	w0,wPrevHole	;save new previous IC value
	bset.b	ioStatus,#iosEOS_ERR	;unexpected end of sector
	bset.b	ioStatus,#iosIO_COMPLETE  ;disk I/O is complete
	bclr.b	driveStatus,#dsNRDA	;make sure 8080 not stuck in loop
	bra	sectorLoop		;all done with sector/track

;-----------------------------------------------------------------------------
; Write Data Loop - A write enable command has been received. Assert ENWD
;     to the 8080 so it begins the transfer of write data to the track
;     buffer. Start the write-clear time period during which a stream of 
;     zeros is written to the disk.
;
;     Data is written using MFM. A four bit shift register in bits 10-7
;     of wShiftReg is used as an index into MFM lookup tables to obtain
;     the pulse pattern to write to disk. Bits 11, 10 contain previously
;     written bits, bit 9 is the bit be written, and bit 8 is the bit
;     to be written next time.
;-----------------------------------------------------------------------------
; Start the write clear period in OC4. Assert write enable to the disk
;    and writing zeros until the write clear period expires.

initWrite:	bclr.w	LATB,#B9_WRITE_GATE	;assert write gate to the drive
	btsc.w	flags,#fLOW_WRITE_CURRENT   ;low write current requested?
	bclr.w	LATB,#B13_WRITE_CURRENT	    ;yes, assert low current signal
	mov.w	#ZERO_BIT,w0	;w0=one bit of zero
	mov.w	w0,SPI1BUF		;send first two zero bits (4us worth)
	mov.w	w0,SPI1BUF
	
	disi	#2		;protect next two cycles
	mov.w	#trackBuf,wInsertPtr	;init insert and remove pointers
	mov.w	wInsertPtr,wRemovePtr
	bclr.b	driveStatus,#dsENWD	;tell 8080 to send us data
	mov.w	#XFER_LENGTH-1,w0	;init transfer length
	mov.w	w0,xferCount
	clr.b	ioStatus		;clear I/O status flags to 8080
	clr.w	checkSum		;init checksum
	clr.w	wShiftReg		;zero the data shift register
	clr.w	bitCount		;zero the bit counter
	mov.w	#WRITE_CLEAR_DELTA,w0	;w0=time until write clear expires
	add.w	OC4R		;set new output compare time
	bclr.w	IFS1,#OC4IF		;make sure OC flag is false

  .if debugPrecomp
	mov.w	preCompTbl,w1	;w1 has precomp table
  .else
	mov.w	#tbloffset(preComp000),w1  ;assume no precomp needed
	mov.b	#PRECOMP125_TRACK,w0	;at or beyond 125ns precomp cylinder?
	btss.w	LATB,#B12_SIDE_SELECT	;if top head, precomp is 8 cylinders sooner
	sub.b	w0,#2*8 
	cp.b	curTrack		;curTrack-w0
	bra	ltu,2f		;no precomp needed
	mov.w	#tbloffset(preComp125),w1  ;else assume 125ns precomp table

1:	mov.b	#PRECOMP250_TRACK,w0	;at or beyond 250ns precomp cylinder?
	btss.w	LATB,#B12_SIDE_SELECT	;if top head, precomp is 8 cylinders sooner
	sub.b	w0,#2*8 
	cp.b	curTrack		;curTrack-w0
	bra	ltu,2f		;leave at 125ns precomp
	mov.w	#tbloffset(preComp250),w1  ;else use 250ns precomp table

  .endif

2:	mov.w	#ZERO_BIT,w0	;w0=one bit of zero

wtWriteClr:	btst.w	IFS1,#OC4IF		;write clear timeout done?
	bra	nz,doWrite		;yes
	btss.w	SPI1STAT,#SPITBF	;room in fifo for another bit?
	mov.w	w0,SPI1BUF		;yes, send another zero bit
	bra	wtWriteClr

; doWrite - The write clear period is done. Send the sync byte with MSBit set and
;    the cylinder number in the lower seven bits. Then loop sending all bytes of
;    the sector/track. Follow this with the two bytes of the checksum.

doWrite:	mov.b	curTrack,WREG	;w0=track number from 8080
	bset.w	w0,#7		;set MSBit as sync
	call	writeW0		;write the sync byte

wrtLoop:	call	wrtNext		;write next byte from trackBuf
	dec.w	xferCount		;do full sector/track
	bra	geu,wrtLoop

	mov.b	checkSum,WREG	;LSB of checksum
	call	writeW0		;write LSB of checksum
	mov.b	checkSum+1,WREG	;MSB of checksum
	call	writeW0		;write MSB of checksum

; All data has been written. Write a byte of trailing zeros, then after that
;    byte is completely written, turn off write enable.
	
	clr.w	w0		;write two zero bytes
	call	writeW0
	clr.w	w0
	call	writeW0

1:	btst.w	SPI1STAT,#SRMPT	;wait for shift register to empty
	bra	z,1b

	bclr.w	flags,#fWRITE_ENABLE	;write command has completed
	bset.w	LATB,#B9_WRITE_GATE	;clear write enable
	bset.w	LATB,#B13_WRITE_CURRENT	;clear possible low write current
	bset.b	ioStatus,#iosIO_COMPLETE  ;disk I/O is complete
	bra	sectorLoop		;all done with sector/track

;-----------------------------------------------------------------------------
; wrtNext - Write the next byte from the track buffer to disk and add it
;    to the 16 bit checksum
;
; writeW0 - Write the value from w0 and do not add the byte to the checksum
;
; w1 assumed to contain pointer to the proper MFM encoding table below.
;-----------------------------------------------------------------------------
wrtNext:	mov.b	[wRemovePtr++],w0	;w0=next byte to write
	and.w	#0xff,w0		;make sure MSB is zero
	add.w	checkSum		;16 bit add to checksum

writeW0:	mov.b	w0,wShiftReg	;put new byte in LSB of shiftreg
	bset.w	bitCount,#3		;set bit counter to 8

writeBits:	sl.w	wShiftReg,wShiftReg	;shift transmit data for next bit
	lsr.w	wShiftReg,#6,w0	;using bits 10-7 as table index
	and.w	#0x01e,w0		;4 bits value x 2 (word address)
	add.w	w0,w1,w0		;w0->MFM bit pattern for current bit
	tblrdl.w	[w0],w0		;w0=MFM bit pattern

1:	btst.w	SPI1STAT,#SPITBF	;wait for room in fifo
	bra	nz,1b

	mov.w	w0,SPI1BUF		;send the bit
	dec.w	bitCount
	bra	nz,writeBits
	return	

;-----------------------------------------------------------------------------
; MFM encoding tables
;    These tables are indexed by a four bit value of transmit data bits
;    in a shift register in which the 8's bit is two bits old, the 2's
;    bit is the value about to be written, and the 1's bit is the future
;    bit. The value retrieved is the bit pattern to write to the SPI to
;    send the proper pulses to the floppy write data line. Each SPI bit
;    is 125ns, so 16 SPI bits encode the 2us MFM cell time. One table
;    includes write pre-compensation for higher tracks, the other table
;    has no pre-comp.   
;-----------------------------------------------------------------------------
preComp000:	.word	0xe7ff	;0000
	.word	0xe7ff	;0001
	.word	0xffe7	;0010
	.word	0xffe7	;0011
	.word	0xffff	;0100
	.word	0xffff	;0101
	.word	0xffe7	;0110
	.word	0xffe7	;0111
	.word	0xe7ff	;1000
	.word	0xe7ff	;1001
	.word	0xffe7	;1010
	.word	0xffe7	;1011
	.word	0xffff	;1100
	.word	0xffff	;1101
	.word	0xffe7	;1110
	.word	0xffe7	;1111		

; 125ns PreComp Table

preComp125:	.word	0xe7ff	;0000
	.word	0xcfff	;0001 early 125ns
	.word	0xffe7	;0010
	.word	0xfff3	;0011 late 125ns
	.word	0xffff	;0100
	.word	0xffff	;0101
	.word	0xffcf	;0110 early 125ns
	.word	0xffe7	;0111
	.word	0xf3ff	;1000 late 125ns
	.word	0xe7ff	;1001
	.word	0xffe7	;1010
	.word	0xfff3	;1011 late 125ns
	.word	0xffff	;1100
	.word	0xffff	;1101
	.word	0xffcf	;1110 early 125ns
	.word	0xffe7	;1111

; 250ns PreComp Table

preComp250:	.word	0xe7ff	;0000
	.word	0x9fff	;0001 early 250ns
	.word	0xffe7	;0010
	.word	0xfff9	;0011 late 250ns
	.word	0xffff	;0100
	.word	0xffff	;0101
	.word	0xff9f	;0110 early 250ns
	.word	0xffe7	;0111
	.word	0xf9ff	;1000 late 250ns
	.word	0xe7ff	;1001
	.word	0xffe7	;1010
	.word	0xfff9	;1011 late 250ns
	.word	0xffff	;1100
	.word	0xffff	;1101
	.word	0xff9f	;1110 early 250ns
	.word	0xffe7	;1111

; 375ns PreComp Table

  .if debugPrecomp

preComp375:	.word	0xe7ff	;0000
	.word	0x3fff	;0001 early 375ns
	.word	0xffe7	;0010
	.word	0xfffc	;0011 late 375ns
	.word	0xffff	;0100
	.word	0xffff	;0101
	.word	0xff3f	;0110 early 375ns
	.word	0xffe7	;0111
	.word	0xfcff	;1000 late 375ns
	.word	0xe7ff	;1001
	.word	0xffe7	;1010
	.word	0xfffc	;1011 late 375ns
	.word	0xffff	;1100
	.word	0xffff	;1101
	.word	0xff3f	;1110 early 375ns
	.word	0xffe7	;1111
  .endif

;-----------------------------------------------------------------------------
; nextHole- return time between the most recent sector/index hole and the next
;     sector/index hole. 
;  returns: w1 = time between pulses (500ns LSB)
;   	wPrevHole = updated with IC2 capture value for this hole
;	clobbers w0,w1
;-----------------------------------------------------------------------------
nextHole:	btst.w	IC2CON1,#ICBNE	;capture occur?	
	bra	z,nextHole		;no, keep waiting
	mov.w	IC2BUF,w0		;w0=counter at new pulse
	sub.w	w0,wPrevHole,w1	;w1=time between pulses			
	mov.w	w0,wPrevHole	;save new previous IC value
	return
	
;****************************************************************************************
;
;  ioPortInt - ISR for handling reads and writes to the Altair FDC control and
;     status registers (done through the PIC EPMP). 
;
;     Some register OUTs from 8080 code are immediately followed by a register IN.
;     Response here needs to be less than 2us (to handle up to a 4MHz processor)
;     between an OUT action that then updates a status register.
;
;     This code makes the assumption that it will never get behind, i.e., it can
;     process the interrupts faster than the 8080 will ever do I/O.
;
;     Note that a drive deselect or new drive select bypasses the normal retfie 
;     interrupt exit and jump directly to the idle state or the verify index state.
;     This means that executing code may not regain control.
;
;****************************************************************************************
ioPortInt:	bclr.w	IFS2,#PMPIF		;clear the PMP interrupt
	
; Read and write data registers are checked first to provide shortest ISR time
;    possible for these interrupts. Data is moved from the track buffer to
;    the PMP (read), or from the PMP to the track buffer (write)

chkReadData: btst.w	PMSTAT,#pmREAD_DATA	;read data register read?
	bra	z,chkWriteData	;no
	mov.b	[wRemovePtr++],[pReadData]  ;put out next byte for 8080
	pop	w0		;restore w0
	retfie	

chkWriteData: btst.w	PMSTAT,#pmWRITE_DATA	;write data register written?
	bra	z,chkSelect		;no
	mov.b	[pWriteData],[wInsertPtr++] ;put data from 8080 into trackBuf
	pop	w0		;restore w0
	retfie

;-------------------------------------------------------------------------------------------
; chkSelect - See if the drive select register is written.
;-------------------------------------------------------------------------------------------
chkSelect:	btst.w	PMSTAT,#pmSELECT_CMD	;select register written?
	bra	z,chkDriveCmd	;no, go check drive command
	mov.b	selectCmd,WREG	;w0=select command, clear status flag
	btst.z	w0,#scDESELECT	;de-select requested?
	bra	nz,enterIdle	;yes, enter the idle state

; A drive select command was written. Verify the passed drive number and see if it is 
;    already selected. If so, just exit. If invalid, do a drive deselect. Otherwise,
;    do a deselect followed by a select.

	and.w	#scDRIVE_MASK,w0	;get drive number alone
	cp.w	w0,#NUM_DRIVES	;valid drive number?
	bra	geu,enterIdle	;no, deselect current drive and idle
	btst.z	[pLATB],w0		;same drive already selected?
	bra	z,ioExit		;yes, just exit

; Select the drive, turn on the motor and initialize drive status

	mov.w	#B_DESELECT,w1	;deselect all on previous drive
	mov.w	w1,LATB
	mov.w	wFFFF,statusRegs	;driveStatus and sectorPos to 0xff
	
	bclr.w	SR,#C		;carry bit to zero
	bsw.c	[pLATB],w0		;assert drive select for new drive
	mov.b	#dsINIT_VALUE,w0	;init drive status register to
	mov.b	WREG,driveStatus	;   default "selected" values

	btsc.w	PORTD,#D2_TRACK0	;track zero asserted from drive?
	bclr.b	driveStatus,#dsTRACK0	;yes, assert track zero

	btsc.w	PORTF,#F0_BUS_PINTE	;PINTE signal from bus asserted?
	bclr.b	driveStatus,#dsINTE	;yes, assert INTE in drive status

	btsc.w	PORTD,#D0_WRITE_PROTECT	;write protect asserted from drive?
	bclr.b	driveStatus,#dsWRITE_PROTECT  ;yes

	bclr.w	LATB,#B11_MOTOR_ON	;turn on motor
	clr.w	motorTimer		;restart motor timer
	clr.w	flags		;init all flags off
	bra	waitMotor		;enter sector processing at waitMotor

;-------------------------------------------------------------------------------------------
; chkDriveCmd - Drive command register written? If so, process it. Simultaneous
;    commands are possible, but some combinations are meaningless. The following
;    logic is implemented:
;
;  if (step in/out or head load/unload) {
;     if (step out)
;        step out
;     else if (step in)
;        step in
;
;     if (head load)
;        load head
;     else if (head unload)
;        unload head
;  }
;  else {
;     if (read enable)
;        set internal read enable flag
;     else {
;        if (write enable)
;           set internal write enable flag
;        if (head current switch)
;           set internal head currrent flag
;     }
;  }
;   
;-------------------------------------------------------------------------------------------
chkDriveCmd: btst.w	PMSTAT,#pmDRIVE_CMD	;drive command written?
	bra	z,chkTrackNum	;no, see if track register written
	mov.b	driveCmd,WREG	;w0=drive command, clear status flag
	btst.b	driveStatus,#dsREADY	;do we show is drive ready?
	bra	nz,ioExit		;no, ignore the command
	push.w	w1		;save w1
	
; Check for any head step or head load/unload commands first. If any of these, immediately
;    set head status to false and disable the sector position register (0xff). This
;    is done because 8080 software may quickly read the status registers after writing 
;    this command. We update the status registers within 19-22 cycles (1.2-1.4us).

	and.b	w0,#dcHEAD_MASK,w1	;any of the head move/load commands?
	bra	z,ioChkRead		;no, go check read enable
	bset.b	driveStatus,#dsHEAD_STATUS   ;de-assert head status flag
	mov.b	wFFFF,[pSectorPos]	;sector register to 0xff
	and.b	w0,#dcMOVE_MASK,w1	;any of the head move commands?
	bra	z,ioChkHead		;no, must be head load/unload
	bset.b	driveStatus,#dsMOVE_HEAD   ;de-assert move head flag

;=================
; Step Out command
;=================
	btst.z	w0,#dcSTEP_OUT	;step out command?
	bra	z,ioStepIn		;no, must be step in command
	btst.w	PORTD,#D2_TRACK0	;already at track 0?
	bra	nz,ioDoTimers	;yes, don't issue step to drive
	mov.w	#STEP_CHAR,w1	;data byte that when serially transmitted...
	mov.w	w1,U2TXREG		;  generates a 2us step pulse
				;(step direction line already high = out)
 	bra	ioDoTimers		;start step timer

;=================
; Step In command 
;=================
ioStepIn:	mov.w	#DIR_IN_CHAR,w1	;data byte that when serially transmitted...
	mov.w	w1,U3TXREG		;  sets step direction low (in) for 5us
	mov.w	#STEP_CHAR,w1	;data byte that when serially transmitted...
	mov.w	w1,U2TXREG		;  generates a 2us step pulse
 
; ioDoTimers - start head step timer and head settle timer using OC1 and OC2

ioDoTimers: push	w0		;preserve w0
	mov.w	ptStepTHD,w0	;w0=step time
	add.w	OC1TMR,WREG		;add timeout to current OC1 timer
	mov.w	w0,OC1R		;store as new OC1 compare value
	mov.w	#HEAD_SETTLE_TIME,w0	;w0=head settle timer
	add.w	OC2TMR,WREG		;add timeout to current OC2 timer
	mov.w	w0,OC2R		;store as new OC2 compare value
	pop	w0		;restore w0
	bclr.w	IFS0,#OC1IF		;clear OC1 interrupt flag
	bclr.w	IFS0,#OC2IF		;clear OC2 interrupt flag
	bset.w	IEC0,#OC1IE		;enable OC1 head step interrupt
	bset.w	IEC0,#OC2IE		;enable OC2 head settle interrupt
	bra	ioCmdExit

;=================
; Head Load command (without a move head command as well). For this drive, the
;   head load command starts the drive motor if not already running.
;=================
ioChkHead:	btst.z	w0,#dcHEAD_LOAD	;head load command?
	bra	z,ioChkUnload	;no
	bclr.b	driveStatus,#dsHEAD_STATUS   ;assert head loaded status

ioLoadHead:	bset.w	flags,#fHEAD_LOADED	;indicate that head is loaded 
	btst.w	LATB,#B11_MOTOR_ON	;motor running?
	bra	z,ioCmdExit		;yes, motor already on
	bclr.w	LATB,#B11_MOTOR_ON	;else, turn on the motor
	bset.w	flags,#fWAIT_MOTOR	;pass control to waitMotor
	bra	ioCmdExit

;=================
; Head Unload command. Clear the head loaded flag.
;=================
ioChkUnload: btst.z	w0,#dcHEAD_UNLOAD	;head unload command?
	bra	z,ioCmdExit		;no, go check for interrupt enable/disable
	bclr.w	flags,#fHEAD_LOADED	;flag that head not loaded or settled
	bra	ioCmdExit		;go exit

;=================
; Read enable command. Set the internal read enable flag. Non-interrupt codes looks at
;    this flag early in the sector to determine whether or not to enable a disk read.
;=================
ioChkRead:	btst.z	w0,#dcREAD_ENABLE	;read enable command?
	bra	z,ioChkWrite	;no, go check write enable
	bset.w	flags,#fREAD_ENABLE	;set read enable flag
	bset.b	driveStatus,#dsNRDA	;de-assert read data handshake
	bra	ioCmdExit

;=================
; Write enable command. Set the internal write enable flag. Non-interrupt codes looks at
;    this flag early in the sector to determine whether or not to enable a disk write.
;=================
ioChkWrite:	btst.z	w0,#dcWRITE_ENABLE	;write enable command?
	bra	z,ioChkHCS		;no
	bset.w	flags,#fWRITE_ENABLE	;set write enable flag
	bset.b	driveStatus,#dsENWD	;de-assert write data handshake

;=================
; Header Current Switch (HCS) command. Set the internal write current flag. 
;    Non-interrupt code looks at this flag when write is selected to assert/clear
;    the write current signal to the drive
;=================
ioChkHCS:	btsc.w	w0,#dcHEAD_CURR_SW	;head current switch command?
	bset.w	flags,#fLOW_WRITE_CURRENT  ;yes, set low current flag

; ioCmdExit - restore register and exit from interrupt or directly to waitMotor
;    based on the fWAIT_MOTOR flag.

ioCmdExit:	pop.w	w1		;restore w1
	btst.w	flags,#fWAIT_MOTOR	;exit into waitMotor?
	bclr.w	flags,#fWAIT_MOTOR
	bra	nz,waitMotor	;yes, exit wait for motor
ioExit:	pop	w0		;otherwise, normal exit
	retfie

;-------------------------------------------------------------------------------------------
; chkTrackNum - See if a value has been written to the track register. If so, save it
;    in curTrack and update side select based on the track number. Even tracks on bottom,
;    odd tracks on top through cylinder 71. For cylinders 72-76, only the bottom side
;    of the disk is used for better reliability with old media
;-------------------------------------------------------------------------------------------
chkTrackNum: btst.w	PMSTAT,#pmTRACK_NUM	;track number register written?
	bra	z,2f		;no, just exit
	mov.b	trackNum,WREG	;save track number in curTrack
	mov.b	WREG,curTrack
	mov.b	#END_TOP_TRACK,w0	;>= max top side track?
	cp.b	curTrack		;curTrack-END_TOP_TRACK
	bra	geu,1f		;yes, force bottom side

	btsc.b	curTrack,#0		;else odd tracks on top side
	bclr.w	LATB,#B12_SIDE_SELECT	;select top head  
	btss.b	curTrack,#0		;even tracks on bottom
1:	bset.w	LATB,#B12_SIDE_SELECT	;select bottom head

2:	pop	w0		;restore and exit
	retfie

;****************************************************************************************
;
;  headMoveInt - ISR to process expiration of OC1. This interrupt occurs 10.5ms after
;      a step in/out command is issued. Update the track0 bit in the status register
;      and assert the move head status bit in the drive status register.
;
;****************************************************************************************
headMoveInt: bclr.w	IEC0,#OC1IE		;disable the OC1 interrupt
	btst.b	driveStatus,#dsREADY	;drive still ready?
	bra	nz,1f		;no, just exit
	btsc.w	PORTD,#D2_TRACK0	;copy track 0 bit without a glitch...
	bclr.b	driveStatus,#dsTRACK0	;   in track 0 flag in register
	btss.w	PORTD,#D2_TRACK0
	bset.b	driveStatus,#dsTRACK0
	bclr.b	driveStatus,#dsMOVE_HEAD  ;assert move head signal
1:	pop.w	w0
	retfie

;****************************************************************************************
;
;  headStatusInt - ISR to process expiration of OC2. This interrupt occurs after a
;      step in/out or head load command is issued. Assert the head loaded status bit 
;      in the drive status register. Drive still ready is qualified by verification
;      that the head loaded bit is asserted.
;
;****************************************************************************************
headStatusInt:
	bclr.w	IEC0,#OC2IE		;disable the OC2 interrupt
 	btsc.w	flags,#fHEAD_LOADED	;head still loaded?
	bclr.b	driveStatus,#dsHEAD_STATUS   ;yes, assert head status flag
	pop.w	w0
	retfie

;****************************************************************************************
;
;  changeInt - ISR to process port change interrupts. 
;     The PINTE line from the 8080 (interrupt enabled signal) generates an interrupt
;     so the INTE status bit in the drive status register can be updated.
;
;****************************************************************************************
changeInt:	bclr.w	IFS1,#CNIF		;clear the port change interrupt flag
	btst.w	PORTF,#F0_BUS_PINTE	;PINTE signal from bus asserted?
	bra	z,pInteOff		;no	
	btss.b	driveStatus,#dsREADY	;skip next inst if drive not ready
	bclr.b	driveStatus,#dsINTE	;assert INTE in drive status
	pop.w	w0		;restore w0 and exit
	retfie

; pInteOff - PINTE not asserted. De-assert INTE in the drive status register

pInteOff:	bset.b	driveStatus,#dsINTE	;de-assert INTE in drive status
	pop.w	w0		;restore w0 and exit
	retfie

;****************************************************************************************
;
;  bootSector - HD Floppy boot code in an original Altair 137 byte sector
;     The disk format and data interchange between the 8080 and the FDC+ is completely
;     different for the HD floppy drive option than for standard Altair drives. This 
;     means a different boot PROM is required to boot the loader found at the start of
;     track zero on an HD floppy drive. As a way to get around this requirement, the
;     main sector loop  returns the standard Altair 137 byte sector below whenever
;     the index pulse is reached (i.e., new sector) AND an HD floppy read is not in
;     progress. This means an Altair boot PROM will receive this sector and jump
;     to it as if it came from a standard Altair disk.
;
;****************************************************************************************
bootSector:	.byte	0x80,0x80,0x00,0xf3,0x31,0x00,0x3f,0x3e
	.byte	0x80,0xd3,0x08,0xaf,0xd3,0x08,0xd3,0x0b
	.byte	0x3e,0x04,0xd3,0x09,0xdb,0x08,0xe6,0x02
	.byte	0xc2,0x11,0x00,0xdb,0x08,0xe6,0x40,0xca
	.byte	0x26,0x00,0x3e,0x02,0xd3,0x09,0xc3,0x11
	.byte	0x00,0xdb,0x09,0x1f,0xda,0x26,0x00,0x21
	.byte	0x00,0x40,0x01,0x00,0x14,0x3e,0x10,0xd3
	.byte	0x09,0xdb,0x08,0xb7,0xfa,0x36,0x00,0xdb
	.byte	0x0a,0xdb,0x0a,0x77,0x23,0xdb,0x0a,0x77
	.byte	0x23,0x0b,0x78,0xb1,0xc2,0x3e,0x00,0xdb
	.byte	0x0b,0x32,0xff,0x3f,0xe6,0x7f,0xc2,0x00
	.byte	0x00,0x3e,0x80,0xd3,0x08,0xc3,0x00,0x40
	.byte	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	.byte	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	.byte	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	.byte	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	.byte	0x00,0x00,0x00,0xff,0xa6,0x00,0x00,0x00
	.byte	0x00,0x00
	.equ	BOOT_SECTOR_LENGTH,.-bootSector

	.end
