;***************************************************************************************
; 	
;  fd3712.s 
;     This module duplicates the iCOM FD3712 drive cabinet from the perspective of
;     the S-100 computer. The FDC+ makes Shugart 800 (or equivalent) drives look like
;     they are an FD3712 drive cabinet connected to the S-100 machine using the S-100
;     interface board that iCOM and Pertec sold. This code also supports 5.25" HD
;     (1.2Mb) drives like the Teac 55-GFR (360 rpm mode).
;
;  Version History
;    1.0   08/21/2021  M. Douglas
;	- Original
;    1.1   09/11/2021  M. Douglas
;	- Update the command processing interrupt routine to minimize the
;	  time until critical output data is updated. Also reduce the ISR
;	  time for status register read events. These require no processing
;	  but generate an interrupt and result in ISR time anyway.
;    1.2   01/13/2026  M. Douoglas
;	  Do NOT reset the write FIFO pointers when the CLEAR command is issued.
;	  This matches the operation of the real hardware and fixes a problem
;	  that FDOS exposed.
;
;-------------------------------------------------------------------------------
; 
;  PIC H/W Assignments
;
;     EPMP (PMP) handles port I/O from the 8080
;     UART2 generates the step signal
;     SPI1 generates write data pulses
;     OC1 is a general purpose in-line timer
;     OC2 is a long time frame interrupt timer
;     IC1 captures read data pulses
;     IC2 captures index pulses
;     TMR4 at 250khz is clock source for OC1, OC2
;
;***************************************************************************************
	.include 	"common.inc" 
 
; The following working registers have permanent variable assignments and cannot
;   be used for any other purpose. 

	.equ	wPrevData,w14	;input capture at previous data pulse
	.equ	pUserRead,w13	;pointer to user read buffer
	.equ	pUserWrt,w12	;pointer to user write buffer
	.equ	pLATB,w11		;pointer to LATB (drive interface)
	.equ	wCrc,w10		;16 bit CRC
	.equ	pDiskBuf,w9		;pointer to disk I/O buffer
	.equ	wXferLen,w8		;disk transfer length
	.equ	pCrcTable,w7	;pointer to CRC table
	.equ	wIsr,w6		;dedicated register for ISR use

; Drive interface equates. Outputs are all asserted low (not inverted from processor
;   to the drive). Inputs are asserted high (inverted from drive to processor).

	.equ	B0_DRIVE_SELECT1,0
	.equ	B1_DRIVE_SELECT2,1
	.equ	B2_DRIVE_SELECT3,2
	.equ	B3_DRIVE_SELECT4,3
	.equ	B6_WRITE_DATA,6
	.equ	B7_STEP_DIRECTION,7
	.equ	B8_TRIM_ERASE,8	;output, not used
	.equ	B9_WRITE_GATE,9
	.equ	B11_HEAD_LOAD,11
	.equ	B12_SIDE_SELECT,12	;output, not used
	.equ	B13_WRITE_CURRENT,13
	.equ	F4_STEP,4	
	.equ	D10_READ_DATA,10
	.equ	D0_WRITE_PROTECT,0
	.equ	D1_READY,1
	.equ	D2_TRACK0,2
	.equ	D3_INDEX,3
	.equ	F1_TRUE_READY,1	;input, not used
	.equ	PORTB_INIT,0x3bcf	;all drive signals off
	.equ	PORTB_INIT0,0x3bce	;same, but drive 0 selected

; Head step equates.

	.equ	HEAD_SETTLE,20000/4	;20ms head settle time (plus step time)
	.equ	STEP_CHAR,0xfe	;character when sent serially generates a 2us
				;pulse at 1Mb/s (start bit and bit 0 low)

; Index equates on IC2

	.equ	INDEX_TIME,166668/4	;166.667ms full revolution (68 is right)
	.equ	INDEX_MAX,170000/4	;102% of full revolution
	.equ	INDEX_MIN,163332/4	;98% of full revolution

; Read and write data equates. Read data pulses are captured by IC1. The timebase for
;    IC1 is 16Mhz. For FM encoding, the bit time is 4us. If no pulse occurs between
;    two 4us pulses, the data bit is a zero. If a pulse is present in between
;    two 4us pulses, the data bit is a one. Writing of data is done with SPI1 at 
;    500ns per SPI bit.

	.equ	NORMAL_CLOCK,0x7f7f	;two bits of normal clock
	.equ	CLOCK_C7_54,0xffff	;bits 5,4 missing clock
	.equ	CLOCK_C7_32,0xff7f	;bit 3 missing clock
	.equ	CLOCK_D7_54,0xff7f	;bit 5 missing clock
	.equ	CLOCK_D7_32,0xff7f	;bit 3 missing clock

	.equ	DATA_OR_CLOCK,44	;separate 2us vs 4us with 16*2.75us
	.equ	DATA_MARK_TIME,1000/4	;1ms to find data mark after address mark

; longTimer - Interrupt driven timer for long timeouts like command abort and 
;    motor off. Output Compare 2 is used to time 100ms tics are counted by the
;    oc2Int interrupt routine.

	.equ	LONG_TIC,100000/4	;100ms per tic
	.equ	STARTUP_TIME,5	;extra 0.5s for motor startup
	.equ	MOTOR_TIMEOUT,4*10	;4 second motor off/head unload timeout
	.equ	CMD_ABORT_TIMEOUT,1*10	;command timeout is 1 second
	.equ	CMD_ABORT_TIME,MOTOR_TIMEOUT-CMD_ABORT_TIMEOUT

; FD3712 status register

	.equ	sDDAM,7		;deleted data mark
	.equ	sDRIVE_FAIL,5	;drive not ready
	.equ	sWRITE_PROTECT,4
	.equ	sCRC_ERROR,3
	.equ	sBUSY,0		;controller busy
	.equ	STATUS_INIT,0x40	;startup value

; FD3712 command register

	.equ	cbCLEAR,7		;clear controller bit
	.equ	cbSHIFT,6		;shift read data bit
	.equ	cbREAD,6		;examine read data bit
	.equ	cbEXECUTE,0		;execute a command bit

	.equ	cREAD,0x03		;read sector
	.equ	cWRITE,0x05		;write sector
	.equ	cREADCRC,0x07	;read to verify CRC
	.equ	cSEEK,0x09		;seek to track
	.equ	cCLRFLAGS,0x0b	;clear error flags
	.equ	cSEEK0,0x0d		;restore to track 0
	.equ	cLOADTRK,0x11	;load track number
	.equ	cLOADCFG,0x15	;load config register
	.equ	cLOADSEC,0x21	;load sector & drive number
	.equ	cLOADWRT,0x31	;load write buffer

; User configuration register bits

	.equ	fFORMAT,5		;formatting disk when true

; flags variable

	.equ	fNEW_DRIVE,0	;new (different) drive selected
	.equ	fREAD_CRC,1		;don't save data, CRC only

; Misc equates

	.equ	PIC_INTS_OFF,0xe0	;write to SR to disable PIC interrupts

;-----------------------------------------------------------------------------
; UpdateCrc macro - Updates wCrc with the byte passed in w0.
;    This code, along with the CRC table generated during initializaiton,
;    implements CRC-16/IBM-3740, aka CRC-16/CCITT-FALSE. Defined by the
;    following parameters: 
;	width=16 poly=0x1021 init=0xffff refin=false refout=false
;    	xorout=0x0000 check=0x29b1 residue=0x0000
;    The XMODEM CRC is the same except init=0x0000
;
;    6 cycles (0.375us) as in-line macro
;    Clobbers w1
;----------------------------------------------------------------------------- 
   .macro UpdateCrc
	lsr.w	wCrc,#8,w1		;w1=MSB of CRC in LSB
	xor.b	w1,w0,w1		;w1=(CRC>>8)^data
	sl.w	w1,w1		;16 bit values in table
	mov.w	[pCrcTable+w1],w1	;w1=crcTable[(CRC>>8)^data]
	sl.w	wCrc,#8,wCrc	;LSB of old CRC into the MSB
	xor.w	wCrc,w1,wCrc	;CRC=(crc << 8)^crcTable[((crc >> 8)^data)]
   .endm

;-----------------------------------------------------------------------------
;  RAM definitions
;-----------------------------------------------------------------------------
	.section *,bss,near
cmdData:	.space	1		;command and data as a word
userData:	.space	1		;data is in the MSB

userDrive:	.space	2		;drive number from user
userTrack:	.space	2		;track number from user
userSec:	.space	2		;sector number from user
userConfig:	.space	2		;config register from user
userStatus:	.space	2		;controller status for user

dataShift:	.space	2		;read data shift
clockShift:	.space	2		;read clcok shift register
clock54:	.space	2		;clock pattern for bits 5,4
clock32:	.space	2		;clock pattern for bits 3,2

flags:	.space	2		;see flag bits above
longTimer:	.space	2		;0.1s LSBit interrupt timer
curTrack:	.space	2		;current track drive is on

addrMark:				;address mark buffer
amTrack:	.space	1
	.space	1
amSector:	.space	1		;sector number
amLength:	.space	1		;sector length code
amCrc:	.space	2		;CRC
	.equ	ADDR_MARK_LEN,$-addrMark
	.space	2		;safety padding

; Read buffer, write buffer and CRC table

	.equ	SECTOR_LEN,128
userReadBuf: .space	SECTOR_LEN		;read buffer
	.equ	USER_READ_END,$
	.space	2		;plus space for CRC

userWrtBuf:	.space	SECTOR_LEN		;write buffer
	.equ	USER_WRT_END,$

crcTable:	.space	512		;CRC table generated during initialization
		
;****************************************************************************************
;  
; FD3712 - entry point for iCOM FD3712 drive type
;
;****************************************************************************************
	.text
	.global	FD3712

FD3712:	mov.w	#STACK_ADDR,w15	;init stack pointer
	mov.w	#PIC_INTS_OFF,w0	;disable all interrupts
	mov.w	w0,SR

; Initialize the parallel port to handle the I/O IN and OUT instructions from the 8080. The
;     PMP is run in addressable slave mode where I/O addresses from the 8080 map as shown
;     below.
;
;     I/O address            PIC register
;	 8	PMDOUT1, PMDIN1 low byte
;	 9	PMDOUT1, PMDIN1 high byte
;	10	PMDOUT2, PMDIN2 low byte (not used)
;	11	PMDOUT2, PMDIN2 high byte (not used)

	mov.w	#0x0281,w0		;enhanced PSP mode, CS1 on PMA14, interrupts on
	mov.w	w0,PMCON1
	mov.w	#0xc000,w0		;PMWR, PMRD enabled
	mov.w	w0,PMCON3
	mov.w	#0x4003,w0		;CS1(A14), A1, A0 enabled
	mov.w	w0,PMCON4
	mov.w	#0x6000,w0		;CS1 on and active high, RD, WR active low
	mov.w	w0,PMCS1CF
	mov.w	#0x0001,w0		;use TTL buffers
	mov.w	w0,PADCFG1
	bset.w	PMCON1,#15		;enable EPMP
 
; Map UART2 transmit to the step line. Receive data is not used, but if left mapped
;    to the default RP63, receive data is low which looks like a continuous break.
;    It is mapped here to RP18 (Port B, bit 5, ICSP Clock) which initialization sets
;    as an input with internal pullup. This pin remains unused and high in normal operation.

	mov.b	#18,w0		;UART Receive data tied high on RP18
	mov.b	WREG,RPINR19L	;UART 2
	mov.b	#5,w0		;UART 2 TX code 5 is on RP10
	mov.b	WREG,RPOR5L

; Initialize UART2 for 8N1 at 1M baud (1us per bit) to generate the step pulse

	mov.w	#0x0008,w0		;serial port off, 8N1, BRGH=1
	mov.w	w0,U2MODE
	mov.w	#3,w0		;1M baud
	mov.w	w0,U2BRG	
	bset.w	U2MODE,#UARTEN	;UART on
	bset.w	U2STA,#UTXEN	;transmitter on too

; Init TMR4 to use a 250khz timebase. This timer is the sync source for the
;   index pulse capture (IC2). TMR4 is also the clock source (not sync source)
;   for OC1 and OC2. The timer is enabled later after IC2 is initialized.

	mov.w	#0x0020,w0		;Fcy/64 for 250khz
	mov.w	w0,T4CON
	mov.w	#0xffff,w0
	mov.w	w0,PR4

; OC1 and OC2 use TMR4 at 250khz as clock source.

	mov.w	#0x0020,w0		;free running, no output pin
	mov.w	w0,OC1CON2
	mov.w	w0,OC2CON2
	mov.w	#0x0803,w0		;TMR4 and one-shot toggle mode
	mov.w	w0,OC1CON1
	mov.w	w0,OC2CON1

;  Map IC2 to the index line. Configure to capture on rising edge (falling edge
;    from the drive is inverted). IC2 is sync'd to TMR4 running at 250khz.

	mov.b	#22,w0		;IC2 detects index pulses on RP22
	mov.b	WREG,RPINR7H
	mov.w	#0x000e,w0		;IC2 sync'd to TMR4
	mov.w	w0,IC2CON2
	mov.w	#0x0803,w0		;rising edge capture, TMR4 is clk source
	mov.w	w0,IC2CON1	

	bset.w	T4CON,#TON		;turn on TMR4 now that IC2 ready

;  Map IC1 to the read data line. Configure to capture on rising edge using the system clock.

	mov.b	#3,w0		;IC1 detects read data pulses on RP3
	mov.b	WREG,RPINR7L
	clr.w	IC1CON2		;IC1 not sync'd to any other source
	mov.w	#0x1c03,w0		;IC1 capture on rising edge, Fcy clock
	mov.w	w0,IC1CON1	
	
;  SPI1 is used to generate the write data waveform. Initialize for 16 bits at 2Mhz.
;     This will send two bits of drive data per 16 bits of SPI data. The SPI is set
;     to interrupt when the FIFO is empty (16 bit word still in shift register).

	mov.b	#7,w0		;SPI1 Data out is on RP6
	mov.b	WREG,RPOR3L		;assign SPI1 data out to disk write data

	mov.w	#0x0001,w0		;select enhanced mode (8 word buffer)
	mov.w	w0,SPI1CON2
	mov.w	#0x1423,w0		;no clk or data out, 16 bit, master, 2MHz
	mov.w	w0,SPI1CON1
	mov.w	#0x0018,w0		;interrupt when fifo empty
	mov.w	w0,SPI1STAT
	bset.w	SPI1STAT,#SPIEN	;enable SPI
	mov.w	#0xffff,w0		;write all 1's so line is idling high
	mov.w	w0,SPI1BUF

; Put our local ISRs in the interrupt table

	mov.w	#tbloffset(ioPortInt),w0    ;PMP interrupt (8080 I/O)
	mov.w	w0,PMPInterrupt

	mov.w	#tbloffset(oc2Int),w0   ;OC2 interrupt (longTimer)
	mov.w	w0,OC2Interrupt

; Initialize drive control lines

	mov.w	#LATB,pLATB		;pLATB is a register pointing to LATB
	mov.w	#PORTB_INIT0,w0
	mov.w	w0,LATB		;drive 0 selected, but all other signals off

; Initialize variables

	clr.w	cmdData		;no command present
	clr.w	userDrive		;drive 0
	clr.w	userTrack		;track 0
	clr.w	curTrack
	clr.w	userSec		;won't match anything
	clr.w	userConfig		;configuration register

	clr.w	flags		;all flags false
	mov.w	#userReadBuf,pUserRead	;init buffer pointers
	mov.w	#userWrtBuf,pUserWrt

	mov.w	#NORMAL_CLOCK,w0	;default to normal data clock
	mov.w	w0,clock54
	mov.w	w0,clock32

	mov.w	#STATUS_INIT,w0	;init user-visible status
	mov.w	w0,userStatus
	mov.w	w0,PMDOUT1		;put status on the port for the 8080

; Generate the crcTable in RAM. Takes less than 1ms.

	clr.w	w0		;w0=data byte 0-255
	mov.w	#0x1021,w1		;w1=CRC polynomial
	mov.w	#crcTable,pCrcTable	;pCrcTable is a register
	
gcByteLoop:	sl.w	w0,#8,wCrc		;starting CRC is byte<<8
	mov.w	#8,w2		;8 bits in the byte

gcBitLoop:	sl.w	wCrc,wCrc
	bra	nc,1f		;XOR poly only if MSBit set
	xor.w	wCrc,w1,wCrc
1:	dec.w	w2,w2
	bra	nz,gcBitLoop

	mov.w	wCrc,[pCrcTable++]	;save value in table
	inc.b	w0,w0		;move to next byte value
	bra	nz,gcByteLoop

	mov.w	#crcTable,pCrcTable	;restore the pCrcTable pointer

; Enable interrupts and run.

	mov.w	PMDIN1,w0		;ensure EPMP is completely reset
	mov.w	PMDIN2,w0
	clr.w	PMDOUT2		;these outputs not used
	bclr.w	PMSTAT,#IBOV	;make sure errors are cleared
	bclr.w	PMSTAT,#OBUF
	bset.w	IEC2,#PMPIE		;enable parallel port interrupt
				;fall into idle loop

;-------------------------------------------------------------------------------
;  Idle loop - do nothing while waiting for the command processing interrupt
;     to select a drive and change the execution point to index verification.
;-------------------------------------------------------------------------------
idle:	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	clr.w	SR		;set interrupt level to zero (no interrupt)

idleLoop:	btst.w	U1STA,#URXDA	;test for a character
	bra	z,idleLoop

	mov.w	U1RXREG,w0		;get the character
	btst.w	PORTC,#C13_MONITOR_SW	;monitor enabled?
	bra	z,idleLoop		;no

	and.w	#0x5f,w0		;convert lower case to upper
	sub.b	#'X',w0		;exit requested?
	bra	nz,idleLoop		;no

	reset			;exit back to boot loader

;-------------------------------------------------------------------------------
;  ioPortInt - ISR for handling reads and writes to the Altair FDC+ registers.
;     These registers are implemented with the PIC EPMP and control all
;     drive activity.
;
;     16 cycles, 1.00 us, total ISR time taken when host does an input
;     25 cycles, 1.56 us, worst case for examine to valid read data
;     23 cycles, 1.44 us, for BUSY to assert for a command
;     28 cycles, 1.75 us, for SHIFT command until new read data valid
;
;   0x08 - Command Out, Status in, Data in
;   0x09 - Data Out
;-------------------------------------------------------------------------------
ioPortInt:	bclr.w	IFS2,#PMPIF		;clear the PMP interrupt
	mov.w	PMDIN2,w0		;remove 2nd word to be safe
	btst.w	PMSTAT,#IB0F	;command register written?
	bra	nz,ipCmdWrite	;yes
	mov.w	PMDIN1,w0		;else remove possible write to data reg
	pop	w0		;restore w0
	retfie			;exit

; ipCmdWrite - Command register has been written

ipCmdWrite:	mov.w	PMDIN1,w0		;cmd in lsb, data in msb
	mov.w	cmdData,wIsr	;wIsr=previous command in lsb
	mov.w	w0,cmdData		;save new command and data
	xor.w	w0,wIsr,w0		;w0=changed bits

	btst.z	w0,#cbEXECUTE	;did execute bit change?
	bra	z,ipNoCmd		;no
	btst.w	cmdData,#cbEXECUTE	;execute now asserted?
	bra	z,ipNoCmd		;no
	
	mov.w	userStatus,w0	;assert a temp busy asap
	bset.w	w0,#sBUSY
	mov.b	WREG,PMDOUT1L 

; Execute bit is newly set, determine the command issued

	btst.w	cmdData,#cbSHIFT	;shift read buffer?
	bra	nz,cmdShift		;yes

	btst.w	cmdData,#cbCLEAR	;clear/reset request?
	bra	nz,cmdClear		;yes
	
	btst.w	userStatus,#sBUSY	;already busy?
	bra	nz,ipStatus		;yes, ignore command
		
	mov.b	#cLOADWRT,w0	;load write buffer?
	cp.b	cmdData
	bra	z,cmdLoadWrt	;yes

	mov.b	#cLOADSEC,w0	;load sector/unit?
	cp.b	cmdData
	bra	z,cmdLoadSec	;yes

	mov.b	cmdData,WREG	;w0=command in lsb
	
	cp.b	w0,#cCLRFLAGS	;clear error flags?
	bra	z,cmdClrFlag

	cp.b	w0,#cLOADTRK	;load track?
	bra	z,cmdLoadTrk

	cp.b	w0,#cLOADCFG	;load config register?
	bra	z,cmdLoadCfg

; The following commands all result in the controller going "busy"  	

	btst.w	userStatus,#sCRC_ERROR	;CRC error bit set?
	bra	nz,ipStatus		;yes, ignore "busy" commands

	cp.b	w0,#cREAD		;read command?
	bra	z,cmdRead

	cp.b	w0,#cWRITE		;write command?
	bra	z,cmdWrite

	cp.b	w0,#cREADCRC	;read CRC command?
	bra	z,cmdReadCrc

	cp.b	w0,#cSEEK0		;restore to track 0?
	bra	z,cmdSeek0

	cp.b	w0,#cSEEK		;seek command?
	bra	z,cmdSeek
				;invalid command, just ignore

; ipNoCmd - Update the output register with status or read data and exit

ipNoCmd:	btst.w	cmdData,#cbREAD	;read data or status?
	bra	z,ipStatus		;go return status
	mov.b	[pUserRead],w0	;else return read data
	mov.b	WREG,PMDOUT1L
	pop	w0		;restore w0
	retfie			;exit

ipStatus:	bclr.w	userStatus,#sWRITE_PROTECT   ;update write protect status
	btsc.w	PORTD,#D0_WRITE_PROTECT	     ;0=not write protected
	bset.w	userStatus,#sWRITE_PROTECT

	mov.w	userStatus,w0	;return controller status
	mov.b	WREG,PMDOUT1L
	pop	w0		;restore w0
	retfie			;exit

;-------------------------------------------------------------------------------
; Command Processing (non-"busy" commands)
;    Commands that don't require the controller to go "busy" are processed
;    fully within the interrupt routine.
;-------------------------------------------------------------------------------

; cmdShift - Shift read buffer (increment buffer pointer)

cmdShift:	mov.b	[++pUserRead],w0	;get new data out asap
	mov.b	WREG,PMDOUT1L

	mov.w	#USER_READ_END,w0	;reach end?
	cp.w	pUserRead,w0
	bra	ltu,csExit		;no
	mov.w	#userReadBuf,pUserRead	;else, wrap to start
	mov.b	[pUserRead],w0	;update read data
	mov.b	WREG,PMDOUT1L

csExit:	pop	w0		;restore w0
	retfie			;exit

; cmdLoadWrt - Load write buffer with next byte and increment pointer

cmdLoadWrt: mov.b	userData,WREG	;buffer the new write data
	mov.b	w0,[pUserWrt++]	
	mov.w	#USER_WRT_END,w0	;wrap buffer pointer if needed
	cp.w	pUserWrt,w0
	bra	ltu,ipStatus	;no wrap required
	mov.w	#userWrtBuf,pUserWrt	;else, wrap to start
	bra	ipStatus

; cmdLoadSec - Load the sector and drive select byte. The sector is saved in
;    userSec, the drive is saved in userDrive and also physically updates
;    drive select lines.

cmdLoadSec:	mov.b	userData,WREG	;drive number is in bits 7,6
	lsr	w0,#6,w1 
	and.w	#3,w1		;w1=selected drive
	mov.w	w1,userDrive	;save drive number

	and.w	#0x1f,w0		;save requested sector number
	mov.w	w0,userSec
	
	btst.z	[pLATB],w1		;same drive already selected?
	bra	z,ipStatus		;yes, just exit
	
	mov.w	#0x0f,w0		;deselect all drives
	ior.w	LATB
	bclr.w	SR,#C		;use carry=0 to select new drive
	bsw.c	[pLATB],w1		;assert proper drive select bit
	bset.w	flags,#fNEW_DRIVE	;indicate that a different drive slected
	
	sl.w	w1,w1		;position drive num for status byte
	mov.w	userStatus,w0
	and.w	#0xf9,w0		;clear bits 2,1
	ior.w	w1,w0,w0		;insert new drive select
	mov.w	w0,userStatus
	bra	ipStatus

; cmdLoadTrk - Save the user specified track number in userTrack

cmdLoadTrk:	mov.b	userData,WREG
	and.w	#0x7f,w0
	mov.w	w0,userTrack
	bra	ipStatus

; cmdLoadCfg - Load user specified data for the configuration register

cmdLoadCfg:	mov.b	userData,WREG
	mov.w	w0,userConfig
	bra	ipStatus

; cmdClrFlags - Clear CRC error and deleted address mark flags

cmdClrFlag:	bclr.w	userStatus,#sDDAM
	bclr.w	userStatus,#sCRC_ERROR
	bra	ipStatus
	
; cmdClear - Stop any drive activity and reset the controller

cmdClear:	mov.w	#PORTB_INIT,w0	;w0=disable all drive signals
	mov.w	userDrive,w1	;w1=currently selected drive
	bclr.w	SR,#C		;use carry=0 to select drive
	bsw.c	w0,w1		;assert proper drive select bit
	mov.w	w0,LATB		;drive selected but all signals off

	bclr.w	userStatus,#sDDAM	;clear status flags
	bclr.w	userStatus,#sCRC_ERROR
	bclr.w	userStatus,#sBUSY

	bclr.w	userStatus,#sWRITE_PROTECT   ;update write protect status
	btsc.w	PORTD,#D0_WRITE_PROTECT	     ;0=not write protected
	bset.w	userStatus,#sWRITE_PROTECT

	mov.w	userStatus,w0
	mov.b	WREG,PMDOUT1L	;update status to 8080

	bclr.w	IEC0,#OC2IE		;disable OC2 interrupt (longTimer)
	clr.w	flags		;clear program flags
     	bra	idle

;-------------------------------------------------------------------------------
; cmdRead - Read sector userSec from current track into userReadBuf
;-------------------------------------------------------------------------------
cmdRead:	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	rcall	busyCmd		;init for a command that goes "busy"
	mov.w	#userReadBuf,pUserRead	;init user's buffer pointer

rdSecHunt:	rcall	findSec		;find the sector specified in userSector
	clr.w	SR		;allow interrupts again
	mov.w	#DATA_MARK_TIME,w0	;give 1ms to find the data mark
	rcall	setTimer
	rcall	sync2Mark		;look for the data mark
	add.b	#5,w0		;verify data mark (0xfb)
	bra	nz,rdSecHunt

	mov.w	#SECTOR_LEN+2,wXferLen	;wXferLen=number of bytes to read (+2 byte CRC)
	mov.w	#userReadBuf,pDiskBuf	;pDiskBuf->where to put data
	rcall	readBuf		;read into buffer
	cp0.w	wCrc
	bra	z,busyExit		;success
	bra	cmdFail		;else, read failed
	
;-------------------------------------------------------------------------------
; cmdReadCrc - Read sector to check CRC without changing the read buffer
;-------------------------------------------------------------------------------
cmdReadCrc:	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	rcall	busyCmd		;init for a command that goes "busy"

rcSecHunt:	rcall	findSec		;find the sector specified in userSector
	clr.w	SR		;allow interrupts again
	mov.w	#DATA_MARK_TIME,w0	;give 1ms to then find the data mark
	rcall	setTimer
	rcall	sync2Mark		;look for the data mark
	add.b	#5,w0		;verify data mark (0xfb)
	bra	nz,rcSecHunt

	bset.w	flags,#fREAD_CRC	;don't store the data read
	mov.w	#SECTOR_LEN+2,wXferLen	;wXferLen=number of bytes to read (+2 byte CRC)
	rcall	readBuf		;read to compute CRC
	cp0.w	wCrc
	bra	z,busyExit		;success
	bra	cmdFail
	
;-------------------------------------------------------------------------------
; cmdWrite - Write sector userSec from userWrtBuf to current track
;-------------------------------------------------------------------------------
cmdWrite:	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	rcall	busyCmd		;init for a command that goes "busy"

	mov.b	#43,w0		;reduced current track
	cp.b	curTrack		;compare current track to 43
	bra	ltu,1f		;tracks 0-42, normal current
	bclr.w	LATB,#B13_WRITE_CURRENT	;else tracks 43-76, low current
	bra	chkFormat
1:	bset.w	LATB,#B13_WRITE_CURRENT	;tracks 0-42, normal current
 
chkFormat:	btst.w	userConfig,#fFORMAT	;format flag set?
	bra	nz,fmtTrack		;yes, go format
	
	rcall	findSec		;find the sector specified in userSector

; At this point, interrupts are disabled and it's been about 4us since the last
;   data pulse of the address mark. Delay 4us more and then use the data out SPI
;   to maintain timing. Each write is two bits and takes 8us. We need to wait 11
;   byte times (352us) from the end of the address mark before we start writing 
;   the six zeros preceding the data mark.

	mov.w	#21,w0		;delay 4us (64 cycles)
1:	dec.w	w0,w0
	bra	nz,1b

	mov.w	#0xffff,w1		;w1=write idle high for two bits
	mov.w	w1,SPI1BUF		;four bits of idle time (16us)
	mov.w	w1,SPI1BUF		;   to ensure fifo empty will occur
	clr.w	SR		; safe to re-enable interrupts now
	 
	mov.w	#41,w0		;delay 41*8us (328us) more
1:	btst.w	IFS0,#SPI1IF	;wait for fifo empty
	bra	z,1b
 	mov.w	w1,SPI1BUF		;two more bits of idle time (8us)
 	bclr.w	IFS0,#SPI1IF	;clear fifo empty flag
	dec.w	w0,w0		;repeat for 336us
	bra	nz,1b

; When the fifo empties, we are 2us from the start of the six zeros we
;    must write. Go ahead and turn on write enable at that time followed
;    by the data mark, sector data, and sector checksum.
		
waitWrite:	btst.w	IFS0,#SPI1IF	;wait for fifo empty
	bra	z,waitWrite
	
	bclr.w	LATB,#B9_WRITE_GATE	;turn on the write gate
	mov.w	#0xfb,w2		;write data mark sequence
	rcall	wrtMarkC7

	mov.w	#SECTOR_LEN,wXferLen	;sector length to write

wrData:	mov.b	[pUserWrt++],w0	;write 128 bytes of sector data
	rcall	writeByte
	mov.w	#USER_WRT_END,w0	;wrap buffer pointer if needed
	cp.w	pUserWrt,w0
	bra	ltu,1f		;no wrap required
	mov.w	#userWrtBuf,pUserWrt	;else, wrap to start
1:	dec.w	wXferLen,wXferLen
	bra	nz,wrData
	
; Write the two byte CRC followed by a gap byte of 0xff. Then wait for all
;   these bits to shift out before turning off the write gate.

	mov.w	wCrc,w2		;snapshot CRC into w2
	lsr.w	wCrc,#8,w0		;write MSB of CRC
	rcall	writeByte
	mov.w	w2,w0		;then LSB
	rCall	writeByte
	mov.w	#0xff,w0		;gap byte
	rCall	writeByte

	mov.w	#PIC_INTS_OFF,w0	;disable interrupts while waiting
	mov.w	w0,SR		;  for final bit to shift out
		
1:	btst.w	SPI1STAT,#SRMPT	;wait for shift register to empty
	bra	z,1b
	bset.w	LATB,#B9_WRITE_GATE	;turn off write gate
	clr.w	SR		;allow interrupts again
	bra	busyExit		;all done

;-------------------------------------------------------------------------------
; fmtTrack - Format the current track. Entered from cmdWrite which has already
;    loaded the head, set head current, etc., for a write operation.
;-------------------------------------------------------------------------------
fmtTrack:	mov.w	IC2BUF,w0		;clear out buffered index captures
	btst.w	IC2CON1,#ICBNE	;empty yet?
	bra	nz,fmtTrack		;no

1:	btst.w	IC2CON1,#ICBNE	;wait for an index pulse	
	bra	z,1b

	mov.w	#0xffff,w1		;w1=write idle high for two bits
	mov.w	w1,SPI1BUF		;four bits of idle time (16us)
	mov.w	w1,SPI1BUF		;   to ensure fifo empty will occur	
	bclr.w	IFS0,#SPI1IF	;clear fifo empty flag
	bclr.w	LATB,#B9_WRITE_GATE	;turn on write gate

	mov.w	#0xff,w0		;write 40 bytes of 0xff
	mov.w	#40,wXferLen
	rcall	writeGap

	mov.w	#0xfc,w2		;write the index mark
	rcall	wrtMarkD7

	mov.w	#0xff,w0		;write 26 byte of 0xff
	mov.w	#26,wXferLen
	rcall	writeGap

; fmtSector - This loop repeats 26 times

fmtSector:	mov.w	#1,w4		;w4=sector number
	
fmtSecLoop:	mov.w	#0xfe,w2		;write the address mark
	rcall	wrtMarkC7
	
	mov.w	curTrack,w0		;write track number
	rcall	writeByte

	mov.w	#0,w0		;unused 0
	rcall	writeByte

	mov.w	w4,w0		;sector number
	rcall	writeByte

	mov.w	#0,w0		;unused zero
	rcall	writeByte

	mov.w	wCrc,w2		;snapshot CRC into w2
	lsr.w	wCrc,#8,w0		;write MSB of CRC
	rcall	writeByte
	mov.w	w2,w0		;then LSB
	rCall	writeByte

	mov.w	#0xff,w0		;write 11 bytes of 0xff
	mov.w	#11,wXferLen
	rcall	writeGap

	mov.w	#0xfb,w2		;write data mark
	rcall	wrtMarkC7

	mov.w	#SECTOR_LEN,wXferLen	;sector length to write
	
fmtData:	mov.b	[pUserWrt++],w0	;write 128 bytes of sector data
	rcall	writeByte

	mov.w	#USER_WRT_END,w0	;wrap buffer pointer if needed
	cp.w	pUserWrt,w0
	bra	ltu,1f		;no wrap required
	mov.w	#userWrtBuf,pUserWrt	;else, wrap to start

1:	dec.w	wXferLen,wXferLen
	bra	nz,fmtData

	mov.w	wCrc,w2		;snapshot CRC into w2
	lsr.w	wCrc,#8,w0		;write MSB of CRC
	rcall	writeByte
	mov.w	w2,w0		;then LSB
	rCall	writeByte

	mov.w	#0xff,w0		;write 27 bytes of 0xff
	mov.w	#27,wXferLen
	rcall	writeGap
	
	inc.w	w4,w4		;increment sector number
	cp.w	w4,#26		;repeat for 26 sectors
	bra	leu,fmtSecLoop

; Write 0xff until the next index mark detected, thrn turn off the write gate

fmtFinish:	mov.w	#0xff,w0		;write another 0xff
	call	writeByte
	btst.w	IC2CON1,#ICBNE	;index pulse occur?	
	bra	z,fmtFinish		;no, do it again

	bset.w	LATB,#B9_WRITE_GATE	;turn off write gate
	bra	busyExit

;-------------------------------------------------------------------------------
; cmdSeek0 - Seek to track 0
;-------------------------------------------------------------------------------
cmdSeek0:	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	rcall	busyCmd		;init for a command that goes "busy"
 	mov.w	#80,w1		;w1=max number of steps to try
	bset.w	LATB,#B7_STEP_DIRECTION   ; stepping out
	clr.w	curTrack		;will be on track 0 when done

s0Loop:	btst.w	PORTD,#D2_TRACK0	;reached track 0?
	bra	nz,seekSettle	;yes
	
	mov.w	#STEP_CHAR,w0	;data byte that when serially transmitted
	mov.w	w0,U2TXREG		;  generates a 2us step pulse
	mov.w	ptStep3712,w0	;step time from parameter table
	rcall	waitTimer		;wait on the step timeout
	dec.w	w1,w1		;maximum step counter
	bra	nz,s0Loop

	bra	cmdFail		;restore failed if here

seekSettle:	mov.w	#HEAD_SETTLE,w0	;wait head settle time
	rcall	waitTimer
	bra	busyExit
	
;-------------------------------------------------------------------------------
; cmdSeek - Seek to track in userTrack
;   Reads the address mark of the current track to establish current
;   position,  steps to the new track and then reads the address mark
;   at the new track to verify.
;-------------------------------------------------------------------------------
cmdSeek:	mov.w	#STACK_ADDR,w15	;re-init stack pointer
	rcall	busyCmd		;init for a command that goes "busy"

	mov.w	curTrack,w0		;init track in the address mark
	mov.w	w0,amTrack		;  in case formatting
	btst.w	userConfig,#fFORMAT	;formatting?
	bra	nz,1f		;yes, skip reading an address mark
	rcall	rdAddrMark		;read an address mark at the start track

1:	mov.b	amTrack,WREG	;compute w0=destination-current
	sub.b	userTrack,WREG		
	bra	z,busyExit		;same track, exit

	bclr.w	LATB,#B7_STEP_DIRECTION	;assume stepping in
	bra	gtu,1f		;yes, stepping in
	bset.w	LATB,#B7_STEP_DIRECTION ;else, stepping out
	neg.b	WREG0,WREG		;w0=positive number of steps
1:	ze	w0,w1		;w1=number of steps to make			

seekLoop:	mov.w	#STEP_CHAR,w0	;data byte that when serially transmitted
	mov.w	w0,U2TXREG		;  generates a 2us step pulse
	mov.w	ptStep3712,w0	;step time from parameter table
	rcall	waitTimer		;wait on the step timeout
	dec.w	w1,w1		;step counter
	bra	nz,seekLoop

; We've stepped to the destination track - verify the track number

	mov.w	#HEAD_SETTLE,w0	;wait head settle time
	rcall	waitTimer
	mov.w	userTrack,w0	;update current track
	mov.w	w0,curTrack
	btst.w	userConfig,#fFORMAT	;formatting?
	bra	nz,busyExit		;yes, skip reading destination mark

	rcall	rdAddrMark		;read an address mark at destination
	mov.b	amTrack,WREG	;at right track?
	cp.b	userTrack
	bra	z,busyExit		;success
	bra	cmdFail		;else, exit with failure

;-----------------------------------------------------------------------------
; busyCmd - Load the head (start motor) if required and start the
;    command and motor timeout using OC2
; 
;    If the head is already loaded and a different drive has not been 
;    selected, then control returns to the caller. Otherwise, the head
;    is loaded and one rotation at the right speed is verified. This
;    validates the motor start-up for 5.25" drives and is longer than
;    the head load/settle time for 8" drives.
;-----------------------------------------------------------------------------
busyCmd:	bset.w	userStatus,#sBUSY	;controller is now busy
	clr.w	SR		;set interrupt level to zero (no interrupt)
	
	bclr.w	IEC0,#OC2IE		;configure motor and cmd timeout
	mov.w	#LONG_TIC,w0	;LONG_TIC is 100ms
	add.w	OC2TMR,WREG
	mov.w	w0,OC2R
	mov.w	#MOTOR_TIMEOUT,w0	;set motor and command timeout
	mov.w	w0,longTimer
	bclr.w	IFS0,#OC2IF		;clear the OC2 interrupt flag
	bset.w	IEC0,#OC2IE		;enable the OC2 interrupt

	btst.w	LATB,#B11_HEAD_LOAD	;head loaded already?
	bra	nz,verifyIdx	;no
	btst.w	flags,#fNEW_DRIVE	;new (different) drive selected?
	bra	nz,verifyIdx	;yes
	
	return			;else, head already loaded and ready

; verifyIndex - Verify one revolution at the right speed
 
verifyIdx:	mov.w	#STARTUP_TIME,w0	;include extra 0.5s for motor startup
	add.w	longTimer
	
	bclr.w	LATB,#B11_HEAD_LOAD	;assert head load (or motor on)
	bclr.w	flags,#fNEW_DRIVE
	
1:	mov.w	IC2BUF,w0		;clear out buffered index captures
	btst.w	IC2CON1,#ICBNE	;empty yet?
	bra	nz,1b		;no

1:	btst.w	IC2CON1,#ICBNE	;get an initial index capture in w2	
	bra	z,1b
	mov.w	IC2BUF,w2		;w2=counter at index pulse

waitIndex:	btst.w	IC2CON1,#ICBNE	;wait for next index pulse	
	bra	z,waitIndex

	mov.w	IC2BUF,w1		;w1=counter at new pulse
	sub.w	w1,w2,w0		;w0=time between pulses			
	mov.w	w1,w2		;w2=time of this capture

	mov.w	#INDEX_MIN,w1	;too short (fast)?
	cp.w	w0,w1
	bra	ltu,waitIndex	;yes, too short
	
	mov.w	#INDEX_MAX,w1	;too long (slow)?
	cp.w	w0,w1
	bra	gtu,waitIndex	;yes, too long
	
	return

;-------------------------------------------------------------------------------
; cmdFail, busyExit - Exit busy state to idle state
;-------------------------------------------------------------------------------
cmdFail:	bset.w	userStatus,#sCRC_ERROR	;indicate command failure
				;fall through to exit busy state	
busyExit:	bclr.w	userStatus,#sBUSY
	clr.w	flags
	mov.w	userStatus,w0
	mov.b	WREG,PMDOUT1L	;update status to 8080
	bra	idle

;-----------------------------------------------------------------------------
; findSec - Find and read address mark that matches userSec
;    If the correct address mark is not found, the longTimer command timeout
;    interrupt will abort this routine.
;
; Clobbers w0-w3
;-----------------------------------------------------------------------------
findSec:	rcall	rdAddrMark		;read next address mark
	mov.w	userSec,w0		;w0=sector to look for
	cp.b	amSector		;match address mark?
	bra	nz,findSec		;no
	
	return

;-----------------------------------------------------------------------------
; rdAddrMark - Read the next address mark into addrMark in RAM
;    If no address mark is found, the longTimer command timeout interrupt
;    will abort this routine.
;
; Clobbers w0-w3
;-----------------------------------------------------------------------------
rdAddrMark:	clr.w	SR		;allow interrupts
	mov.w	#0xffff,w0		;give max time to find address mark
	rcall	setTimer
	rcall	sync2Mark		;sync to an address mark or data mark
	add.b	#2,w0		;test for address mark (0xfe)
	bra	nz,rdAddrMark	;not an address mark

	mov.w	#ADDR_MARK_LEN,wXferLen	;wXferLen=length to read
	mov.w	#addrMark,pDiskBuf	;pDiskBuf->destination
	mov.w	#PIC_INTS_OFF,w0	;disable interrupts (mainly for writes)
	mov.w	w0,SR
	rcall	readBuf		;read the address mark
	cp0.w	wCrc		;CRC should be zero here
	bra	nz,rdAddrMark	;error, try again

	return

;-----------------------------------------------------------------------------
; sync2Mark - Sync to a sector mark or a data mark 
;    The mark type byte is returned in w0. The mark is found by verifying
;    leading zeros prior to the mark and then looking for the 0xc7 missing
;    clock pattern. On exit, the CRC has been restarted and data timing is
;    on a byte boundary.
;
;    Monitors the OC1 interrupt flag for timeout. Zero is returned in w0
;    (an invalid mark byte) if timeout occurs.
;
; Clobbers w0-w3
;-----------------------------------------------------------------------------
sync2Mark:	mov.w	IC1BUF,w0		;clear out old data captures
	btst.w	IC1CON1,#ICBNE	;empty yet?
	bra	nz,sync2Mark	;no
	
smRestart:	btst.w	IFS0,#OC1IF		;timeout yet?
	bra	z,1f		;no
	clr.w	w0		;else, return invalid mark type
	return
	
1:	mov.w	#16,w2		;search for 16 zeros (4us periods)
	rcall	sequence4
	bra	nz,smRestart

; Valid length of zeros detected, look for a 0xC7 clock pattern with
;    a 0xF8-0xFF data pattern. This is detected by looking for the
;    following pulse timing after a sequence of zeros:
;
;    Clock (C7)  1   1   0   0   0   1   1   1
;    Data (Fx)     1   1   1   1   1   x   x   x
;    Time (us)    2 2 2  4   4   4  2

1:	mov.w	#1,w2		;now wait for a 2us period
	rcall	sequence2
	bra	nz,1b
	
	mov.w	#2,w2		;require two more 2us times
	rcall	sequence2
	bra	nz,smRestart

	mov.w	#3,w2		;require three 4us times
	rcall	sequence4
	bra	nz,smRestart

	mov.w	#1,w2		;require one more 2us time
	rcall	sequence2
	bra	nz,smRestart

	mov.w	#3,w2		;read next three bits as data
	rcall	rbLoop		;inside readByte routine
	ior.b	#0xf8,w0		;form 0xf8 - 0xff

	mov.w	#0xffff,wCrc	;preset CRC to all 1's
	UpdateCrc			;include the mark in the CRC (inline macro)
	return
 
; sequence 2 - Validate a sequence of pulses separated by 2us. The number of pulses
;   required is specified in w2. Returns zero true if success, non-zero otherwise.

sequence2:	mov.w	#DATA_OR_CLOCK-1,w3	;w3=2us vs 4us threshold

s2NxtPulse:	btst.w	IC1CON1,#ICBNE	;wait for a data pulse	
	bra	z,s2NxtPulse

	mov.w	IC1BUF,w1		;w1=data edge capture time
	sub.w	w1,wPrevData,w0	;w0=time between data pulses
	mov.w	w1,wPrevData
	cp.w	w0,w3		;2us or 4us?
	bra	gtu,s2Exit		;4us - abort
	dec.w	w2,w2		;decrement pulse count
	bra	nz,s2NxtPulse	
s2Exit:	return

; sequence 4 - Validate a sequence of pulses separated by 4us. The number of pulses
;   required is specified in w2. Returns zero true if success, non-zero otherwise.

sequence4:	mov.w	#DATA_OR_CLOCK,w3	;w3=2us vs 4us threshold

s4NxtPulse:	btst.w	IC1CON1,#ICBNE	;wait for a data pulse	
	bra	z,s4NxtPulse

	mov.w	IC1BUF,w1		;w1=data edge capture time
	sub.w	w1,wPrevData,w0	;w0=time between data pulses
	mov.w	w1,wPrevData
	cp.w	w0,w3		;2us or 4us?
	bra	ltu,s4Exit		;2us - abort
	dec.w	w2,w2		;decrement pulse count
	bra	nz,s4NxtPulse	
s4Exit:	return

;-----------------------------------------------------------------------------
; readBuf - Read number of bytes specified in wXferLen into RAM pointed to
;    by pDiskBuf
;
; Clobbers w0-w3
;-----------------------------------------------------------------------------
readBuf:	rcall	readByte		;get next byte
	btss.w	flags,#fREAD_CRC	;don't save in READ_CRC command
	mov.b	w0,[pDiskBuf++]
	dec.w	wXferLen,wXferLen
	bra	nz,readBuf
	return	

;-----------------------------------------------------------------------------
; readByte - Read and return the next byte from disk in W0. The CRC is
;    updated with the new byte. 
; Clobbers w0-w3
;-----------------------------------------------------------------------------
readByte:	mov.w	#8,w2		;w2=bit counter
	mov.w	#DATA_OR_CLOCK,w3	;w3=2us vs 4us threshold

rbLoop:	sl.w	dataShift		;make room for next bit

rbWait:	btst.w	IC1CON1,#ICBNE	;data pulse capture occur?	
	bra	z,rbWait		;no, keep waiting

	mov.w	IC1BUF,w1		;w1=data edge capture time
	sub.w	w1,wPrevData,w0	;w0=time between data pulses
	cp.w	w0,w3		;2us period?
	bra	geu,1f		;no
	bset.w	dataShift,#0	;yes, set data bit to 1
	bra	rbWait

1:	mov.w	w1,wPrevData	;end of normal 4us bit cell
	dec.w	w2,w2		;decrement and loop for 8 bits
	bra	nz,rbLoop
	
	mov.w	dataShift,w0	;return byte in w0
	UpdateCrc			;in-line macro
	return

;-------------------------------------------------------------------------------
; wrtMarkD7
; wrtMarkC7 - Write the mark byte passed in w2 with the 0xd7 or 0xc7 missing
;   clock pattern. This includes the six leading zeros prior to the mark byte.
;   The CRC is re-started with the mark byte.
; Clobbers w1,w3,w5
;-------------------------------------------------------------------------------
wrtMarkD7:	mov.w	#CLOCK_D7_54,w5	;write mark byte with 0xd7 clock
	mov.w	#CLOCK_D7_32,w3
	bra	1f

wrtMarkC7:	mov.w	#CLOCK_C7_54,w5	;write mark byte with 0xc7 clock
	mov.w	#CLOCK_C7_32,w3

1:	mov.w	#0x00,w0		;write 6 bytes of 0x00
	mov.w	#6,wXferLen
	rcall	writeGap

	mov.w	w5,clock54		;set missing clock pattern
	mov.w	w3,clock32

	mov.w	#0xffff,wCrc	;init CRC
	mov.w	w2,w0		;write the mark byte from w4
	rcall	writeByte

	mov.w	#NORMAL_CLOCK,w0	;restore normal clock
	mov.w	w0,clock54
	mov.w	w0,clock32
	return

;-------------------------------------------------------------------------------
; writeGap - Write the byte in w0 to the disk wXferLen consecutive times
; Clobbers w1,wXferLEn
;-------------------------------------------------------------------------------
writeGap:	rcall	writeByte
	dec.w	wXferLen,wXferLen
	bra	nz,writeGap
	return

;-------------------------------------------------------------------------------
; writeByte - Write the byte in w0 to the disk
; Clobbers w1
;-------------------------------------------------------------------------------
writeByte:	btst.w	IFS0,#SPI1IF	;fifo empty? (previous 8 bits done)	
	bra	z,writeByte		;no

; Bits 7-6

	mov.w	#NORMAL_CLOCK,w1	;w1=clock pattern
	btsc.w	w0,#7		;bit 7
	bclr.w	w1,#11		;pulse at 2us for a one
	btsc.w	w0,#6		;bit 6
	bclr.w	w1,#3		;pulse at 2us for a one
	mov.w	w1,SPI1BUF		;send bits 7 and 6

; Bits 5-4

	mov.w	clock54,w1		;w1=clock pattern
	btsc.w	w0,#5		;bit 5
	bclr.w	w1,#11		;pulse at 2us for a one
	btsc.w	w0,#4		;bit 4
	bclr.w	w1,#3		;pulse at 2us for a one
	mov.w	w1,SPI1BUF		;send bits 5 and 4

; Bits 3-2

	mov.w	clock32,w1		;w1=clock pattern
	btsc.w	w0,#3		;bit 3
	bclr.w	w1,#11		;pulse at 2us for a one
	btsc.w	w0,#2		;bit 2
	bclr.w	w1,#3		;pulse at 2us for a one
	mov.w	w1,SPI1BUF		;send bits 3 and 2

; Bits 1-0

	mov.w	#NORMAL_CLOCK,w1	;w1=clock pattern
	btsc.w	w0,#1		;bit 1
	bclr.w	w1,#11		;pulse at 2us for a one
	btsc.w	w0,#0		;bit 0
	bclr.w	w1,#3		;pulse at 2us for a one
	mov.w	w1,SPI1BUF		;send bits 1 and 0

	bclr.w	IFS0,#SPI1IF	;clear fifo empty flag
	UpdateCrc			;inline macro
	return

;-------------------------------------------------------------------------------
; startTimer - Start timer and return
; waitTimer - Start timer and wait for expiration before returning
;
;    Timer routines use OC1. Duration is passed as word in w0 with LSbit
;    of 4us. The OC1IF bit of IFS0 signals completion. 
;-------------------------------------------------------------------------------
setTimer:	add.w	OC1TMR,WREG		;add timeout to current time
	mov.w	w0,OC1R		;store as new OC1 compare value
	bclr.w	IFS0,#OC1IF		;clear OC1 interrupt flag	
	return
	
waitTimer:	add.w	OC1TMR,WREG		;add timeout to current time
	mov.w	w0,OC1R		;store as new OC1 compare value
	bclr.w	IFS0,#OC1IF		;clear OC1 interrupt flag

1:	btst.w	IFS0,#OC1IF		;wait for timeout to occur
	bra	z,1b
	
	return

;-------------------------------------------------------------------------------
; oc2Int
;-------------------------------------------------------------------------------
oc2Int: 	bclr.w	IFS0,#OC2IF		;clear OC2 interrupt flag
	mov.w	#LONG_TIC,w0	;compute next 100ms wakeup
	add.w	OC2R
	
	dec.w	longTimer		;timer expire?
	bra	z,oc2MotOff		;yes, turn off motor

	btst.w	userStatus,#sBUSY	;still busy with a command?
	bra	z,1f		;no, just exit
	btst.w	LATB,#B9_WRITE_GATE	;in the middle of a write?
	bra	z,1f		;yes, ignore since command will finish
		
	mov.w	#CMD_ABORT_TIME,w0	;time to abort a hung command?
	cp.w	longTimer
	bra	z,cmdFail		;yes
	
1:	pop.w	w0		;else, just exit
	retfie

oc2MotOff:	bset.w	LATB,#B11_HEAD_LOAD	;unload head (motor off)
	bclr.w	IEC0,#OC2IE		;disable OC2 interrupt
	pop.w	w0
	retfie

	.end
