;***************************************************************************************
;
;  COMMON.S
;    Common data and startup code. Includes module version table.
;
;  Version History
;    1.0   11/01/14  M. Douglas	
;	- Original		
;    1.1   02/23/16  M. Douglas
;	- Update to support user selection of step rates and index offset
;    1.2   04/22/20  M. Douglas
;	- Update version table for new release. Add TMR2 interrupt vector,
;	  though nothing uses it yet.
;    1.3   06/19/20  M. Douglas
;	- Add step time parameter for the new (HD) high-density drive type
;	- Default soft5as8 to 3ms step time instead of 10.5ms
;	- Update version table for the new release
;    1.4   01/17/21  M. Douglas
;	- Update version table for the new release
;    1.5   06/11/21  M. Douglas
;	- Update version table for the new release
;    1.6   06/27/21  M. Douglas
;	- Update version table for the new release
;    1.7   09/05/21  M. Douglas
;	- Update for FD3712 drive type.
;    1.8   01/13/26  M. Douglas
;	- Add a 3/4 second delay before starting selected disk drive code to
;	  allow the S-100 bus to stabilize. This fixed a problem someone with
;	  a Z80 CPU was having.
;	- The __Default_Interrupt vector in this module has been enabled.
;	  Previously the linker-generated ISR was used.
;	- The NormalMode entry point in this module sets the SPLIM register
;	  (stack limit) to ensure SPLIM agrees with the memory map of the 
;	  currently programmed version of the application. Previously, SPLIM
;	  was set only by the bootloader, which in turn, is not updated when
;	  a user does a firmware update.
;
;***************************************************************************************
	.include 	"common.inc" 

;----------------------------------------------------------------------------------
;  Parameter Table
;     Non-volatile parametes are saved in a 512 word block in flash memory at the
;     fixed range of 0x13800-0x13BFF. During initialization, parameters are read
;     from this area into ParamTable in RAM. The constant table ParamDefaults
;     contains the default values for the table in the event the table needs to
;     be restored to its defaults.
;----------------------------------------------------------------------------------
	.equ	PARAM_VERSION,2	   ;change if table layout changes

; ParamTable - ram copy of parameter table

	.global	ptBaudRate,ptIndexCal,ptStepT8,ptStepT58,ptStepT5,ptStepTHD
	.global	ptStep3712,ptStep3812
	.section *,bss,near
ParamTable:
ptVersion:	.space	2	;table version
ptBaudRate:	.space	2	;serial drive baud rate
ptIndexCal:	.space	2	;Shugart to Pertec index timing adjustment
ptStepT8:	.space	2	;step time for 8" drive (drive type 1)
ptStepT58:	.space	2	;step time for 5.25" as 8" (drive type 4)
ptStepT5:	.space	2	;step time for 5.25" drive (drive type 3)
ptStepTHD:	.space	2	;step time for high density drive (type 5)
ptStep3712:	.space	2	;step time for FD3712 drive
ptStep3812:	.space	2	;step time for FD3812 drive
ptCheckSum:	.space	2	;checksum
	.equ	PARAM_LENGTH,($-ParamTable)/2-1  ;length in words without checksum

; ParamDefault - default values for the parameter table

	.text
ParamDefault:
pdVersion:	.word	PARAM_VERSION	;used to detect an incompatible table version
pdBaudRate:	.word	BAUD_403200		;default baud rate 403.2K
pdIndexCal:	.word	INDEX_360		;360us Shugart to Pertec index timing offset
pdStepT8:	.word	STEP_TIME10		;Altair 10.5ms step time
pdStepT58:	.word	STEP_TIME3		;5.25" HD floppy, 3ms step
pdStepT5:	.word	STEP_TIME48		;Altair minidisk 48ms step
pdStepTHD:	.word	STEP_TIME3		;High density drive 3ms step
pdStep3712:	.word	STEP_TIME8		;Shugart 8ms step time
pdStep3812:	.word	STEP_TIME8		;Shugart 8ms step time

sParamsReset: .asciz	"\r\n\n*** Parameters Reset to Default Values ***\r\n"

;----------------------------------------------------------------------------------
;  Module Version Table
;----------------------------------------------------------------------------------
	.global	ModuleTable
	.text

; MASTER release record
masterName:	.asciz	"MASTER RELEASE "
masterVer:	.asciz	"    1.8"
masterDate:	.asciz	"   01/13/2026"

; COMMON module
mod1Name:	.asciz	"Common         "	;module name (15 chars)
mod1Ver:	.asciz	"    1.8"		;version string (7 chars)
mod1Date:	.asciz	"   01/13/2026"	;version date (13 chars)

; MONITOR module
mod2Name:	.asciz	"Monitor        "
mod2Ver:	.asciz	"    1.5"
mod2Date:	.asciz	"   01/13/2026"

; SHUGART-80X module
mod3Name:	.asciz	"Shugart-80X    "
mod3Ver:	.asciz	"    1.3"
mod3Date:	.asciz	"   06/19/2020"

; ALTAIR-8IN module
mod4Name:	.asciz	"Altair-8in     "
mod4Ver:	.asciz	"    1.2"
mod4Date:	.asciz	"   06/19/2020"

; SHUGART-400 module
mod5Name:	.asciz	"Shugart-400    "
mod5Ver:	.asciz	"    1.3"
mod5Date:	.asciz	"   06/19/2020"

; ALTAIR_MINIDISK 8 module
mod6Name:	.asciz	"Altair Mindisk "
mod6Ver:	.asciz	"    1.2"
mod6Date:	.asciz	"   06/19/2020"

; SOFT5-AS-8 module
mod7Name:	.asciz	"Soft 5.25 as 8 "
mod7Ver:	.asciz	"    1.5"
mod7Date:	.asciz	"   06/27/2021"

; SERIAL DRIVE
mod8Name:	.asciz	"Serial Drive   "
mod8Ver:	.asciz	"    1.2"
mod8Date:	.asciz	"   06/19/2020"

; HDFLOPPY (High Density) DRIVE
mod9Name:	.asciz	"1.5Mb Floppy   "
mod9Ver:	.asciz	"    1.1"
mod9Date:	.asciz	"   07/12/2020"

; FD3712
mod10Name:	.asciz	"iCOM FD3712(SD)"
mod10Ver:	.asciz	"    1.2"
mod10Date:	.asciz	"   01/13/2026"

; FD3812
mod11Name:	.asciz	"iCOM FD3812(DD)"
mod11Ver:	.asciz	"    1.0"
mod11Date:	.asciz	"    TBD      "

;  Module Table

ModuleTable: .word	masterName,masterVer,masterDate
	.word	mod1Name,mod1Ver,mod1Date
	.word	mod2Name,mod2Ver,mod2Date
	.word	mod3Name,mod3Ver,mod3Date
	.word	mod4Name,mod4Ver,mod4Date
	.word	mod5Name,mod5Ver,mod5Date
	.word	mod6Name,mod6Ver,mod6Date
	.word	mod7Name,mod7Ver,mod7Date
	.word	mod8Name,mod8Ver,mod8Date
	.word	mod9Name,mod9Ver,mod9Date
	.word	mod10Name,mod10Ver,mod10Date
	.word	mod11Name,mod11Ver,mod11Date

	.word	0		;end of table

;----------------------------------------------------------------------------------
;   Common variables 
;----------------------------------------------------------------------------------
	.section *,bss,near
	.global	DriveType
DriveType:	.space	2		;value of drive type switches

;----------------------------------------------------------------------------------
;  User Interrupt vectors
;     PIC interrupts are hard vectored to entry points later in this file. Each of these
;     routines immediately jumps to the corresponding ISR specified in the vectors defined
;     immediately below.
;----------------------------------------------------------------------------------
	.global	PMPInterrupt,OC1Interrupt,OC2Interrupt,OC3Interrupt
	.global	IC3Interrupt,CNInterrupt,T1Interrupt,T2Interrupt	
PMPInterrupt:  .space	2		;parallel port interrupt (I/O from 8080)
OC1Interrupt:  .space	2		;Output Compare 1 interrupt
OC2Interrupt:  .space	2		;Output Compare 2 interrupt
OC3Interrupt:  .space	2		;Output Compare 3 interrupt
IC3Interrupt:  .space	2		;Input Capture 3 interrupt
CNInterrupt:   .space	2		;Change Notification interrupt
T1Interrupt:   .space	2		;Timer 1 interrupt
T2Interrupt:   .space	2		;Timer 2 interrupt


;****************************************************************************************
;  
;   CommonInit, NormalMode, MonitorMode - entry vectors start at address 0x200. 
;       THESE ENTRY VECTORS MUST REMAIN AT THIS LOCATION AND IN THIS ORDER.
;
;****************************************************************************************
	.section *,address(0x200),code
	.global  CommonInit,NormalMode, MonitorMode
CommonInit:  bra	commonInit		;additional common hardware initialization	
NormalMode:  bra	normalEntry		;normal FDC operation
MonitorMode: bra	MonitorEntry	;monitor access via serial port

	.text
;------------------------------------------------------------------------------------------
;  commonInit - Common hardware and software initialization
;      Initialization common to all modules but not done in the bootloader is
;      performed here. This entry is called immediately after the bootloader does 
;      its own HW initialization.
;------------------------------------------------------------------------------------------
;  Convert the drive type switch settings to a four bit value in DriveType. The option switch
;  bits are scattered across a few different ports and are collected together here.

commonInit: clr.w	w0		;form drive type in w0
	mov.w	#3,w1		;w1=bit number
	btst.w	PORTC,#C14_DRV_TYPE3_SW	;drive type bit 3
	bsw.z	w0,w1		;copy to w0 bit 3

	mov.w	#2,w1		;w1=bit number
	btst.w	PORTG,#G2_DRV_TYPE2_SW	;drive type bit 2
	bsw.z	w0,w1		;copy to w0 bit 2

	mov.w	#1,w1		;w1=bit number
	btst.w	PORTG,#G3_DRV_TYPE1_SW	;drive type bit 1
	bsw.z	w0,w1		;copy to w0 bit 1

	mov.w	#0,w1		;w1=bit number
	btst.w	PORTF,#F7_DRV_TYPE0_SW	;drive type bit 0
	bsw.z	w0,w1		;copy to w0 bit 0

	mov.w	w0,DriveType	;save result

; Load parameter table from flash memory. If the table version is different than
;   expected or if the checksum is wrong, re-write the table with default values.

	rcall	readParams		;read the param table
	bra	z,paramsGood	;table is good
	mov.w	#tbloffset(sParamsReset),w0  ;display "parameters reset" message
	rcall	DispString		;DispString in monitor module
1:	btst.w	U1STA,#TRMT		;wait till transmission finished
	bra	z,1b		;   before we continue
	rcall	ResetParams		;load defaults into RAM table
	rcall	WriteParams		;write the RAM table to flash	

paramsGood:	
	
;  Turn on PINTE pull down for the prototype board as this input is floating

;	bset.w	CNPD5,#CN68PDE	;PINTE pull-down on

	return			;return to bootloader

;------------------------------------------------------------------------------------------
;  normalEntry
;      After a 3/4 second delay to make sure the S-100 bus is stable, Jump
;      to the proper routine based on drive type from the option switches.
;      Note: LSB of minidisk and 8" serial drives MUST BE 0 and 1 respectively.
;------------------------------------------------------------------------------------------

normalEntry:
	mov.w	#STACK_ADDR,w15	;init stack pointer
	mov.w	#STACK_END,w0	;stack pointer limit
	mov.w	w0,SPLIM	
	
	mov.w	#61,w1		;delay about 0.75 second
delay:	dec.w	w0,w0		;12.3ms per 65536
	bra	nz,delay
	dec.w	w1,w1
	bra	nz,delay

	mov.w	DriveType,w0	;w0 = drive type 0-15
	bra	w0		;jump into the table
	bra	Altair8		;00 original Altair 8" floppy drive
	bra	Shugart8		;01 8" Shugart 80x drives as Altair 8"
	bra	AltairMini		;02 original Altair mindisk drive
	bra	Shugart5		;03 5.25" Shugart 400 drives as minidisk
	bra	Soft5as8		;04 5.25" Teac 55GFR as Altair 8"
	bra	HdFloppy		;05 Teac 55GFR with 1.5Mb
	bra	SerialDrive		;06 Serial port 5.25" (mindisk)
	bra	SerialDrive		;07 Serial port 8" drive
	bra	FD3712		;08 iCOM 3712 (single density)
	bra	FD3712		;09 iCOM 3812 (double density)
;	bra	FD3812		;09 iCOM 3812 (double density)
	bra	AltairMini		;0A original Altair mindisk drive
	bra	Shugart5		;0B 5.25" Shugart 400 drives as minidisk
	bra	Soft5as8		;0C 5.25" Teac 55GFR as Altair 8"
	bra	HdFloppy		;0D Teac 55GFR with 1.5Mb
	bra	SerialDrive		;0E Serial port 5.25" (mindisk)
	bra	SerialDrive		;0F Serial port 8" drive

;------------------------------------------------------------------------------------------
;  readParams
;     Read flash parameter table into ParamTable in RAM. Returns with zero false if
;     there is a checksum error or if the table version number doesn't match the version
;     expected by this code. Otherwise, the zero flag is set. Clobbers w0-w4
;------------------------------------------------------------------------------------------
readParams:	mov.w	#tblpage(PARAM_FLASH),w0    ;set page register for table in flash
	mov.w	w0,TBLPAG
	mov.w	#tbloffset(PARAM_FLASH),w1   ;w1->flash parameter table
	mov.w	#ParamTable,w2	;w2->ram parameter table
	mov.w	#PARAM_LENGTH,w3	;w3=number of words to read
	clr.w	w4		;w4=checksum

readLoop:	tblrdl.w	[w1++],w0		;w0=next word from flash table
	mov.w	w0,[w2++]		;store word in RAM
	add.w	w0,w4,w4		;update checksum
	dec.w	w3,w3
	bra	nz,readLoop

; Validate checksum

	tblrdl.w	[w1++],w0		;w0=checksum from flash table
	clr.w	TBLPAG		;restore page register to low memory
	cp.w	w0,w4		;match computed?
	bra	nz,readExit		;no, exit

; Validate table version

	mov.w	ptVersion,w0	;w0=table version read from flash
	cp.w	w0,#PARAM_VERSION	;same as expected ?
readExit:	return			;return with zero or non-zero status

;------------------------------------------------------------------------------------------
;  ResetParams
;     Copy the default parameter table to the ParamTable in RAM. Clobbers w1-w3
;------------------------------------------------------------------------------------------
	.global	ResetParams
ResetParams: mov.w	#tbloffset(ParamDefault),w1   ;w1->default parameter table
	mov.w	#ParamTable,w2	;w2->parameter table in RAM
	mov.w	#PARAM_LENGTH,w3	;w3=number of words to read

resetLoop:	tblrdl.w	[w1++],[w2++]	;move default word to RAM word
	dec.w	w3,w3
	bra	nz,readLoop
	return	

;------------------------------------------------------------------------------------------
;  WriteParams
;     Write flash parameter table from the RAM parameter table. A full 512 word block
;     is written since this is the minimum size that can erased. Clobbers w0-w4
;------------------------------------------------------------------------------------------
	.global	WriteParams
WriteParams: mov.w	#tblpage(PARAM_FLASH),w0     ;set page register for table in flash
	mov.w	w0,TBLPAG
	mov.w	#tbloffset(PARAM_FLASH),w1   ;w1->flash parameter table
	rcall	erasePage		;erase the 512 word page in flash
	mov.w	#ParamTable,w2	;w2->parameter table in RAM
	mov.w	#PARAM_LENGTH,w3	;w3=number of words to write
	clr.w	w4		;w4=checksum

; Write ParamTable into flash. 512 total words are written in 8 rows of 64 words.

	mov.w	#8,w5		;w5=8 rows of 64 words
writeLoop:	rcall	writeRow		;load and write 64 word row
	dec.w	w5,w5
	bra	nz,writeLoop
	clr.w	TBLPAG		;restore page register to low memory
	return

;---------------------------------------------------------------------------
;  erasePage - erase the 512 program word page pointed to by w1. Erase fails
;      for unknown reasons if done too soon after finishing a write operation
;      or an immediately preceding erase operation. A sequence of nops
;      fixes it for some reason. Clobbers w0.
;---------------------------------------------------------------------------
erasePage:	mov.w	#0x4042,w0		;write enable, erase a page
	mov.w	w0,NVMCON
	tblwtl.w	w1,[w1]		;set the address to erase
	disi	#5		;protect unlock sequence
	mov.b	#0x55,w0		;do the unlock
	mov.w	w0,NVMKEY
	mov.b	#0xaa,w0
	mov.w	w0,NVMKEY
	bset.w	NVMCON,#WR		;start the erase
	nop
	nop
	return

;---------------------------------------------------------------------------
;  writeRow - write the next 64 words from ParamTable in RAM into flash.
;      The checksum in w4 is written when w3 (word count) reaches zero.
;      After the checksum, 0xFFFF is written to all remaining locations.
;	w1 -> flash address to program
;	w2 -> Parameter table in RAM 
;	w3 = count of table words to write
;	w4 = checksum
;      Clobbers w0,w6
;---------------------------------------------------------------------------
writeRow:	mov.w	#0x4001,w0		;write enable, write a block
	mov.w	w0,NVMCON
	mov.w	#64,w6		;w6 = 64 program word counter
	
; writeLoop - put 0xff in high word. Low word depends on the word count in
;    w3. If w3 < 0, then we're past the end of ParamTable and filling flash
;    with 0xffff. If w3 = 0 then we need to write the checksum. Otherwise,
;    while w3 > 0, write the value from ParamTable.
	
rowLoop:	mov.w	#0xffff,w0		;force high byte to 0xff
	tblwth.b	w0,[w1]		;write lsb of high word
	cp0.w	w3		;past end of data (count negative)
	bra	lt,1f		;yes, write 0xffff already in w0
	mov.w	w4,w0		;w0=checksum
	bra	z,1f		;count = 0 means write checksum
	mov.w	[w2++],w0		;w0=next word from ParamTable
	add.w	w0,w4,w4		;update checksum
1:	tblwtl.w	w0,[w1++]		;write the next word
	dec.w	w3,w3		;decrement table word count
	dec.w	w6,w6		;decrement row word count
	bra	nz,rowLoop

;  All 64 write registers loaded, write them to program memory

	disi	#5		;protect unlock sequence
	mov.b	#0x55,w0		;do the unlock
	mov.w	w0,NVMKEY
	mov.b	#0xaa,w0
	mov.w	w0,NVMKEY
	bset.w	NVMCON,#WR		;start the write operation
	nop
	nop
	return

;****************************************************************************************
;  Interrupt entry and dispatch.
;     PIC hardware enters interrupts at the entry points below. Each of these routines
;     then immediately jumps to the corresponding ISR specified in the vectors defined
;     at the top of this file. This allows different drive emulation files to specify
;     their own ISRs.
;****************************************************************************************

;  __DefaultInterrupt is used by the linker script to specify an ISR for unused
;      interrupt vectors. It is supplied here and simply resets the processor.
	.global	__DefaultInterrupt
__DefaultInterrupt:
	reset

;--------------------------------------------------
; Enhanced Parallel Master Port Interrupts (used for IN/OUT from 8080)
	.global	__PMPInterrupt	
__PMPInterrupt: push.w	w0		;save w0
	mov.w	PMPInterrupt,w0	;get user ISR address for this interrupt
	goto	w0		;jump to user ISR

;--------------------------------------------------
; Output Compare 1 Interrupt
	.global	__OC1Interrupt	
__OC1Interrupt: push.w	w0		;save w0
	mov.w	OC1Interrupt,w0	;get user ISR address for this interrupt
	goto	w0

;--------------------------------------------------
; Output Compare 2 Interrupt
	.global	__OC2Interrupt	
__OC2Interrupt: push.w	w0		;save w0
	mov.w	OC2Interrupt,w0	;get user ISR address for this interrupt
	goto	w0

;--------------------------------------------------
; Output Compare 3 Interrupt
	.global	__OC3Interrupt	
__OC3Interrupt: push.w	w0		;save w0
	mov.w	OC3Interrupt,w0	;get user ISR address for this interrupt
	goto	w0

;--------------------------------------------------
; Input Capture 3 Interrupt
	.global	__IC3Interrupt	
__IC3Interrupt: push.w	w0		;save w0
	mov.w	IC3Interrupt,w0	;get user ISR address for this interrupt
	goto	w0

;--------------------------------------------------
; CN Interrupt (drive ready line changed state)
	.global	__CNInterrupt	
__CNInterrupt: push.w	w0		;save w0
	mov.w	CNInterrupt,w0	;get user ISR address for this interrupt
	goto	w0

;--------------------------------------------------
; Timer 1 Interrupt
	.global	__T1Interrupt
__T1Interrupt: push.w	w0		;save w0
	mov.w	T1Interrupt,w0	;get user ISR address for this interrupt
	goto	w0

;--------------------------------------------------
; Timer 2 Interrupt
	.global	__T2Interrupt
__T2Interrupt: push.w	w0		;save w0
	mov.w	T2Interrupt,w0	;get user ISR address for this interrupt
	goto	w0
