;***************************************************************************************
;
;  BOOTLOADER.S
;    Bootstrap loader and programming executive. This module is located in high 
;    memory at 0x14000 and is never erased. The RESET instruction jumps into
;    this code at the fixed address of 0x14000. The program does some minimal
;    hardware-specific initialization and then checks the MONITOR switch to determine
;    how to proceed. If the MONITOR switch is set high, the user is requesting entry  
;    into the monitor using the serial port. The user is prompted whether they want to
;    re-program the firmware or jump to the monitor. If the MONITOR switch is not set, 
;    control passes to the main run-time entry point.
;
; ***** IMPORTANT *****
;    The address of __reset must remain 0x14000 because the firmware update code
;    forces the address of the reset vector in memory to 0x14000. Also, the .global 
;    entry "FirmUpdate" cannot move since code external to this module calls 
;    this entry point.
;
;  Version History
;    1.0	11/01/14	M. Douglas
;	- Original		
;
;***************************************************************************************
	.include 	"common.inc"  
 
;  Configuration word 1  Change GCP_OFF to GCP_ON for code protect
;     JTAG off, code protect ON, write protect off, use icsp set 1, watchdog off	
	CONFIG	__CONFIG1, JTAGEN_OFF & GCP_OFF & GWRP_OFF & ICS_PGx1 & FWDTEN_OFF

;  Configuration word 2
;     Two speed startup off, 96mhz PLL on, 4Mhz from 4Mhz to PLL, clock switching disabled,
;     oscillator output on RC15 disabled, I/O lock not permanent , primary oscillator XT with PLL	
	CONFIG	__CONFIG2, IESO_OFF & PLL96MHZ_ON & PLLDIV_NODIV & FNOSC_PRIPLL & FCKSM_CSDCMD & OSCIOFNC_ON & IOL1WAY_OFF & POSCMOD_XT

;  Configuration word 3 
;     Segment write protection is at upper end of memory, config words protected, segment
;     protection enabled, 2nd Osc input not used, lower boundary of write protect is page 80
;     which translates to program address 0x14000 (page 79 used to be safe).
	CONFIG	__CONFIG3, WPEND_WPENDMEM & WPCFG_WPCFGEN & WPDIS_WPEN & WPFP_WPFP79 & SOSCSEL_EC

; Equates

	.equ	MAX_IN_LEN,7	;maximum number of characters to read
	.equ	TIMEOUT_1,31	;1 second timeout = 1/32.768ms
	.equ	CR,13		;ascii carriage return
	.equ	LF,10		;ascii line feed
	.equ	BS,8		;ascii back space character

;  Variable space. Forced to the start of RAM (0x800) so that maximum RAM is available
;     for downloading a new program image. Download operations clobber the RAM
;     of other modules.

	.section *,address(0x800),bss
inbuf:	.space	MAX_IN_LEN+1	;buffer for terminal read operations
flenLsw:	.space	2		;LSW of file length
flenMsw:	.space	2		;MSW of file length
timeout:	.space	2		;counts 32.768ms tics for transfer timeout
localStack:	.space	2		;stack space grows up towards 0x900

; Program storage in RAM starts at 0x900 and runs through 0x187ff. This gives 0x17f00 bytes
;    of program that can be buffered. This equates to a max program address of 0xff55
;    (2*0x17f00)/3 that can be downloaded

	.section *,address(0x900),bss	
bufStart:	.space	2		;start of download buffer

;  RESET entry point must be 0x14000. The rest of the bootloader code follows.

	.section *,address(0x14000),code
	.global 	__reset		;linker referenced entry point
__reset:	mov.w	#STACK_ADDR,w15	;init stack pointer
	mov.w	#STACK_END,w0	;stack pointer limit
	mov.w	w0,SPLIM	

; All analog capable ports must set to digital as analog is the default..

	clr.w	ANSB
	clr.w	ANSC
	clr.w	ANSD
	clr.w	ANSF
	clr.w	ANSG

; Init data direction registers and initial output values
 
	mov.w	#0x3bcf,w0		;port b init - disk outputs set high (off)
	mov.w	w0,LATB
	mov.w	#0xc030,w0		;port b direction - A1,A0 and ICSP are inputs
	mov.w	w0,TRISB

	mov.w	#0x0000,w0		;port c init - no outputs used (most not present)
	mov.w	w0,LATC
	mov.w	#0xf000,w0		;port c direction - XTAL, option switch inputs:
	mov.w	w0,TRISC		;   drive type bit 3 and monitor mode select

	mov.w	#0x0000,w0		;port d init - interrupt output off, mostly inputs
	mov.w	w0,LATD
	mov.w	#0x0e3f,w0		;port d direction - 5 of 6 disk inputs, interrupt clear
	mov.w	w0,TRISD
	
	mov.w	#0x0000,w0		;port e init - e15-e8 not preset, e7-e0 is data bus
	mov.w	w0,LATE
	mov.w	#0x00ff,w0		;port e direction - set data bus as inputs
	mov.w	w0,TRISE
		
	mov.w	#0x0030,w0		;port f init - UART1 xmit high, drive step output high
	mov.w	w0,LATF
	mov.w	#0x008b,w0		;port f direction - f15-f8 not present, UART1 Rcv, drive
	mov.w	w0,TRISF		;  "true ready," option switch drive type b0, PINTE inputs

	mov.w	#0x0000,w0		;port g init - most of port not present
	mov.w	w0,LATG
	mov.w	#0x000c,w0		;port g direction - option switch input: drive type b2,1
	mov.w	w0,TRISG
	mov.w	#0x0001,w0		;USB Transceiver must be disabled for port G b2,3 to work
	mov.w	w0,U1CNFG2

; Turn on pull-ups for the ICSP pins and the option switch inputs.

	bset.w	CNPU1,#CN6PUE	;ICSP pull-up on
	bset.w	CNPU1,#CN7PUE	;ICSP pull-up on
	bset.w	CNPU1,#CN0PUE	;drive type switch 3 pull-up on
	bset.w	CNPU1,#CN1PUE	;monitor switch pull-up on
	bset.w	CNPU6,#CN83PUE	;drive type switch 2 pull-up on
	bset.w	CNPU6,#CN84PUE	;drive type switch 1 pull-up on

; Map UART1 to the proper pins

	mov.b	#16,w0		;UART 1 Receive data on RP16
	mov.b	WREG,RPINR18L
	mov.b	#3,w0		;UART 1 TX code 3 is on RP17
	mov.b	WREG,RPOR8H

; Initialize UART1 for 8N1 at 9600 baud

	mov.w	#0x0008,w0		;serial port off, 8N1, BRGH=1
	mov.w	w0,U1MODE
	mov.w	#416,w0		;9600 baud
	mov.w	w0,U1BRG
	bset.w	U1MODE,#UARTEN	;UART on
	bset.w	U1STA,#UTXEN	;transmitter on too

; Call CommonInit to implement common hardware and software initialization

	call	CommonInit

; See if MONITOR entry is requested (MONITOR dip switch set high).
	
	btst.w	PORTC,#C13_MONITOR_SW	;monitor entry requested?
	bra	nz,pgmMenu		;yes, go put up program option menu
	goto	NormalMode		;otherwise, enter normal processing mode

;-------------------------------------------------------------------------------------
;   pgmMenu - Display menu to give the user the option to update firmware or
;       to enter the monitor.
;-------------------------------------------------------------------------------------
;  Init timer 2 for 32.768ms wrap-around tics. This timer is used to measure
;    timeouts when programming.

pgmMenu:	mov.w	#0x0010,w0		;divide 16Mhz by 8 = 2 Mhz
	mov.w	w0,T2CON
	mov.w	#0xffff,w0		;full period of 65536 counts = 32.768ms
	mov.w	w0,PR2
	clr.w	TMR2		;start timer at zero
	bset.w	T2CON,#TON		;turn on timer
	bclr.w	IFS0,#T2IF		;make sure overflow flag is clear

;  display menu and process repsonse

mainMenu:	mov.w	#tblpage(sMenu),w0	;obtain full address of menu message
	mov.w	w0,TBLPAG
menuLoop:	mov.w	#tbloffset(sMenu),w0	;display the main menu
	rcall	dispString
	mov.w	#inbuf,w0
	rcall	inputString
	mov.b	inbuf,WREG		;w0 = menu response
	sub.b	#'0',w0		;force ascii 1-9 to binary 1-9
	cp.b	w0,#1		;enter monitor?
	bra	z,goMonitor		;yes
	cp.b	w0,#2		;update firmware?
	bra	nz,menuLoop		;no

; Update firmware selected

	rcall	FirmUpdate
	bra	mainMenu

; Enter monitor

goMonitor:	clr.w	TBLPAG		;restore reset condition of page zero
	goto	MonitorMode		;enter monitor

;-------------------------------------------------------------------------------------
;   FirmUpdate (external entry) - update firmware via serial port download
;       This entry point verifies the user wants to perform the update. If not,
;       the routine does a return. This entry can be called by programs outside
;       this file which are running in a different address range, so the TBLPAG
;       logic must be present.
;-------------------------------------------------------------------------------------
	.section *,address(0x14800),code	;force fixed address with room before
				;it so initialization code can change
				;without FirmUpdate moving.
	.global	FirmUpdate
FirmUpdate: mov.w	#tblpage(sConfirm),w0	;obtain full address of confirmation message
	mov.w	w0,TBLPAG
	mov.w	#tbloffset(sConfirm),w0	  ;display confirmation message
	rcall	dispString
	mov.w	#inbuf,w0		;read confirmation response
	rcall	inputString
	mov.b	#'y',w0		;accept 'y' or 'Y'
	cp.b	inbuf		
	bra	z,doUpdate
	mov.b	#'Y',w0
	cp.b	inbuf
	bra	z,doUpdate		;yes, do the update
	clr.w	TBLPAG		;restore TBLPAG to low memory
	return

;-------------------------------------------------------------------------------------
;   doUpdate - the user has confirmed he wants to perform the update. A new stack
;      pointer is set up to free up more RAM. Any exit after this is a RESET.
;-------------------------------------------------------------------------------------
doUpdate:	mov.w	#0xe0,w0		   ;disable all interrupts
	ior.w	SR
	mov.w	#localStack,w15	   ;use new stack pointer
	mov.w	#tbloffset(sSendFile),w0   ;prompt to send the file
	rcall	dispString	
	rcall	rcvUpdateFile	   ;receive the update file into memory
	bra	nz,abortUpdate	   ;user aborted
	mov.w	#tbloffset(sPgmming),w0    ;tell user we're programming
	rcall	dispString
	rcall	writeUpdate		   ;write the update to program memory
	mov.w	#tblpage(sComplete),w0	   ;full address of "completed" message
	mov.w	w0,TBLPAG
	mov.w	#tbloffset(sComplete),w0
	rcall	dispString
1:	btst.w	U1STA,#TRMT		   ;stay here until all characters gone
	bra	z,1b
	reset

; abortUpdate - user aborted the update process by terminating the file
;    send operation early.

abortUpdate: mov.w	#tbloffset(sAbort),w0
	rcall	dispString
1:	btst.w	U1STA,#TRMT		    ;stay here until all characters gone
	bra	z,1b
	reset

;-------------------------------------------------------------------------------------
;   rcvUpdateFile - receive the update file into memory. We start at bufStart
;      in the near data space, then switch to eds pages for the last 64K.
;	w1 = pointer into memory (32K pages)
;	w2 = msbit set or cleared as required to access near or eds space
;	w3 = eds page value
;
;-------------------------------------------------------------------------------------
;  Wait for 1st character, then receive a total of four containing the length
;     of the rest of the file. Length is four bytes big-endian.

rcvUpdateFile:
	rcall	wait1stChar		;wait for 1st character and start timeouts
	mov.b	WREG,flenMsw+1	;save MSB of the MSW of file length
	rcall	waitCharTo		;receive char with 1sec timeout
	bra	nz,rcvAbort
	mov.b	WREG,flenMsw	;LSB of the MSW of file length
	rcall	waitCharTo		;receive char with 1sec timeout
	bra	nz,rcvAbort
	mov.b	WREG,flenLsw+1	;MSB of the LSW of file length
	rcall	waitCharTo		;receive char with 1sec timeout
	bra	nz,rcvAbort
	mov.b	WREG,flenLsw	;LSB of the LSW of file length

; rcvLoop - receive data into RAM until timeout occurs	

	mov.w	#bufStart,w1	;w1 = memory pointer
	clr.w	w2		;w2 msbit set when in EDS 
	clr.w	w3		;w3 = EDS page number
rcvLoop:	rcall	waitCharTo		;receive char with 1sec timeout
	bra	nz,rcvDone		;timeout, all done
	mov.b	w0,[w1+w2]		;store the character
	inc.w	w1,w1
	bra	nn,rcvLoop		;store until bit 15 comes on
	clr.w	w1		;start w1 over at zero (32K at a time)
	bset.w	w2,#15		;set MSB to force EDS access from here on
	inc.w	w3,w3		;increment page number
	mov.w	w3,DSWPAG
	bra	rcvLoop

; rcvDone - data reception has stopped. See if we have received the data length expected.

rcvDone:	btsc.w	w3,#0		;bit 15 of LSW is in bit 0 of PAG
	bset.w	w1,#15
	lsr.w	w3,w3		;bit 1 of PAG is bit 0 of length MSW
	mov.w	#bufStart,w0	;compute final address - bufStart
	sub.w	w1,w0,w0		;w0 = LSW of result
	cp.w	flenLsw		;LSW match?
	bra	nz,rcvAbort
	subb.w	w3,#0,w0		;w0 = MSW of result		
	cp.w	flenMsw		;MSW match?
	bra	nz,rcvAbort
	bset.w	SR,#Z		;set zero status
rcvAbort:	return

;---------------------------------------------------------------------------
;  writeUpdate - The update is received and in memory. Now burn the
;     update into program memory. Registers are used as follows:
;	w1 = pointer into RAM (source of data) (32K pages)
;	w2 = msbit set or cleared as required to access near or eds space
;	w3 = eds page value
;	w4 = program memory address
;	w5 = max address to program + 1
;---------------------------------------------------------------------------
writeUpdate: sl.w	flenLsw,WREG	;convert byte length to program length
	rlc.w	flenMsw		;... pgmLen = (2* byteLen)/3
	mov.w	flenMsw,w1		;w0,w1 = byteLen * 2
	mov.w	#3,w2		;divide by 3
	repeat	#17		;divide must be done 18 times
	div.ud	w0,w2		;result will be in w0
	mov.w	w0,w5		;w5 = max address to program

; Force the first instruction (double-word instruction) to "goto 0x14000" to be
;    safe. This is the RESET location and reset should always jump into this
;    boot program.

	mov.w	#bufStart,w1	;w1 is read buffer pointer
	mov.b	#0x04,w0		;force "goto 0x14000, NOP as 1st
	mov.b	w0,[w1++]		;...two instructions
	mov.b	#0x40,w0		
	mov.b	w0,[w1++]
	clr.b	[w1++]
	clr.b	[w1++]
	clr.b	[w1++]
	mov.b	#0x01,w0		;ms word of address		
	mov.b	w0,[w1]

; The programming loop does a page erase of 512 instructions followed by 8 row
;    writes of 64 instructions to update the erased page.

	mov.w	#bufStart,w1	;w1 = memory pointer
	clr.w	w2		;w2 msbit set when in EDS 
	clr.w	w3		;w3 = EDS page number
	clr.w	w4		;w4 = program address
	clr.w	TBLPAG		;our programmed memory is all in page zero
eraseLoop:	rcall	erasePage		;erase a page of program memory at w4
	mov.w	#8,w8		;write 8 rows of 64 words per page
pgmLoop:	rcall	writeRow		;load and write 64 program word row
	cp0.w	w4		;address wrap to zero?
	bra	z,updateDone	;yes, we're done
	cp.w	w4,w5		;w5 has max address of new program
	bra	geu,updateDone	;addr >= max?
	dec.w	w8,w8		;otherwise, finish 8 rows
	bra	nz,pgmLoop
	bra	eraseLoop

updateDone: return

;---------------------------------------------------------------------------
;  writeRow - write the next 64 instructions into program memory. Assumes:
;	w1 = pointer into RAM (source of data) (32K pages)
;	w2 = msbit set or cleared as required to access near or eds space
;	w3 = eds page value
;	w4 = program memory address to program
;   Clobbers w6
;---------------------------------------------------------------------------
writeRow:	mov.w	w4,w0		;display current address from w4
	rcall	dispHex
	mov.w	#0x4001,w0		;write enable, write a block
	mov.w	w0,NVMCON
	mov.w	#64,w6		;w6 = 64 program word counter
writeLoop:	rcall	getByte		;get lsb of upper word
	tblwth.b	w0,[w4]		;write lsb of high word
	rcall	getByte		;get msb of low word
	swap.w	w0		;put in upper byte of w0
	rcall	getByte		;get lsb of low word
	tblwtl.w	w0,[w4++]		;write the low word
	dec.w	w6,w6		;repeat 64 times
	bra	nz,writeLoop

;  All 64 write registers loaded, write them to program memory

	disi	#5		;protect unlock sequence
	mov.b	#0x55,w0		;do the unlock
	mov.w	w0,NVMKEY
	mov.b	#0xaa,w0
	mov.w	w0,NVMKEY
	bset.w	NVMCON,#WR		;start the write operation
	nop
	nop
	return

;---------------------------------------------------------------------------
;  getByte - retrieve next byte from RAM for programming in w0. Assumes:
;	w1 = pointer into RAM (source of data) (32K pages)
;	w2 = msbit set or cleared as required to access near or eds space
;	w3 = eds page value
;---------------------------------------------------------------------------
getByte:	mov.b	[w1+w2],w0		;get the next byte
	inc.w	w1,w1
	bra	n,1f		;reached 32K boundary?
	return			;no

1:	clr.w	w1		;start w1 over at zero (32K at a time)
	bset.w	w2,#15		;set MSB to force EDS access from here on
	inc.w	w3,w3		;increment page number
	mov.w	w3,DSRPAG
	return

;---------------------------------------------------------------------------
;  erasePage - erase the 512 program word page pointed to by w4. Erase fails
;      for unknown reasons if done too soon after finishing a write operation
;      or an immediately preceding erase operation. A sequence of nops
;      fixes it for some reason.
;---------------------------------------------------------------------------
erasePage:	mov.w	#0x4042,w0		;write enable, erase a page
	mov.w	w0,NVMCON
	tblwtl.w	w4,[w4]		;set the address to erase
	disi	#5		;protect unlock sequence
	mov.b	#0x55,w0		;do the unlock
	mov.w	w0,NVMKEY
	mov.b	#0xaa,w0
	mov.w	w0,NVMKEY
	bset.w	NVMCON,#WR		;start the erase
	nop
	nop
	return

;---------------------------------------------------------------------------
;  dispString - display a string from PROGRAM MEMORY pointed to by w0.
;      Clobbers w0, w1
;---------------------------------------------------------------------------
dispString: tblrdl.b	[w0++],w1		;get the next byte of the message
	cp0.b	w1		;zero is end of string - exit
	bra	nz,1f		
	return
1:	btst.w	U1STA,#UTXBF	;loop until room for a character
	bra	nz,1b
	mov.w	w1,U1TXREG		;transmit the character
	bra	dispString

;---------------------------------------------------------------------------
;  dispHex
;      Writes the value passed in w0 to the screen as a four digit hex value
;      Clobbers nothing
;---------------------------------------------------------------------------
dispHex:	swap.w	w0		;put msb into lsb
	swap.b	w0		;put msn into lsn
	rcall	dispNibble
	swap.b	w0		;put lsn back into lsn
	rcall	dispNibble
	swap.w	w0		;put lsb back into lsb
	swap.b	w0		;put msn into lsn
	rcall	dispNibble
	swap.b	w0		;put lsn back into lsn
	rcall	dispNibble
	mov.b	#CR,w0
	rcall	dispChar
	return

dispNibble: push.w	w0
	and.w	#0x0f,w0		;get the nibble alone
	cp.w	w0,#0x0a		;see if a-f
	bra	ltu,1f		;ok to print
	add.w	#7,w0		;correct for making 'A' to 'Z'	
1:	add.w	#0x30,w0		;make printable
2:	btst.w	U1STA,#UTXBF	;loop until room for a character
	bra	nz,2b
	mov.w	w0,U1TXREG		;transmit the character
	pop.w	w0
	return

; dispChar - display character passed in w0

dispChar:	btst.w	U1STA,#UTXBF	;loop until room for a character
	bra	nz,dispChar
	mov.w	w0,U1TXREG		;transmit the character
	return

;---------------------------------------------------------------------------
;  inputString - input a string to the buffer pointed to by w0. Handles
;     backspace. Maximum of MAX_IN_LEN characters read. String is null terminated.
;     Clobbers w1-w3
;---------------------------------------------------------------------------
inputString: push.w	w0		;save the start of buffer address
	clr.w	w2		;received (valid) character count
waitChar:	btst.w	U1STA,#URXDA	;wait for a character
	bra	z,waitChar
	mov.w	U1RXREG,w1		;get the character
	and.w	#0x7f,w1		
	cp.b	w1,#CR		;C/R?
	bra	z,inputDone
	cp.b	w1,#BS		;back space?
	bra	z,backSpace
	mov.w	w1,w3		;save the characgter
	sub.b	#' ',w1		;make sure it's a printable character
	bra	ltu,waitChar	;not, ignore it
	cp.w	w2,#MAX_IN_LEN	;room for more?
	bra	geu,waitChar	;no, ignore it
	inc.w	w2,w2		;otherwise, count the character
	mov.w	w3,U1TXREG		;echo the character	
	mov.b	w3,[w0++]		;and store the character
	bra	waitChar

backSpace:	cp0.w	w2		;already at zero characters?
	bra	z,waitChar		;yes, do nothing
	dec.w	w2,w2		;otherwise, decrement the count
	dec.w	w0,w0		;and the buffer pointer
	mov.w	w1,U1TXREG		;echo backspace, space, backspace to do a delete
	mov.w	#' ',w1
	mov.w	w1,U1TXREG
	mov.w	#BS,w1
	mov.w	w1,U1TXREG	
	bra	waitChar

inputDone:	clr.b	[w0]		;store null terminator
	mov.w	#0x0d,w1		;echo CR and two LFs
	mov.w	w1,U1TXREG
	mov.w	#0x0a,w1	
	mov.w	w1,U1TXREG
	mov.w	w1,U1TXREG	
	pop.w	w0		;restore original buffer pointer
	return

;---------------------------------------------------------------------------
;  wait1stChar - wait for first character to be received, then start
;      a one second timeout. The character read is returned in w0. 
;---------------------------------------------------------------------------
wait1stChar: btst.w	U1STA,#URXDA	;wait for character
	bra	z,wait1stChar
	clr.w	TMR2		;re-init timer 2
	bclr.w	IFS0,#T2IF
	mov.w	#TIMEOUT_1,w0	;start one second timeout
	mov.w	w0,timeout
	mov.w	U1RXREG,w0		;return the character in w0
	return

;---------------------------------------------------------------------------
;  waitCharTo - wait for character with one second timeout. Raising the 
;      AUX L switch is the same as timeout occuring. The Z flag is returned 
;      false for timeout or AUX, otherwise, Z flag is returned true for normal
;      exit. The character read is returned in w0.
;---------------------------------------------------------------------------
waitCharTo:	btst.w	U1STA,#URXDA	;wait for character
	bra	nz,haveByte
 	btst.w	IFS0,#T2IF		;32.768ms tic?
	bra	z,waitCharTo	;no
	bclr.w	IFS0,#T2IF		;clear the tic
	dec.w	timeout		;decrement the one second timeout
	bra	nz,waitCharTo	;loop if not timed out yet
1:	bclr.w	SR,#Z		;z clear = timeout exit
	return

haveByte:	mov.w	#TIMEOUT_1,w0	;re-start the one second timeout
	mov.w	w0,timeout
	mov.w	U1RXREG,w0		;get the character
	bset.w	SR,#Z		;z set = normal exit
	return

;-------------------------------------------------------------------------------------
;   User prompts
;--------------------------------------------------------------------------------------
sMenu:	.ascii	"\r\n\n===== Altair FDC+ Bootloader Menu 1.0 =====\r\n\n"
	.ascii	"1) Enter monitor\r\n"
	.ascii	"2) Update firmware\r\n\n"
	.asciz	"Enter Choice: "

sConfirm:	.ascii	"\r\n\n===== Update Firmware =====\r\n\n"
	.ascii	"This operation updates the Altair FDC+ firmware.\r\n"
	.ascii	"Do not perform this operation unless you have been\r\n"
	.ascii	"instructed to perform the update.\r\n\n"
	.asciz	"Continue (y/n)? "

sSendFile:  .asciz	"Send the update file now (8-bit, binary mode)... "

sPgmming:	.ascii	"\r\nFile Received."
	.asciz	"\r\nWriting Program Memory...\r\n"

sComplete:	.asciz	"\r\nFirmware Update Complete.\r\n"

sAbort:	.asciz	"\r\nUpdate Aborted or File Length Error.\r\n"

	.end

