/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  Wildcard.c                                                               */
/*                                                                           */
/*  Wildcard functions for Standard Software archival utilities.             */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  11/08/91  130D.01  D. Gilson     /  Problems with wildcard_mappable.     */
/*                                   /  PLE 15330074   From Spec is <*>*     */
/*                                   /  To Spec is <temp>*>* returns erc 5.  */
/*                                   /  This is valid and worked on previous */
/*                                   /  This fix breaks build_restore_spec.  */
/*                                   /  So, build_restore_spec must also be  */
/*                                   /  fixed.                               */
/*  11/30/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#define debug

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Archive utilities that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define BuildFileSpec
#define ParseFileSpec
#define WildCardCheck

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define sdType

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
pragma Pop(List);

/* Function prototypes defined before the functions themselves are declared */

private Boolean wildcards_isomorphic(char *string1, unsigned string1_len,
                                     char *string2, unsigned string2_len);
private unsigned count_wildcards(char *string, unsigned string_len);

pragma Page(1);
/*-----------------------------------------------------------------------------
 For the purposes of the restore utilities, the "from" file specifications
 must be mappable to the "to" specifications, including any applicable
 wildcarding.  For the {Node}, [Volume] and <Directory> fields this means that
 either a) the "to" field has no wildcarding (in which case a one-to-one or a
 many-to-one mapping takes place as indicated by the absence or presence of
 wildcards in the "from" field, respectively) or b) the "from" and "to" fields
 must be isomorphic with respect to each other (so that a like-to-like mapping
 takes place).  For the filename portion of the specification, strict
 isomorphism is enforced (although it is possible to imagine other schema,
 they don't make good, intuitive sense).  Lastly, wildcards are verboten in
 the password field (very strange things happen when files are opened with
 wildcarded password specifications!). */

Boolean wildcards_mappable(sdType *sd_from, sdType *sd_to) {

   char node_from[MAX_NODE_LENGTH], node_to[MAX_NODE_LENGTH],
      volume_from[MAX_VOLUME_LENGTH], volume_to[MAX_VOLUME_LENGTH],
      directory_from[MAX_DIRECTORY_LENGTH],
      directory_to[MAX_DIRECTORY_LENGTH],
      filename_from[MAX_FILENAME_LENGTH], filename_to[MAX_FILENAME_LENGTH],
      password_from[MAX_PASSWORD_LENGTH], password_to[MAX_PASSWORD_LENGTH];
   unsigned node_from_len, node_to_len, volume_from_len, volume_to_len,
      directory_from_len, directory_to_len, filename_from_len,
      filename_to_len, password_from_len, password_to_len;

   ParseFileSpec(0, sd_from->pb, sd_from->cb, FALSE, &node_from,
                 &node_from_len, &volume_from, &volume_from_len,
                 &directory_from, &directory_from_len, &filename_from,
                 &filename_from_len, &password_from, &password_from_len,
                 FALSE, 0);
   ParseFileSpec(0, sd_to->pb, sd_to->cb, FALSE, &node_to, &node_to_len,
                 &volume_to, &volume_to_len, &directory_to, &directory_to_len,
                 &filename_to, &filename_to_len, &password_to,
                 &password_to_len, FALSE, 0);

   if (wildcards_isomorphic(sd_from->pb, sd_from->cb - password_from_len 
                           ,sd_to->pb, sd_to->cb - password_to_len))
      return(TRUE);													/* m01 */
   if (     count_wildcards(node_to, node_to_len) != 0
         && !wildcards_isomorphic(node_from, node_from_len, node_to,
                                  node_to_len))
      return(FALSE);
   if (     count_wildcards(volume_to, volume_to_len) != 0
         && !wildcards_isomorphic(volume_from, volume_from_len, volume_to,
                                  volume_to_len))
      return(FALSE);
   if (     count_wildcards(directory_to, directory_to_len) != 0
         && !wildcards_isomorphic(directory_from, directory_from_len,
                                  directory_to, directory_to_len))
      return(FALSE);
   if (!wildcards_isomorphic(filename_from, filename_from_len, filename_to,
                             filename_to_len))
      return(FALSE);
   if (     count_wildcards(password_from, password_from_len) != 0
         || count_wildcards(password_to, password_to_len) != 0)
      return(FALSE);
   return(TRUE);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Determine whether or not any wildcard characters are present in the string. */

private unsigned count_wildcards(char *string, unsigned string_len) {

   unsigned i, j;

   for (i = j = 0; i < string_len; i++)
      if (string[i] == '*' || string[i] == '?')
         j++;
   return(j);

}

/*-----------------------------------------------------------------------------
 Two strings are considered to be isomorphic with respect to each other for
 the purpose of wildcarding if there are the same number of wildcard
 characters ('*' and '?') in both strings and the derived strings that consist
 of solely the wildcard charcters are equal to each other. */

private Boolean wildcards_isomorphic(char *string1, unsigned string1_len,
                                     char *string2, unsigned string2_len) {

   unsigned i, j;

   i = j = 0;
   while (TRUE) {
      while (i < string1_len && string1[i] != '*' && string1[i] != '?')
         i++;
      while (j < string2_len && string2[j] != '*' && string2[j] != '?')
         j++;
      if (i == string1_len)
         return(j == string2_len);
      else if (j == string2_len)
         return(FALSE);
      else if (string1[i++] != string2[j++])
         return(FALSE);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Assemble a file specification that may be used to create a file on the
 destination medium for the current file from the archive medium.  Use the
 detailed {Node}[Volume]<Directory>Filename^Password information from the
 archive sentinel record and process it according to the 'File list from' and
 'File list to' wildcard specifications.  This means transferring the portions
 that match a wildcard in the 'File list from' specification to the
 corresponding locations dictated by the 'File list to' specification, keeping
 the literal characters of the 'File list to' specification in between. */

void build_restore_spec(sdType *sd_archive_spec, sdType *sd_from,
                        sdType *sd_to, sdType *sd_restore_spec) {

   char c, wildcard_length[16];
   char archive_node[MAX_NODE_LENGTH], archive_volume[MAX_VOLUME_LENGTH],
      archive_directory[MAX_DIRECTORY_LENGTH],
      archive_filename[MAX_FILENAME_LENGTH],
      archive_dir_and_file[MAX_DIRECTORY_LENGTH + 2 + MAX_FILENAME_LENGTH];
   char from_node[MAX_NODE_LENGTH], from_volume[MAX_VOLUME_LENGTH],
      from_directory[MAX_DIRECTORY_LENGTH], from_filename[MAX_FILENAME_LENGTH],
      from_dir_and_file[MAX_DIRECTORY_LENGTH + 2 + MAX_FILENAME_LENGTH];
   char to_node[MAX_NODE_LENGTH], to_volume[MAX_VOLUME_LENGTH],
      to_directory[MAX_DIRECTORY_LENGTH], to_filename[MAX_FILENAME_LENGTH],
      to_dir_and_file[MAX_DIRECTORY_LENGTH + 2 + MAX_FILENAME_LENGTH];
   char restore_node[MAX_NODE_LENGTH], restore_volume[MAX_VOLUME_LENGTH],
      restore_directory[MAX_DIRECTORY_LENGTH],
      restore_filename[MAX_FILENAME_LENGTH],
      restore_dir_and_file[MAX_DIRECTORY_LENGTH + 2 + MAX_FILENAME_LENGTH];
   unsigned archive_node_len, archive_volume_len, archive_directory_len,
      archive_filename_len,archive_dir_and_file_len;
   unsigned from_node_len, from_volume_len, from_directory_len,
      from_filename_len, from_dir_and_file_len;
   unsigned to_node_len, to_volume_len, to_directory_len, to_filename_len,
      to_dir_and_file_len;
   unsigned restore_node_len, restore_volume_len, restore_directory_len,
      restore_filename_len, restore_dir_and_file_len;
   unsigned i, j, k, n, wildcard_start[16];

   ParseFileSpec(0, sd_archive_spec->pb, sd_archive_spec->cb, FALSE,
                 archive_node, &archive_node_len, archive_volume,
                 &archive_volume_len, archive_directory,
                 &archive_directory_len, archive_filename,
                 &archive_filename_len, NULL, 0, FALSE, 0);
   ParseFileSpec(0, sd_from->pb, sd_from->cb, FALSE, from_node, &from_node_len,
                 from_volume, &from_volume_len, from_directory,
                 &from_directory_len, from_filename, &from_filename_len, NULL,
                 0, FALSE, 0);
   ParseFileSpec(0, sd_to->pb, sd_to->cb, FALSE, to_node, &to_node_len,
                 to_volume, &to_volume_len, to_directory, &to_directory_len,
                 to_filename, &to_filename_len, NULL, 0, FALSE, 0);
   if ((restore_node_len = to_node_len) != 0) {
      WildCardCheck(from_node, from_node_len, archive_node, archive_node_len,
                    &wildcard_length, &n);
      for (i = j = k = 0; k < n; i++)
         if (from_node[i] == '*' || from_node[i] == '?') {
            wildcard_start[k] = j;
            j += wildcard_length[k++];
         } else
            j++;
      for (i = j = k = 0; i < to_node_len && j < sizeof(restore_node); i++)
         if (((c = to_node[i]) == '*' || c == '?') && k < n) {
            memcpy(&restore_node[j], &archive_node[wildcard_start[k]],
                   wildcard_length[k]);
            j += wildcard_length[k++];
         } else
            restore_node[j++] = c;
      restore_node_len = j;
   }
   if ((restore_volume_len = to_volume_len) != 0) {
      WildCardCheck(from_volume, from_volume_len, archive_volume,
                    archive_volume_len, &wildcard_length, &n);
      for (i = j = k = 0; k < n; i++)
         if (from_volume[i] == '*' || from_volume[i] == '?') {
            wildcard_start[k] = j;
            j += wildcard_length[k++];
         } else
            j++;
      for (i = j = k = 0; i < to_volume_len && j < sizeof(restore_volume); i++)
         if (((c = to_volume[i]) == '*' || c == '?') && k < n) {
            memcpy(&restore_volume[j], &archive_volume[wildcard_start[k]],
                   wildcard_length[k]);
            j += wildcard_length[k++];
         } else
            restore_volume[j++] = c;
      restore_volume_len = j;
   }
   memset(&from_dir_and_file, 0, sizeof(from_dir_and_file));
   BuildFileSpec(0, &from_dir_and_file, &from_dir_and_file_len, NULL,
                 sizeof(from_dir_and_file), FALSE, NULL, 0, NULL, 0,
                 from_directory, from_directory_len, from_filename,
                 from_filename_len, FALSE, NULL, 0, FALSE, 0);
   memset(&to_dir_and_file, 0, sizeof(to_dir_and_file));
   BuildFileSpec(0, &to_dir_and_file, &to_dir_and_file_len, NULL,
                 sizeof(to_dir_and_file), FALSE, NULL, 0, NULL, 0,
                 to_directory, to_directory_len, to_filename,
                 to_filename_len, FALSE, NULL, 0, FALSE, 0);
   memset(&archive_dir_and_file, 0, sizeof(archive_dir_and_file));
   BuildFileSpec(0, &archive_dir_and_file, &archive_dir_and_file_len, NULL,
                 sizeof(archive_dir_and_file), FALSE, NULL, 0, NULL, 0,
                 archive_directory, archive_directory_len, archive_filename,
                 archive_filename_len, FALSE, NULL, 0, FALSE, 0);
   memset(&restore_dir_and_file, 0, sizeof(restore_dir_and_file));

   if (count_wildcards(from_dir_and_file, from_dir_and_file_len) != 
       count_wildcards(to_dir_and_file, to_dir_and_file_len)) {

      if ((restore_directory_len = to_directory_len) != 0) {
         WildCardCheck(from_directory, from_directory_len, archive_directory,
                       archive_directory_len, &wildcard_length, &n);
         for (i = j = k = 0; k < n; i++)
            if (from_directory[i] == '*' || from_directory[i] == '?') {
               wildcard_start[k] = j;
               j += wildcard_length[k++];
            } else
               j++;
         for (i = j = k = 0; i < to_directory_len &&
                             j < sizeof(restore_directory); i++)
            if (((c = to_directory[i]) == '*' || c == '?') && k < n) {
               memcpy(&restore_directory[j],
                      &archive_directory[wildcard_start[k]],
                      wildcard_length[k]);
               j += wildcard_length[k++];
            } else
               restore_directory[j++] = c;
         restore_directory_len = j;
      }
      
      if ((restore_filename_len = to_filename_len) != 0) {
         WildCardCheck(from_filename, from_filename_len, archive_filename,
                       archive_filename_len, &wildcard_length, &n);
         for (i = j = k = 0; k < n; i++)
            if (from_filename[i] == '*' || from_filename[i] == '?') {
               wildcard_start[k] = j;
               j += wildcard_length[k++];
            } else
               j++;
         for (i = j = k = 0; i < to_filename_len &&
                             j < sizeof(restore_filename); i++)
            if (((c = to_filename[i]) == '*' || c == '?') && k < n) {
               memcpy(&restore_filename[j],
                      &archive_filename[wildcard_start[k]],
                      wildcard_length[k]);
               j += wildcard_length[k++];
            } else
               restore_filename[j++] = c;
         restore_filename_len = j;
      }
      
   }
   else {
      if ((restore_dir_and_file_len = to_dir_and_file_len) != 0) {
         WildCardCheck(from_dir_and_file, from_dir_and_file_len
                      ,archive_dir_and_file,archive_dir_and_file_len
                      ,&wildcard_length, &n);
         for (i = j = k = 0; k < n; i++)
            if (from_dir_and_file[i] == '*' || from_dir_and_file[i] == '?') {
               wildcard_start[k] = j;
               j += wildcard_length[k++];
            } else
               j++;
         for (i = j = k = 0;    i < to_dir_and_file_len
                             && j < sizeof(restore_dir_and_file); i++)
            if (((c = to_dir_and_file[i]) == '*' || c == '?') && k < n) {
            memcpy(&restore_dir_and_file[j],
                      &archive_dir_and_file[wildcard_start[k]],
                      wildcard_length[k]);
               j += wildcard_length[k++];
            } else
               restore_dir_and_file[j++] = c;
         restore_dir_and_file_len = j;
      }
      ParseFileSpec(0, restore_dir_and_file, restore_dir_and_file_len, FALSE,
                    NULL, 0, NULL, 0, 
                    restore_directory,&restore_directory_len,
                    restore_filename, &restore_filename_len,
                    NULL,0, FALSE, 0);
   }



   memset(sd_restore_spec->pb, 0, sd_restore_spec->cb);
   BuildFileSpec(0, sd_restore_spec->pb, &sd_restore_spec->cb, NULL,
                 sd_restore_spec->cb, (restore_volume_len == 0), restore_node,
                 restore_node_len, restore_volume, restore_volume_len,
                 restore_directory, restore_directory_len, restore_filename,
                 restore_filename_len, FALSE, NULL, 0, FALSE, 0);
}
