/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  backup.c                                                                 */
/*                                                                           */
/*                                                                           */
/*  Disk volume backup for CTOS Archive utilities.                           */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  07/10/92  122K.12  D. Gilson     /  P15511893 Suppress Confirm msg should*/
/*                                      not be tied to non-interactive       */
/*  04/24/92  122H.11  D. Gilson     /  archive names under 12 chars need    */
/*                                   /  a dir spec but those over 12 don't.  */
/*                                      Make all need dir spec. P15513331    */
/*  03/20/92  122G.10  D. Gilson     /  PLE 15413900 + Cleanup               */
/*  12/18/91  130E.09  D. Gilson     /  Added multiple drive support to be   */
/*                                      able to backup to more than 1 drive. */
/*  10/03/91  130D.08  D. Gilson     /  Correct Archive.Spec to only accept  */
/*                                      [TapeSpec]0 and [TapeSpec]+ and not  */
/*                                      [tapeSpec]n when backing up.  Various*/
/*                                      ercs happen when using [tapeSpec]n   */
/*                                      like 9074 after a few files are      */
/*                                      written to the tape. PLE 15278790    */
/*  06/20/91  130A.07  P. Johansson  /  Store all passwords, even encrypted, */
/*                                      in the archive dataset.              */
/*  05/28/91  121J.06  P. Johansson  /  Terminate position string with zero  */
/*                                      so that atoi() works correctly.      */
/*  05/10/91  121J.05  P. Johansson  /  Additional summary messages.         */
/*  04/12/91  121J.04  P. Johansson  /  Bug when verification suppressed.    */
/*  03/25/91  121H.03  P. Johansson  /  Support multiple volume backup to    */
/*                                      [XXX]+ datasets on tape.             */
/*  01/30/91  121F.02  P. Johansson  /  Explicit OpenByteStream for [Vid] so */
/*                                      that video filtering works.          */
/*  01/14/91  121F.01  P. Johansson  /  Support invocation from "Backup      */
/*                                      Volume" and "Tape Backup Volume"     */
/*                                      command forms (old, 12.0 style).     */
/*  12/13/90  121E.00  P. Johansson  /  Created.                             */
/*                                      that video filtering works.          */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

/* Suppress C run-time (only CTOS functionality needed) */

pragma Off(List);
#include <stub.h>
pragma Pop(List);

/* There are no procedures in the Archive utilities that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocExch
#define BuildFileSpec
#define CloseFile
#define CSubparams
#define GetDateTime
#define GetVHB
#define NlsULCmpB
#define OpenFile
#define ParseFileSpec
#define QueryDCB
#define RgParam
#define Send
#define ULCmpB
#define Wait

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define dcb_type
#define FhbType
#define sdType
#define vhb_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern iob_type *allocate_iob(unsigned buffer_size);
extern void backup_and_or_verify_fhbs(Boolean final_close);
extern void create_archive_io_process(void);
extern void deallocate_iob(iob_type *iob);
extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern void get_archive_datasets(unsigned param_num); 		/* m09 */
extern void fetch_bit_map(void);
extern mfd_entry_type *find_mfd_entry(char directory[]);
extern void initialize_msgs(void);
extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern void log_page_header(void);
extern unsigned open_log_file(void);
extern void open_video(void);
extern unsigned open_volume(unsigned volume_index);
extern unsigned parse_parameters(param_descr_type *param_descr, unsigned n);
extern void release_bad_spots(void);
extern void release_hidden_files(void);
extern unsigned long scan_mfd(unsigned long mfd_vda, unsigned mfd_pages,
                              Boolean verify_directories);
extern sdType *standard_msg(unsigned msg_num);
extern void verify_bit_map(void);
extern void verify_directories(void);
extern void verify_free_fhbs(void);
extern void verify_mfd(void);
extern void vid_only_msg(unsigned nls_msg_index, unsigned signature,
                         sdType nls_parms[], unsigned nls_parms_len);

/* Error return codes used by this module */

#define FsErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern archive_type archive;
extern FhbType fhb;
extern filesys_type filesys;
extern journal_type journal;
extern char sbVerRun[];

/* Global variables exported by this manuscript */

path_type path;
summary_type summary;
target_type target;
vlpb_type vlpb;

/* Function prototypes defined before the functions themselves are declared */

void scan_volumes(void);
void summarize_backup(void);
void validate_parameters(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 Short and sweet, just munch through all the necessary functions for disk
 volume backup...*/

void main(void) {

   unsigned erc, i;

   open_video();		/* Before ANY [Vid] use! (so filter works) */
   initialize_msgs();		/* Use a message file if one exists */
   if ((erc = AllocExch(&target.msg_exch)) != ercOK)
      exit_with_msg(erc, 0);
   validate_parameters();	/* Check syntax/semantics of the VLPB's */
   if (!vlpb.suppress_backup)
      create_archive_io_process();	/* Parallel, for efficiency */
   for (i = 0; i < CSubparams(VOLUME_PARAM); i++) {
      log_page_header();
      archive.sequence = 0;
      memset(&summary, 0, sizeof(summary));
      memset(&path, 0, sizeof(path));
      if (open_volume(i) != ercOK)	/* Is the volume accessible? */
         continue;			/* No, just skip it */
      if (!vlpb.suppress_verify) {
         log_msg((vlpb.suppress_backup) ? NLS_VERIFYING_FHBS
                                        : NLS_ARCHIVING_FILES, 1, NULL, 0);
         fetch_bit_map();	/* Allocate memory and store the bit map */
      }
      backup_and_or_verify_fhbs(i == CSubparams(VOLUME_PARAM) - 1);
      if (!vlpb.suppress_verify) {
         release_hidden_files();/* Some files don't have FHB's! */
         release_bad_spots();	/* For old-style VHB's, only */
         verify_free_fhbs();	/* Check integrity of free FHB chain */
         verify_mfd();		/* Check consistency of MFD */
         verify_bit_map();	/* Bit map should now be OK (release memory) */
         verify_directories();	/* Use released memory for work area here */
      }
      summarize_backup();	/* Report our (intermediate) findings... */
   }
   exit_with_msg(ercOK, 0);	/* ...but eventually leave gracefully */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The parameters for the executive command form for 'Backup Volume' are
 described in the table below.  As much as possible, they are parsed by the
 generic routine 'parse_parameters' and are transferred into appropriate
 fields in a variable-length parameter block (Vlpb).  If the generic parsing is
 successful, other tests that involve dependencies between the parameters are
 applied. */

private void validate_parameters(void) {

   unsigned long current_time = 0;
   unsigned i;									/* m09 */
   char device[MAX_DEVICE_LENGTH], directory[MAX_DIRECTORY_LENGTH],
      filename[MAX_FILENAME_LENGTH], node[MAX_NODE_LENGTH],
      position[MAX_FILENAME_LENGTH], volume[MAX_VOLUME_LENGTH];
   unsigned device_len, directory_len, erc, filename_len, n_exec_param,
      node_len, password_len, position_len, spec_len, volume_len;
   Boolean floppy_default = FALSE, tape_default = FALSE;
   sdType nls_parms[] = {         NULL, 0,
                         (void *) &sbVerRun[1], sbVerRun[0]};
   static param_descr_type exec_param[] = {
      NULL, 0, 0, 0,		/* Dummy corresponds to name of command */
      NULL, 0, 0, 0,		/* Multiple volumes checked separately */
      NULL, 0, 0, 0,		/* Ditto for their corresponding passwords */
      &vlpb.backup_after, date_param, sizeof(vlpb.backup_after),
       NLS_ARCHIVE_AFTER_DATE,
      &vlpb.suppress_backup, Boolean_param, sizeof(vlpb.suppress_backup),
       NLS_SUPPRESS_ARCHIVE,
      &vlpb.suppress_verify, Boolean_param, sizeof(vlpb.suppress_verify),
       NLS_SUPPRESS_VERIFY,
/*m09 &archive.spec, ascii_param, sizeof(archive.spec), NLS_ARCHIVE_DATASET,*/
      NULL, 0, 0, 0, 											/* m09 */
      &vlpb.overwrite_OK, Boolean_param, sizeof(vlpb.overwrite_OK),
       NLS_DELETE_EXISTING,
      &journal.filespec, ascii_param, sizeof(journal.filespec), NLS_LOG_FILE,
      &vlpb.display_structures, Boolean_param, sizeof(vlpb.display_structures),
       NLS_DISPLAY_STRUCTURES,
      &vlpb.verify_write, Boolean_param, sizeof(vlpb.verify_write), NLS_VERIFY,
      &vlpb.noninteractive, Boolean_param, sizeof(vlpb.noninteractive),
       NLS_NONINTERACTIVE,
      &vlpb.suppress_confirmation_msg, Boolean_param,
       sizeof(vlpb.suppress_confirmation_msg),NLS_SUPPRESS_CONFIRMATION_MSG};
   sdType *sd_archive, sd_case;

   RgParam(0, 0, &nls_parms[0]);
   if (nls_parms[0].pb != NULL && nls_parms[0].cb != 0)
      log_msg(NLS_COMMAND_VERSION, 1, nls_parms, sizeof(nls_parms));
   RgParam(0, 1, &sd_case);
   if (sd_case.cb == 2 && ULCmpB(sd_case.pb, "BD", 2) == 0xFFFF) {
      n_exec_param = last(exec_param) + 1;	/* "Backup Volume" form */
      floppy_default = TRUE;
   } else if (sd_case.cb == 2 && ULCmpB(sd_case.pb, "BT", 2) == 0xFFFF) {
      n_exec_param = last(exec_param);		/* "Tape Backup Volume" */
      exec_param[last(exec_param) - 1] = exec_param[last(exec_param)];
      tape_default = TRUE;
   } else
      n_exec_param = last(exec_param) + 1;	/* "Volume Archive" form */
   if ((erc = parse_parameters(exec_param, n_exec_param)) != ercOK)
      exit_with_msg(erc, 0);
   scan_volumes();	/* See if all are accessible, calculate memory needs */
   if (vlpb.backup_after != 0) {
      GetDateTime(&current_time);
      if (vlpb.backup_after > current_time)
         exit_with_msg(ercSyntax, NLS_FUTURE_DATE);
   }
/*   if (CSubparams(VOLUME_ARCHIVE_DATASET_PARAM) > 1)
      exit_with_msg(ercSyntax, NLS_TOO_MANY_DATASETS); 			m09 */
   get_archive_datasets(VOLUME_ARCHIVE_DATASET_PARAM);		/*  m09 */
   if (archive.spec[archive.device_num][0] == 0) {
      sd_archive = standard_msg((floppy_default) ? NLS_DEFAULT_FLOPPY
                                                 : NLS_DEFAULT_DATASET);
      memcpy(&archive.spec[archive.device_num][1], sd_archive->pb,
             archive.spec[archive.device_num][0] = _min(sd_archive->cb,
             last(archive.spec[archive.device_num])));
   }
   erc = ParseFileSpec(0, &archive.spec[archive.device_num][1],
                       archive.spec[archive.device_num][0],
                       FALSE, node,
                       &node_len, device, &device_len, NULL, 0, position,
                       &position_len, &archive.password[archive.device_num][1],
                       &password_len,
                       FALSE, 3);
   archive.password[archive.device_num][0] = password_len;
   if (erc == ercOK) {
      archive.sequential = TRUE;
      BuildFileSpec(0, &archive.spec[archive.device_num][1], &spec_len, NULL,
                    last(archive.spec[archive.device_num]), FALSE,
                    node, node_len, device,
                    device_len, NULL, 0, NULL, 0, FALSE, NULL, 0, FALSE, 0);
      archive.spec[archive.device_num][0] = spec_len;
      archive.current_position = 0xFFFF;	/* Unknown on the first tape */
      if (position_len == 1 && position[0] == '+')
         archive.target_position = 0xFFFF;	/* Sentinel value means '+' */
      else if (position_len != 0)  { /* m08... */
         position[position_len] = 0;
         archive.target_position = atoi(position);
         if (archive.target_position != 0)
             exit_with_msg(ercSyntax, NLS_INVALID_TAPE_MARK_SPEC);
      } /* ...m08 m09... */
      for (i = 1; i < archive.device_num_last; i++) {
         erc = ParseFileSpec(0, &archive.spec[i][1],
                             archive.spec[i][0],
                             FALSE, node,
                             &node_len, device, &device_len, NULL, 0, position,
                             &position_len,
                             &archive.password[i][1],
                             &password_len,
                             FALSE, 3);
         archive.password[i][0] = password_len;
         if (erc == ercOK) {
            BuildFileSpec(0, &archive.spec[i][1], &spec_len, NULL,
                          last(archive.spec[i]), FALSE,
                          node, node_len, device,
                          device_len, NULL, 0, NULL, 0, FALSE, NULL, 0,
                          FALSE, 0);
            archive.spec[i][0] = spec_len;
         }      											  /* ...m09 */
      }
   } else {
      if (CSubparams(VOLUME_ARCHIVE_DATASET_PARAM) > 1)			/*  m09 */
         exit_with_msg(ercSyntax, NLS_TOO_MANY_DATASETS); 		/*	m09 */
      exec_param[6].type = path_param;
      if ((erc = parse_parameters(exec_param,
                                  last(exec_param) + 1)) != ercOK)
         exit_with_msg(erc, 0);
      ParseFileSpec(0, &archive.spec[archive.device_num][1],
                    archive.spec[archive.device_num][0], FALSE, node,
                    &node_len, volume, &volume_len, directory,
                    &directory_len, filename, &filename_len,
                    &archive.password[archive.device_num][1],
                    &password_len, FALSE, 0);
      if (directory_len == 0 )									/* m11 */
         exit_with_msg(ercSyntax, NLS_DIRECTORY_ERROR);			/* m11 */
      archive.password[archive.device_num][0] = password_len;
      BuildFileSpec(0, &archive.spec[archive.device_num][1], &spec_len, NULL,
                    last(archive.spec[archive.device_num]),
                    (volume_len == 0), node, node_len,
                    volume, volume_len, directory, directory_len, filename,
                    filename_len, FALSE, NULL, 0, FALSE, 0);
      archive.spec[archive.device_num][0] = spec_len;
      sd_archive = standard_msg(NLS_ARCHIVE_SUFFIX);
      memcpy(
      &archive.spec[archive.device_num][archive.spec[archive.device_num][0]+1],
      sd_archive->pb, sd_archive->cb);
      archive.spec[archive.device_num][0] += sd_archive->cb;
   }
   if (!vlpb.suppress_backup) {
      if (!archive.sequential && CSubparams(VOLUME_PARAM) > 1)
         exit_with_msg(ercSyntax, NLS_TOO_MANY_VOLUMES);
      if (vlpb.noninteractive && !vlpb.overwrite_OK)
         exit_with_msg(ercSyntax, NLS_MUST_OVERWRITE);
   }
   if ((erc = open_log_file()) != ercOK)
      exit_with_msg(erc, NLS_CANT_OPEN_LOG_FILE);
   vlpb.volume_archive = vlpb.write_archive = TRUE;

}

/*-----------------------------------------------------------------------------
 This procedure performs two important functions.  First, for all volume names
 specified it ascertains that the volume exists and may be opened for read
 access with the password supplied.  Second, it calculates how much memory is
 going to be needed to hold the MFD and (optionally) the allocation bit map
 during the backup process.  This knowledge will help the archive IO process
 to efficiently allocate memory to data transfer buffers while reserving
 enough for these purposes.  Although multiple volume backup is permitted for
 tape datasets, only, the type of backup medium is not known at this time so
 we cannot reject the command on a syntax error just because there is more
 than one volume specified. */

private void scan_volumes(void) {

   old_dcb_type dcb;
   char node[MAX_NODE_LENGTH], password[MAX_PASSWORD_LENGTH],
      volume[MAX_VOLUME_LENGTH], volume_spec[MAX_FILE_SPECIFICATION_LENGTH];
   unsigned erc, i, node_len, passwords, password_len, volumes, volume_len,
      volume_spec_len;
   unsigned long memory_needed;
   sdType nls_parms[1], sd_password = {NULL, 0}, sd_volume;
   vhb_type vhb;

   if ((volumes = CSubparams(VOLUME_PARAM)) == 0)
      exit_with_msg(ercSyntax, NLS_VOLUME_REQUIRED);
   passwords = CSubparams(VOLUME_PASSWORD_PARAM);
   for (i = 0; i < volumes; i++) {
      RgParam(VOLUME_PARAM, i, &sd_volume);
      if (passwords > 1 && i >= passwords) {
         sd_password.pb = NULL;
         sd_password.cb = 0;
      } else
         RgParam(VOLUME_PASSWORD_PARAM, (i < passwords) ? i : 0, &sd_password);
      node_len = volume_len = password_len = 0;
      if (ParseFileSpec(0, sd_volume.pb, sd_volume.cb, FALSE, node, &node_len,
                        volume, &volume_len, NULL, 0, NULL, 0, password,
                        &password_len, FALSE, 1) != ercOK) {
         nls_parms[0] = *standard_msg(NLS_VOLUME);
         log_msg(NLS_SYNTAX_ERROR, 1, nls_parms, sizeof(nls_parms));
         exit_with_msg(ercSyntax, NLS_VOLUME_INVALID);
      }
      if (password_len == 0)
         memcpy(password, sd_password.pb,
                password_len = _min(sizeof(password), sd_password.cb));
      memset(volume_spec, 0, sizeof(volume_spec));
      BuildFileSpec(0, nls_parms[0].pb = volume_spec, &volume_spec_len,
                    &nls_parms[0].cb, sizeof(volume_spec), FALSE, node,
                    node_len, volume, volume_len, NULL, 0, NULL, 0, FALSE,
                    NULL, 0, FALSE, 0);
      if ((erc = QueryDCB(volume_spec, volume_spec_len, &dcb,
                          sizeof(dcb))) != ercOK) {
         log_msg((erc == ercNoSuchVolume || erc == ercBadFileSpec)
                  ? NLS_NO_SUCH_VOLUME : NLS_CANT_ACCESS,
                 1, nls_parms, sizeof(nls_parms));
         exit_with_msg(erc, 0);
      }
      if ((erc = OpenFile(&filesys.handle, volume_spec, volume_spec_len,
                          password, password_len, modeRead)) != ercOK) {
         log_msg(NLS_CANT_OPEN_VOLUME, 1, nls_parms, sizeof(nls_parms));
         exit_with_msg(erc, 0);
      }
      if (GetVHB(volume_spec, volume_spec_len, &vhb, sizeof(vhb)) == ercOK) {
         memory_needed = (vlpb.suppress_verify)
                          ? 0 : ((unsigned long) vhb.allocPageCnt) * PAGE_SIZE;
         memory_needed += scan_mfd(vhb.lfaMfdBase, vhb.cPagesMfd,
                                   !vlpb.suppress_verify);
      }
      CloseFile(filesys.handle);
      filesys.memory_for_structures = _max(filesys.memory_for_structures,
                                           memory_needed);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 One of the path components (directory) has potentially changed at this
 point.  Obtain new information that corresponds to the file handle supplied
 in the call and then compare it to prior information saved in the archive
 control block. */

void update_path(void) {

   unsigned i;
   mfd_entry_type *mfd_entry;

   if (     fhb.dirName[0] != path.directory[0]
         || NlsULCmpB(NULL, &fhb.dirName[1], &path.directory[1],
                      fhb.dirName[0], &i) != ercOK
         || i != 0xFFFF) {
      if ((mfd_entry = find_mfd_entry((char *) fhb.dirName)) == NULL) {
         memcpy(path.directory, fhb.dirName, fhb.dirName[0] + 1);
         path.directory_password[0] = 0;
         path.directory_protection = DEFAULT_DIRECTORY_PROTECTION;
         path.directory_pages = 0;
      } else {
         memcpy(path.directory, mfd_entry->sbDirectory,
                mfd_entry->sbDirectory[0] + 1);
         memcpy(path.directory_password, mfd_entry->sbPassword,
                mfd_entry->sbPassword[0] + 1);
         path.directory_protection = mfd_entry->defaultAccessCode;
         path.directory_pages = mfd_entry->cPages;
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
*/

void create_archive_sentinel(unsigned long file_length) {

   unsigned checksum = 0, i;
   archive_sentinel_type *archive_sentinel;
   iob_type *iob;

   iob = allocate_iob(sizeof(archive_sentinel_type));
   iob->ctrl = BOF;
   iob->available = sizeof(archive_sentinel_type);
   archive_sentinel = iob->data;
   memset(archive_sentinel, 0, sizeof(*archive_sentinel));
   archive_sentinel->signature = ARCHIVE_SIGNATURE;    /* m10 */	
   archive_sentinel->record_type = BOF;
   archive_sentinel->pages_data = 
         ((file_length != 0)
         ? file_length
         : (fhb.endOfFileLfa + PAGE_SIZE - 1)
                                   ) / PAGE_SIZE;
   memcpy(archive_sentinel->node, path.node, path.node[0] + 1);
   memcpy(archive_sentinel->volume, path.volume, path.volume[0] + 1);
   memcpy(archive_sentinel->directory, path.directory, path.directory[0] + 1);
   memcpy(archive_sentinel->directory_password, path.directory_password,
          path.directory_password[0] + 1);
   archive_sentinel->directory_protection = path.directory_protection;
   archive_sentinel->directory_pages = path.directory_pages;
   memcpy(archive_sentinel->filename, fhb.fileName, fhb.fileName[0] + 1);
   memcpy(archive_sentinel->file_password, fhb.password, fhb.password[0] + 1);
   archive_sentinel->directory_type = fhb.ObjectType;
   archive_sentinel->protection = fhb.accessProtection;
   archive_sentinel->control.suppress_backup = fhb.fNoSave;
   archive_sentinel->control.hidden_file = fhb.fNoDirPrint;
   archive_sentinel->control.prevent_delete = fhb.fNoDelete;
   archive_sentinel->control.read_only = fhb.fReadOnly;
   archive_sentinel->ms_dos_magic = fhb.bDosMagic;
   archive_sentinel->ms_dos_attribute = fhb.bDosAttribute;
   archive_sentinel->object_type = fhb.userObjectType;
   archive_sentinel->creation_date = fhb.creationDT;
   archive_sentinel->modification_date = fhb.modificationDT;
   archive_sentinel->access_date = fhb.accessDT;
   archive_sentinel->expiration_date = fhb.expirationDT;
   archive_sentinel->eof_lfa = fhb.endOfFileLfa;
   memcpy(archive_sentinel->distrix_info, fhb.rgbDistrix,
          sizeof(fhb.rgbDistrix));
   memcpy(archive_sentinel->application_info, fhb.application,
          sizeof(fhb.application));
   archive_sentinel->dataset_creation_time = archive.creation_time;
   for (i = 0; i < sizeof(archive_sentinel_type) / 2; i++)
      checksum += ((unsigned *) archive_sentinel)[i];
   archive_sentinel->checksum = -(checksum + ARCHIVE_SIGNATURE);  /* m10 */
   Send(archive.msg_exch, iob);

}

void create_archive_EOD(Boolean final_close) {

   unsigned checksum = 0, i;
   archive_sentinel_type *archive_sentinel;
   iob_type *iob;

   iob = allocate_iob(sizeof(archive_sentinel_type));
   iob->ctrl = EOD;
   iob->mode.final_close = final_close;
   iob->available = sizeof(archive_sentinel_type);
   archive_sentinel = iob->data;
   memset(archive_sentinel, 0, sizeof(*archive_sentinel));
   archive_sentinel->signature = ARCHIVE_SIGNATURE;   /* m10 */
   archive_sentinel->record_type = EOD;
   for (i = 0; i < sizeof(archive_sentinel_type) / 2; i++)
      checksum += ((unsigned *) archive_sentinel)[i];
   archive_sentinel->checksum = -(checksum + ARCHIVE_SIGNATURE);  /* m10 */
   Send(archive.msg_exch, iob);
   do
      Wait(target.msg_exch, &iob);
   while (iob->ctrl != SYNC);
   deallocate_iob(iob);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Pretty much as its name suggests... */

private void summarize_backup(void) {

   sdType nls_parms[] = {(void *) &summary.count[BACKED_UP_OK],
                                  sizeof(summary.count[BACKED_UP_OK]),
                         (void *) &summary.count[IO_ERRORS],
                                  sizeof(summary.count[IO_ERRORS]),
                         (void *) &summary.count[OTHER_ERRORS],
                                  sizeof(summary.count[OTHER_ERRORS]),
                         (void *) &summary.count[NOT_BACKED_UP],
                                  sizeof(summary.count[NOT_BACKED_UP]),
                         (void *) &summary.count[BYPASSED],
                                  sizeof(summary.count[BYPASSED]),
                         (void *) &summary.count[TOTAL_FILES],
                                  sizeof(summary.count[TOTAL_FILES])};

   if (!vlpb.suppress_verify) {
      log_msg(NLS_VERIFICATION_COMPLETE, 1, NULL, 0);
      if (filesys.invalid)
         log_msg(NLS_REINITIALIZE_VOLUME, 1, NULL, 0);
   }
   if (!vlpb.suppress_backup)
      log_msg(NLS_FILES_ARCHIVED, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[IO_ERRORS] > 0)
      log_msg(NLS_FILES_IO_ERRORS, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[OTHER_ERRORS] > 0)
      log_msg(NLS_FILES_OTHER_ERRORS, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[NOT_BACKED_UP] > 0)
      log_msg(NLS_FILES_NOT_ARCHIVED, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[BYPASSED] > 0)
      log_msg(NLS_FILES_BYPASSED, 1, nls_parms, sizeof(nls_parms));
   log_msg(NLS_TOTAL_FILES, 1, nls_parms, sizeof(nls_parms));
   vid_only_msg(NLS_NEWLINE, 1, NULL, 0);

}
