/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  archive_msgs.c                                                           */
/*                                                                           */
/*  Default messages and functions for CTOS archival utilities.              */
/*                                                                           */
/*  Functions to perform archive medium I/O for the utilities.               */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  07/16/92  122J.20  D. Gilson     /  Corrected screwed up logic in Proceed*/
/*  07/07/92  122J.19  D. Gilson     /  Added new msg for invalid dev spec.  */
/*                                      NLS_DEVICE_INVALID_2 PLE 15591773    */
/*  06/11/92  122I.18  D. Gilson     /  Corrected lock_video and proceed to  */
/*                                      delay and handle more erc correctly. */
/*  05/01/92  122G.17  D. Gilson     /  Added new proceed procedure to fix   */
/*                                   /  End of tape not being written from   */
/*                                   /  Selective Archive.                   */
/*  03/23/92  122G.16  D. Gilson     /  changed  NLS_FILE_LIST_FROM_FORMAT   */
/*  11/07/91  130D.15  D. Gilson     /  changed NLS_VERIFICATION_SUSPECT     */
/*                                   /  added and/or integrity               */
/*  10/07/91  130D.14  D. Gilson     /  Added NLS_INVALID_TAPE_MARK_SPEC     */
/*  08/22/91  130B.13  W. Chiu       /  Added NlS_REPLACE(32499).            */
/*  08/05/91  130A.12  P. Johansson  /  New message for service buffer too   */
/*                                      small.                               */
/*  07/16/91  130A.11  P. Johansson  /  New message for '[Overwrite ok?]';   */
/*                                      modify 'log_msg()' so that 'done'    */
/*                                      message isn't split over page break. */
/*  05/10/91  121J.10  P. Johansson  /  Additional summary messages for      */
/*                                      Volume Archive and Restore Archive.  */
/*  05/08/91  121J.09  P. Johansson  /  Build canonical form of message file */
/*                                      specification for Restore Archive;   */
/*                                      reinitialize message when an error   */
/*                                      is returned by GetMsgUnexpanded.     */
/*  05/04/91  121J.08  C.G. Naik     /  Expand message file area only once.  */
/*  05/01/91  121J.07  P. Johansson  /  "Processing file header n of N ... " */
/*  04/18/91  121J.06  P. Johansson  /  Buffer parameters for configuration  */
/*                                      file.                                */
/*  04/08/91  121J.05  C.G. Naik     /  Delay in log_msg(), vid_only_msg()   */
/*                                      to avoid deadlocks.                  */
/*  03/24/91  121H.04  P. Johansson  /  Additional messages.                 */
/*  02/11/91  121G.03  P. Johansson  /  Additional messages.                 */
/*  01/30/91  121F.02  P. Johansson  /  Procedure to explicitly open [Vid]   */
/*                                      so that video filtering works.       */
/*  01/23/91  121F.01  P. Johansson  /  Additional messages.                 */
/*  12/17/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Archive utilities that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define Beep
#define BuildFileSpec
#define CloseByteStream
#define CloseMsgFile
#define Delay
#define ENLS_GetChar
#define ENLS_MapCharToStdValue
#define ErrorExit
#define ErrorExitString
#define ExpandAreaSL
#define ExpandLocalMsg
#define GetDateTime
#define GetMsg
#define GetMsgUnexpanded
#define GetMsgUnexpandedLength
#define InitMsgFile
#define NlsClass
#define NlsULCmpB
#define nPrint
#define OpenByteStream
#define OutputBytesWithWrap
#define ParseFileSpec
#define PutChar
#define RgParam
#define ShrinkAreaSL
#define WriteBsRecord
#define WriteByte

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define FhbType
#define sdType

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
#include <kbd.h>
#define KbdErc
#include <erc.h>
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern Boolean lockset(Boolean *lock);

/* External variables imported by this module */

extern char sbVerRun[];

/* Global variables exported by this manuscript */

char msg_file_spec[MAX_FILE_SPECIFICATION_LENGTH + 1];
journal_type journal;

/* Static variables global within this manuscript */

private char command_id[64];
private char msg_buffer[1024];
private char msg_cache[1024];
private void *msg_expansion = &msg_expansion;
private char run_time[64];
private sdType log_parms[] = {(void *) command_id, sizeof(command_id),
                              (void *) &run_time, sizeof(run_time),
                              (void *) &journal.page, sizeof(journal.page)};

/* Function prototypes defined before the functions themselves are declared */

void log_page_header(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 Here are all the messages used by the Sequential Access service.  If
 possible, the pointers to these compiled strings are replaced by text that
 comes from a (potentially) nationalized message file.  If no message file is
 available, these will have to do.  The table MUST be terminated with message
 number zero and a null pointer to a zero length string (in case anyone
 searches for a message number that is not present). */

private msg_descriptor_type frequent_msg[] = {
   default_msg(NLS_DONE, "done\n"),
   default_msg(NLS_NEWLINE, "\n"),
   default_msg(NLS_VID, "[Vid]"),
   default_msg(NLS_PROCESSING_FILE_N, "\x0FFX\x001"
                                      "\x0FFC\x000\x001"
                                      "Processing file %0N ... "
                                      "\x0FFX\x000"),
   default_msg(NLS_PROCESSING_FILE_N_OF, "\x0FFX\x001"
                                         "\x0FFC\x000\x001"
                                         "Processing file %0N of %1N ... "
                                         "\x0FFX\x000"),
   default_msg(NLS_PROMPT, "\n    (Press GO to confirm, CANCEL to deny, or "
                           "FINISH to exit the utility)\n"),
   default_msg(NLS_PROCESSING_FILE_HEADER_N_OF, "\x0FFX\x001"
                                                "\x0FFC\x000\x001"
                                                "Processing file header %0N "
                                                "of %1N ... "
                                                "\x0FFX\x000"),
   default_msg(NLS_PAGE_HEADER, "\f%0S      Page %2N\n%1S\n\n"),
   default_msg(NLS_DATE_TIME, "%2D14 %2D9"),
   default_msg(NLS_INITIAL_FHB_HEADER, "\nInitial file header %0N chains to "
                                       "file header %2N\n   File: %3S\n   "
                                       "End of file LFA: %4N\n"),
   default_msg(NLS_EXTENSION_FHB_HEADER, "\nExtension file header %0N for "
                                         "file %1N chains to file header "
                                         "%2N\n"),
   default_msg(NLS_FREE_FHB_HEADER, "Free file header %0N chains to file "
                                    "header %2N\n"),
   default_msg(NLS_FHB_RUN, "      Run at VDA %5N for %6N\n"),
   default_msg(NLS_FHB_RUN_SUMMARY, "   Total runs: %7N\n"),
   default_msg(NLS_DIRECTORY, "\nDirectory: <\x0FFLN%1S\x0FFLF> Total pages: "
                              "%2N Default access level: %3N\n"),
   default_msg(NLS_FILE_WITHOUT_DIRECTORY, "    File: \x0FFLN%1S\x0FFLF\n"),
   default_msg(NLS_LOCAL, "Local"),
   default_msg(NLS_MASTER, "Master"),
   default_msg(NLS_MFD, "<Sys>MFD.Sys"),
   default_msg(NLS_DEFAULT_WILDCARD, "<*>*"),
   default_msg(NLS_CONFIRM_ARCHIVE, "Archive \x0FFLN%2S\x0FFLF?"),
   default_msg(NLS_CANT_ARCHIVE, "***** Unable to archive \x0FFLN%2S\x0FFLF "
                                 "(Error %3N)\n"),
   default_msg(NLS_BACKING_UP, "Archiving \x0FFLN%2S\x0FFLF ... "),
   default_msg(NLS_CONFIRM_RESTORE, "Restore \x0FFLN%0S\x0FFLF to "
                                    "\x0FFLN%1S\x0FFLF?"),
   default_msg(NLS_FILE_EXISTS, "\x0FFLN%1S\x0FFLF already exists.  "
                                "Overwrite?"),
   default_msg(NLS_DID_NOT_RESTORE, "Did not restore \x0FFLN%0S\x0FFLF to "
                                    "\x0FFLN%1S\x0FFLF (target file already "
                                    "exists)\n"),
   default_msg(NLS_RESTORING, "Restoring \x0FFLN%0S\x0FFLF to "
                              "\x0FFLN%1S\x0FFLF%3S ... "),
   default_msg(NLS_OVERWRITING, " (overwriting)"),
   default_msg(NLS_LIST_ONLY, "\x0FFLN%0S\x0FFLF\n"),
   default_msg(NLS_REPLACE, "\x0FFLN%1S\x0FFLF is in use.  "
                            "Replace?")};

private msg_descriptor_type infrequent_msg[] = {
   default_msg(NLS_COMMAND_VERSION, "%0S %1S\n\n"),
   default_msg(NLS_VOLUME, "Volume or device name"),
   default_msg(NLS_PASSWORD, "Volume or device password"),
   default_msg(NLS_ARCHIVE_AFTER_DATE, "Incremental from"),
   default_msg(NLS_SUPPRESS_ARCHIVE, "Suppress backup?"),
   default_msg(NLS_SUPPRESS_VERIFY, "Suppress verification?"),
   default_msg(NLS_ARCHIVE_DATASET, "Archive dataset"),
   default_msg(NLS_DELETE_EXISTING, "Delete existing archive dataset?"),
   default_msg(NLS_LOG_FILE, "Print file"),
   default_msg(NLS_DISPLAY_STRUCTURES, "Display structures?"),
   default_msg(NLS_VERIFY, "Verify write?"),
   default_msg(NLS_NONINTERACTIVE, "Suppress user interaction?"),
   default_msg(NLS_FILE_LIST, "File list"),
   default_msg(NLS_CONFIRM, "Confirm each?"),
   default_msg(NLS_FILE_LIST_FROM, "File list from"),
   default_msg(NLS_FILE_LIST_TO, "File list to"),
   default_msg(NLS_OVERWRITE_OK, "Overwrite OK?"),
   default_msg(NLS_ARCHIVE_SEQUENCE, "Dataset sequence number"),
   default_msg(NLS_MERGE_WITH_EXISTING_FILE, "Merge with existing file?"),
   default_msg(NLS_LIST_FILES_ONLY, "List files only?"),
   default_msg(NLS_SUPPRESS_CONFIRMATION_MSG, "Suppress confirmation "
                                              "message?"),
   default_msg(NLS_CASE_INVALID, "Command case (from Sys.Cmds or $RUN "
                                 "directive) is not recognized\n"),
   default_msg(NLS_RGPARAM_ERROR, "internal error fetching parameter(s)\n"),
   default_msg(NLS_WRONG_NUMBER_ARGUMENTS, "invalid number of arguments\n"),
   default_msg(NLS_SYNTAX_ERROR, "Syntax error in parameter '%0S': "),
   default_msg(NLS_FILE_SPEC_INVALID, "file specification not in "
                                      "'{Node}[Volume]<Directory>Filename' "
                                      "form"),
   default_msg(NLS_NODE_INVALID, "invalid node name\n"),
   default_msg(NLS_VOLUME_INVALID, "invalid device or volume name\n"),
   default_msg(NLS_DIRECTORY_INVALID, "invalid directory name\n"),
   default_msg(NLS_FILENAME_INVALID, "invalid file specification; file name "
                                     "missing or too long\n"),
   default_msg(NLS_PASSWORD_INVALID, "invalid password\n"),
   default_msg(NLS_DATE_TIME_INVALID, "syntax error in date/time\n"),
   default_msg(NLS_YEAR_INVALID, "year is invalid\n"),
   default_msg(NLS_YES_OR_NO_EXPECTED, "expecting yes or no\n"),
   default_msg(NLS_NUMBER_INVALID, "invalid number\n"),
   default_msg(NLS_SEQ_DEVICE_PARAMETERS, "Sequential Access Device "
                                          "Parameters"),
   default_msg(NLS_MIN_RECORD_SIZE, "\n  [Minimum (or fixed) record size "
                                    "(512)]"),
   default_msg(NLS_MAX_RECORD_SIZE, "\n  [Maximum record size (for variable "
                                    "records)]"),
   default_msg(NLS_BLOCK_SIZE, "\n  [Block size (fixed records, defaults to "
                               "record size)]"),
   default_msg(NLS_DENSITY, "\n  [Recording density code]"),
   default_msg(NLS_SPEED, "\n  [Transport speed code]"),
   default_msg(NLS_UNBUFFERED, "\n  [Unbuffered mode?]"),
   default_msg(NLS_ERASE_ON_CLOSE, "\n  [Erase to EOM after close?]"),
   default_msg(NLS_REWIND_ON_CLOSE, "\n  [Rewind after close?]"),
   default_msg(NLS_SEQ_CONFIG_FILE, "[Sys]<Sys>%0Config.Sys"),
   default_msg(NLS_MIN_RECORD_SIZE_FIELD, "MinRecordSize"),
   default_msg(NLS_MAX_RECORD_SIZE_FIELD, "MaxRecordSize"),
   default_msg(NLS_BLOCK_SIZE_FIELD, "BlockSize"),
   default_msg(NLS_DENSITY_FIELD, "DensityCode"),
   default_msg(NLS_SPEED_FIELD, "SpeedCode"),
   default_msg(NLS_UNBUFFERED_FIELD, "UnbufferedMode"),
   default_msg(NLS_ERASE_ON_CLOSE_FIELD, "EraseOnClose"),
   default_msg(NLS_REWIND_ON_CLOSE_FIELD, "RewindOnClose"),
   default_msg(NLS_CLEAR_VID1, "\x0FFX\x001"
                               "\x0FFF \x000\x001\x028\x001"
                               "\x0FFX\x000"),
   default_msg(NLS_CANT_OPEN_LOG_FILE, "Unable to open print file\n"),
   default_msg(NLS_ARCHIVE_SUFFIX, ".01"),
   default_msg(NLS_MOUNT_ARCHIVE_DISK, "Please mount %0S"),
   default_msg(NLS_ARCHIVE_FILE_EXISTS, "Archive dataset already exists, "
                                        "overwrite?"),
   default_msg(NLS_CREATING_ARCHIVE, "Creating archive dataset ... "),
   default_msg(NLS_DISK_FULL, "Not enough space available, use another "
                              "disk\n"),
   default_msg(NLS_CANT_CREATE_ARCHIVE_DATASET, "Error %1N while creating "
                                                "archive dataset\n"),
   default_msg(NLS_ARCHIVE_DATASET_EXISTS, "An archive dataset already "
                                           "exists, overwrite?"),
   default_msg(NLS_CANT_OPEN_ARCHIVE_DATASET, "Error %1N while opening "
                                           "archive dataset\n"),
   default_msg(NLS_NEED_SEQ_ACCESS_SERVICE, "The Sequential Access service "
                                            "must be installed before tape(s) "
                                            "can be used\n"),
   default_msg(NLS_DEFAULT_DATASET, "[QIC]"),
   default_msg(NLS_DEFAULT_FLOPPY, "[Archive]<Sys>"),
   default_msg(NLS_OLD_BACKUP_BOV, "<<< BACKUP TAPE - "),
   default_msg(NLS_OLD_BACKUP_EOV, "/// END OF REEL - "),
   default_msg(NLS_OLD_BACKUP_EOD, "*** END OF TAPE - "),
   default_msg(NLS_ARCHIVE_BOV, "ARKIVE BOV"),
   default_msg(NLS_ARCHIVE_EOV, "ARKIVE EOV"),
   default_msg(NLS_ARCHIVE_EOD, "ARKIVE EOD"),
   default_msg(NLS_OEM_TEXT, "          "),
   default_msg(NLS_DEVICE_INVALID, "Invalid device specification %0S.  "
                                   "Device specification must be of the "
                                   "form\n[XXXXm]n or [XXXXm]+ where XXXX "
                                   "is a device name (e.g. QIC, TAPE, DDS, "
                                   "HITC),\nm is an optional unit number and "
                                   "n is an optional file sequence number."),
   default_msg(NLS_APPEND_INVALID, "Archive dataset must be of the form [XXX] "
                                   "or [XXX]N, where XXX is a device "
                                   "name\n(e.g. QIC, TAPE, DDS) and N is an "
                                   "optional file sequence number\n"),
   default_msg(NLS_DEVICE_IN_USE, "%0S is in use by another process\n"),
   default_msg(NLS_WRITE_PROTECTED, "The medium in device %0S is "
                                   "write-protected\n"),
   default_msg(NLS_DEVICE_NOT_READY, "Device %0S is off line or not ready\n"),
   default_msg(NLS_MEDIUM_FULL, "There is no more room available on the "
                                "medium in device %0S\n"),
   default_msg(NLS_POSITIONING_ERROR, "Error %2N while positioning medium on "
                                      "device %0S\n"),
   default_msg(NLS_REQUEST_INTERVENTION, "Please correct the problem.  Do you "
                                         "wish to continue?"),
   default_msg(NLS_RETENSIONING, "Retensioning the medium on %0S ... "),
   default_msg(NLS_CONFIRM_ERASE_OLD_FORMAT, "%0S0 contains an archive "
                                             "dataset created on %1S\n(%2S).  "
                                             "Do you wish to erase it?"),
   default_msg(NLS_CONFIRM_ERASE, "%0S0 contains an archive dataset created "
                                  "on %1D14 %1D9.  Do you wish to erase it?"),
   default_msg(NLS_ERASING, "Erasing the medium on %0S ... "),
   default_msg(NLS_OLD_FORMAT_DATASET_EXISTS, "An archive dataset already "
                                              "exists on %0S%3N (created on "
                                              "%1S), overwrite?"),
   default_msg(NLS_ARCHIVE_SEQ_DATASET_EXISTS, "An archive dataset already "
                                               "exists on %0S%3N (created on "
                                               "%1D14 %1D9), overwrite?"),
   default_msg(NLS_MOUNT_ARCHIVE_TAPE, "\nPlease mount archive dataset volume "
                                       "%2N on %0S"),
   default_msg(NLS_NO_ARCHIVE_LABEL, "No valid archive dataset label is "
                                     "present on %0S%1N\n"),
   default_msg(NLS_ARCHIVE_SEQUENCE_ERROR, "Archive dataset volume %2N on %0S "
                                           "is out of sequence"),
   default_msg(NLS_RESTORING_FROM_OLD_FORMAT, "Restoring from archive dataset "
                                              "created on %1S\n\n"),
   default_msg(NLS_RESTORING_FROM, "Restoring from archive dataset created "
                                   "on %1D14 %1D9\n\n"),
   default_msg(NLS_REWINDING, "Rewinding the medium on %0S ... "),
   default_msg(NLS_NO_BIT_MAP, "***** Volume home block invalid: Zero-length "
                               "allocation bit map\n"),
   default_msg(NLS_NO_MFD, "***** Volume home block invalid: Zero-length "
                           "MFD.Sys\n"),
   default_msg(NLS_NO_FILE_HEADERS, "***** Volume home block invalid: "
                                    "Zero-length FileHeaders.Sys\n"),
   default_msg(NLS_VHB_IO_ERROR, "***** I/O error reading Volume Home Block "
                                 "at VDA %0N\n"),
   default_msg(NLS_NO_INITIAL_VHB, "***** No initial Volume Home Block\n"),
   default_msg(NLS_INVALID_INITIAL_VHB, "***** Invalid initial Volume Home "
                                        "Block at VDA %0N\n"),
   default_msg(NLS_INVALID_WORKING_VHB, "***** Invalid working Volume Home "
                                        "Block at VDA %0N\n"),
   default_msg(NLS_VHB_HEADER, "\n   Volume name: %1S\n"
                               "   Creation date: %2D3\n"
                               "   Last modified: %3D3\n"
                               "   Free pages: %4N\n"
                               "   Free file headers: %5N\n\n"),
   default_msg(NLS_BAD_BLOCK_IO_ERROR, "***** Error reading bad block page "
                                       "%0N\n"),
   default_msg(NLS_CANT_VERIFY_BIT_MAP, "\nInsufficient memory to verify "
                                        "allocation bit map\n"),
   default_msg(NLS_BIT_MAP_TOO_SMALL, "***** Volume capacity (%0N) exceeds "
                                      "the size of the allocation bit map\n"),
   default_msg(NLS_BIT_MAP_IO_ERROR, "***** Error reading allocation bit "
                                     "map\n"),
   default_msg(NLS_ZERO_LENGTH_RUN, "***** Zero length run at VDA %0N\n"),
   default_msg(NLS_RUN_EXCEEDS_EOM, "***** Run at VDA %0N for %1N exceeds "
                                    "disk size\n"),
   default_msg(NLS_DUPLICATE_ALLOCATION, "***** Duplicate allocation in bit "
                                         "map at VDA %0N\n"),
   default_msg(NLS_PAGE_NOT_IN_BIT_MAP, "***** Unused page outside of range "
                                        "of allocation bit map at VDA %2N\n"),
   default_msg(NLS_PAGE_IN_LIMBO, "***** Unused page not free in bit map at "
                                  "VDA %2N\n"),
   default_msg(NLS_FREE_PAGES, "\nNumber of unallocated pages: %0N\n"),
   default_msg(NLS_FREE_PAGES_DISAGREEMENT, "***** Volume Home Block free "
                                            "page count (%1N) does not match "
                                            "bit map\n"),
   default_msg(NLS_PRIMARY_FHB_IO_ERROR, "***** I/O error %0N reading primary "
                                         "File Header page %1N\n"),
   default_msg(NLS_NO_ALTERNATE_FHB, "***** No alternate for primary File "
                                     "Header page %1N\n"),
   default_msg(NLS_ALTERNATE_FHB_IO_ERROR, "***** I/O error %0N reading "
                                           "alternate File Header page %1N\n"),
   default_msg(NLS_FHB_DISAGREEMENT, "***** Primary File Header page %1N and "
                                     "alternate File Header differ\n"),
   default_msg(NLS_PRIMARY_FHB_INVALID, "***** Primary File Header %0N "
                                        "invalid\n"),
   default_msg(NLS_ALTERNATE_FHB_INVALID, "***** Alternate File Header %0N "
                                          "invalid\n"),
   default_msg(NLS_EXT_FHB_CHAIN_BROKEN, "***** Extension File Header chain "
                                         "is broken\n"),
   default_msg(NLS_NO_DIRECTORY_ENTRY_FOR_FHB, "***** No directory entry "
                                               "found for File Header page "
                                               "%1N\n   File: <%2S>%3S\n"),
   default_msg(NLS_CHECKSUM_ERROR, "   Checksum error\n"),
   default_msg(NLS_FHB_PAGE_ERROR, "   Page number error (%1N in the File "
                                   "Header should be %0N)\n"),
   default_msg(NLS_FHB_SEQ_ERROR, "   Sequence number error\n"),
   default_msg(NLS_EXT_FHB_NO_RUNS, "   Extension File Header has no runs\n"),
   default_msg(NLS_FHB_NOT_FULL, "   File Header extension link error (File "
                                 "Header runs are not filled)\n"),
   default_msg(NLS_FILENAME_ERROR, "   Filename is null or too long\n"),
   default_msg(NLS_DIRECTORY_ERROR, "   Directory name is null or too long\n"),
   default_msg(NLS_FREE_FHBS_BROKEN, "***** Free file header(s) chain is "
                                     "broken\n"),
   default_msg(NLS_MFD_PAGE_INVALID, "***** MFD page %0N invalid\n"),
   default_msg(NLS_MFD_PAGE_IO_ERROR, "***** Error reading MFD page %0N\n"),
   default_msg(NLS_MFD_OVERFLOW, "WARNING: MFD page %0N overflows\n"),
   default_msg(NLS_MFD_ENTRY_INVALID, "***** Invalid MFD entry for "
                                      "<\x0FFLN%1S\x0FFLF> (page count "
                                      "zero)\n"),
   default_msg(NLS_RAW_DIRECTORY_PAGE_IO_ERROR, "***** Error reading "
                                                "directory page at VDA %0N\n"),
   default_msg(NLS_DIRECTORY_PAGE_IO_ERROR, "***** Error reading directory "
                                            "page %0N\n"),
   default_msg(NLS_RAW_DIRECTORY_PAGE_INVALID, "***** Directory page at VDA "
                                               "%0N invalid\n"),
   default_msg(NLS_DIRECTORY_PAGE_INVALID, "***** Directory page %0N "
                                           "invalid\n"),
   default_msg(NLS_DIRECTORY_FHB_INVALID, "***** Directory entry references "
                                          "invalid File Header page %4N\n"),
   default_msg(NLS_DIRECTORY_OVERFLOW, "WARNING: Directory page %0N "
                                       "overflows\n"),
   default_msg(NLS_CANT_VERIFY_DIRECTORY, "Insufficient memory to verify "
                                          "directory pages\n"),
   default_msg(NLS_CANT_SORT_FILENAMES, "Insufficient memory to sort and "
                                        "display filename(s)\n"),
   default_msg(NLS_VERIFYING_FHBS, "\nVerifying file headers\n"),
   default_msg(NLS_ARCHIVING_FILES, "\nVerifying file headers during "
                                    "archive\n"),
   default_msg(NLS_VERIFYING_FREE_FHBS, "\nVerifying free file headers "
                                        "chain\n"),
   default_msg(NLS_VERIFYING_MFD, "\nVerifying master file directory\n"),
   default_msg(NLS_VERIFYING_BIT_MAP, "\nVerifying allocation bit map\n"),
   default_msg(NLS_VERIFYING_DIRECTORIES, "\nVerifying directories\n"),
   default_msg(NLS_REINITIALIZE_VOLUME, "***** SOURCE VOLUME INVALID - "
                                        "SUGGEST REINITIALIZATION\n"),
   default_msg(NLS_VERIFICATION_COMPLETE, "\nVolume structure verification "
                                          "complete\n"),
   default_msg(NLS_MUST_OVERWRITE, "'[Delete existing archive dataset?]' must "
                                   "be 'Yes'\n    if user interaction is "
                                   "suppressed\n"),
   default_msg(NLS_VOLUME_REQUIRED, "You must provide 'Volume or device "
                                    "name'\n"),
   default_msg(NLS_CONFIRM_INCONSISTENT, "'[Confirm each?]' cannot be 'Yes' "
                                         "if user interaction is "
                                         "suppressed\n"),
   default_msg(NLS_FUTURE_DATE, "The date given is in the future\n"),
   default_msg(NLS_WILDCARD_MISMATCH, "Wildcards in 'File list from' ('%0S') "
                                      "don't match 'File list to' ('%1S')\n"),
   default_msg(NLS_CANT_OPEN_DEVICE, "Unable to open device %0S, error "
                                     "code %1N\n"),
   default_msg(NLS_NO_SUCH_VOLUME, "%0S does not exist locally or on "
                                   "the server\n"),
   default_msg(NLS_CANT_ACCESS, "Unable to successfully access %0S\n"),
   default_msg(NLS_CANT_OPEN_VOLUME, "Unable to open volume %0S\n"),
   default_msg(NLS_NO_VHB, "Unable to locate a volume home block -- no data "
                           "can be recovered from this volume\n"),
   default_msg(NLS_ARCHIVE_OF, "Volume archive of %0S\n"),
   default_msg(NLS_INCREMENTAL_FROM, "Incremental from %2D14 %2D9\n"),
   default_msg(NLS_VERIFICATION_SUSPECT, "\nDevice is in use, volume "
                                         "verification and/or integrity "
                                         "may be suspect"),
   default_msg(NLS_TOO_MANY_VOLUMES, "You cannot archive more than one volume "
                                     "unless tape(s) are used\n"),
   default_msg(NLS_OVERWRITE_EXPLICIT, "'[Overwrite ok?]' cannot be blank "
                                       "when you suppress user interaction\n"),
   default_msg(NLS_FILE_LIST_REQUIRED, "'File list' parameter(s) must be "
                                       "provided\n"),
   default_msg(NLS_EXPANDING_WILDCARDS, "Expanding wildcard(s) ... "),
   default_msg(NLS_TOO_MANY_WILDCARDS, "\nInsufficient buffer space for "
                                       "wildcard expansion\n"),
   default_msg(NLS_WILDCARD_EXPANSION_ERROR, "\nError in wildcard "
                                             "expansion\n"),
   default_msg(NLS_WILDCARD_TOO_COMPLICATED, "\nWildcard expression too "
                                             "complicated for expansion\n"),
   default_msg(NLS_IO_ERROR, "***** I/O Error %0N at VDA %1N\n"),
   default_msg(NLS_FILE_IO_ERROR, "***** I/O error at LFA %4N\n"),
   default_msg(NLS_FILE_ERROR, "***** Error %1N\n"),
   default_msg(NLS_FILES_ARCHIVED, "\nNumber of files archived: %0N\n"),
   default_msg(NLS_FILES_IO_ERRORS, "Number of files with I/O errors: %1N\n"),
   default_msg(NLS_FILES_OTHER_ERRORS, "Number of files with other errors: "
                                       "%2N\n"),
   default_msg(NLS_FILES_NOT_ARCHIVED, "Number of files not accessible: "
                                       "%3N\n"),
   default_msg(NLS_FILES_BYPASSED, "Number of files bypassed: %4N\n"),
   default_msg(NLS_TOTAL_FILES, "\nTotal number of files on volume: %5N\n"),
   default_msg(NLS_TOO_MANY_DATASETS, "'Archive dataset' must be left blank "
                                      "(default) or have only one entry\n"),
   default_msg(NLS_FILE_LISTS_DISAGREE, "'File list from' must have the same "
                                        "number of arguments as 'File list "
                                        "to'\n"),
   default_msg(NLS_FILE_LIST_FROM_FORMAT, "'File list from' must be in the "
                                          "form "
                                          "{Node}[Volume]<Directory>Filename "
                                          "(both {Node} and [Volume] are "
                                          "optional)\n"
                                          "Possible cause of problem: %0S\n"),
   default_msg(NLS_SEQUENCE_INVALID, "'Sequence number' must be between 1 and "
                                     "99, inclusive\n"),
   default_msg(NLS_CANT_RESTORE, "***** Unable to restore \x0FFLN%0S\x0FFLF "
                                 "to \x0FFLN%1S\x0FFLF (Error %2N)\n"),
   default_msg(NLS_CREATING_DIRECTORY, "Creating directory \x0FFLN%3S\x0FFLF "
                                       "... "),
   default_msg(NLS_DIRECTORY_CREATION_ERROR, "***** Unable to create "
                                             "directory (Error %2N)\n"),
   default_msg(NLS_FILES_PROCESSED, "\nNumber of files processed: %0N\n"),
   default_msg(NLS_FILES_RESTORED, "Number of files restored without error: "
                                   "%1N\n"),
   default_msg(NLS_FILES_RESTORED_WITH_ERRORS, "Number of files restored with "
                                               "I/O errors: %2N\n"),
   default_msg(NLS_FILES_WITHOUT_FHB, "Number of files without header "
                                      "information: %3N\n"),
   default_msg(NLS_FILES_NOT_RESTORED, "Number of files which could not be "
                                       "restored: %4N\n"),
   default_msg(NLS_FILES_UNRECOVERABLE, "Number of files not recoverable: "
                                        "%5N\n"),
   default_msg(NLS_CREATING_SEQ_DATASET, "\nCreating archive dataset on "
                                         "%0S%1N\n"),
   default_msg(NLS_SERVICE_BUFFER_SIZE, "\n  [Service buffer size, in bytes]"),
   default_msg(NLS_TOTAL_SERVICE_BUFFERS, "\n  [Total service buffers]"),
   default_msg(NLS_SERVICE_BUFFER_SIZE_FIELD, "ServiceBufferSize"),
   default_msg(NLS_TOTAL_SERVICE_BUFFERS_FIELD, "TotalServiceBuffers"),
   default_msg(NLS_CANT_MOUNT_ARCHIVE, "User intervention required to mount "
                                       "next archive dataset\n"),
   default_msg(NLS_SERVICE_BUFFER_TOO_SMALL, "This archive dataset requires a "
                                             "minimum service buffer size of "
                                             "%2N\n"),
   default_msg(NLS_INVALID_TAPE_MARK_SPEC,"Invalid tape mark specification.  "
                                          "Valid tape mark specifications for"
                                          "writing to tape are 0 and + only. "
                                          "\n"),
   default_msg(NLS_DEVICE_INVALID_2,"Invalid device specification %0S. "
                                    "Device specification must be of the \n"
                                    "form [XXXX] where XXXX is a device "
                                    "name (e.g. QIC, TAPE, DDS, HITC)."
                                    "\n"),
   default_msg(0, "")};

pragma Page(1);
/*-----------------------------------------------------------------------------
 Attempt to open the standard message file in order to replace the frequently
 used hard-coded messages (above) with native-language messages if they are
 available.  The infrequently used messages are retrieved on an as-needed
 basis from the message file. */

void initialize_msgs(void) {

   char directory[MAX_DIRECTORY_LENGTH], filename[MAX_FILENAME_LENGTH],
      volume[MAX_VOLUME_LENGTH];
   unsigned directory_len, filename_len, i, msg_expansion_size = 0,
      msg_file_spec_len, msg_len, volume_len;

   if (msg_expansion == &msg_expansion) {		/* First time? */
      ParseFileSpec(0, MSG_FILE, last(MSG_FILE), FALSE, NULL, NULL, volume,
                    &volume_len, directory, &directory_len, filename,
                    &filename_len, NULL, NULL, TRUE, 0);
      BuildFileSpec(0, &msg_file_spec[1], &msg_file_spec_len, NULL,
                    last(msg_file_spec), FALSE, NULL, 0, volume, volume_len,
                    directory, directory_len, filename, filename_len, FALSE,
                    NULL, 0, TRUE, 0);
      msg_file_spec[0] = msg_file_spec_len;
   }
   if (     (InitMsgFile(&msg_file_spec[1], msg_file_spec[0],
                         NULL, 0, msg_buffer, sizeof(msg_buffer), msg_cache,
                         sizeof(msg_cache)) == ercOK)
         && msg_expansion == &msg_expansion) {
      for (i = 0; i <= last(frequent_msg); i++)
         if (GetMsgUnexpandedLength(frequent_msg[i].msg_num,
                                    &msg_len) == ercOK) {
            if (     msg_len > frequent_msg[i].msg_len
                  && ExpandAreaSL(msg_len, selector_of(frequent_msg[i].msg),
                                  &frequent_msg[i].msg) == ercOK)
               frequent_msg[i].msg_len = msg_len;
            GetMsgUnexpanded(frequent_msg[i].msg_num, frequent_msg[i].msg,
                             frequent_msg[i].msg_len,
                             &frequent_msg[i].msg_len);
         }
      for (i = 0; i < last(infrequent_msg); i++)
         if (     GetMsgUnexpandedLength(infrequent_msg[i].msg_num,
                                         &msg_len) == ercOK
               && msg_len > infrequent_msg[i].msg_len)
            msg_expansion_size += msg_len;
      if (     msg_expansion_size == 0 
            || ExpandAreaSL(msg_expansion_size, selector_of(msg_expansion),
                            &msg_expansion) != ercOK)
         msg_expansion = NULL;
   }
}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Descriptor information for the above-defined messages may be returned by
 calling the function below with a message number.  If found, the address of
 the sdType that describes the message is returned.  Callers should be careful
 that their code can handle a null sdType (in case they have asked for a
 message not in the table. */

sdType *standard_msg(unsigned msg_num) {

   unsigned erc, i, j, msg_len;

   for (i = 0; i <= last(frequent_msg); i++)
      if (msg_num == frequent_msg[i].msg_num)
         return((sdType *) &frequent_msg[i].msg);
   for (i = 0; i < last(infrequent_msg); i++)
      if (msg_num == infrequent_msg[i].msg_num) {
         for (j = 1; j <= 2; j++)
            if ((erc = GetMsgUnexpandedLength(infrequent_msg[i].msg_num,
                                              &msg_len)) == ercOK)
               break;
            else if (j == 1)
               initialize_msgs();
         if (erc == ercOK) {
            if (msg_len > infrequent_msg[i].msg_len && msg_expansion != NULL) {
               infrequent_msg[i].msg = msg_expansion;
               offset_of(msg_expansion) +=
                                         (infrequent_msg[i].msg_len = msg_len);
            }
            GetMsgUnexpanded(infrequent_msg[i].msg_num,
                             infrequent_msg[i].msg,
                             infrequent_msg[i].msg_len,
                             &infrequent_msg[i].msg_len);
         }
         break;
      }
   return((sdType *) &infrequent_msg[i].msg);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When an error condition has occurred, display the ending message on the
 screen (and add it to the log file, if one is open).  Then, if a log file is
 in use, obtain the error code message that ErrorExit will display and add it
 to the log file as well.  Finally, quit. */

void exit_with_msg(unsigned erc, unsigned nls_msg_index) {

   char msg[256];
   unsigned junk, msg_len;
   sdType *sd_msg;

   if (nls_msg_index != 0) {
      sd_msg = standard_msg(nls_msg_index);
      if (ExpandLocalMsg(NULL, 0, sd_msg->pb, sd_msg->cb, msg, sizeof(msg),
                         &msg_len, FALSE) == ercOK) {
         OutputBytesWithWrap(msg, msg_len);
         if (journal.bswa != NULL)
            WriteBsRecord(journal.bswa, msg, msg_len, &junk);
      }
   }
   if (journal.bswa != NULL) {
      if (erc != ercOK) {
         CloseMsgFile();
         if (     InitMsgFile(ERC_MSG_FILE, last(ERC_MSG_FILE), NULL, 0,
                              msg_buffer, sizeof(msg_buffer), msg_cache,
                              sizeof(msg_cache)) == ercOK
               && GetMsg(erc, NULL, 0, msg, sizeof(msg), &msg_len) == ercOK)
            WriteBsRecord(journal.bswa, msg, msg_len, &junk);
      }
      CloseByteStream(journal.bswa);
   }
   ErrorExit(erc);

}

pragma Page(1);
private Boolean video_lock = FALSE;

/* This procedure needs an explaination....

Using a signature of 0FFxxh will cause a formfeed and new page in the log.


*/

void log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
             unsigned nls_parms_len) {

   Boolean static delay_page_break = FALSE, force_page_break = FALSE;
   char msg[256];
   unsigned erc, i, j, junk, msg_len;
   sdType *sd_msg;

   sd_msg = standard_msg(nls_msg_index);
   if (ExpandLocalMsg(nls_parms, nls_parms_len, sd_msg->pb, sd_msg->cb, msg,
                      sizeof(msg), &msg_len, FALSE) == ercOK)
      do {
         while (lockset(&video_lock) == TRUE)
            Delay(1);
         if (     journal.video_semaphore == 0
               || journal.video_semaphore == (signature & 0x00FF)) {
            if (signature >> 8) {
               memcpy(&journal.prev_msg[1], msg,
                      journal.prev_msg[0] = msg_len);
               journal.prev_signature = signature & 0x00FF;
            } else if (journal.prev_msg[0] != 0) {
               if (journal.prev_signature == signature) {
                  if (journal.interrupted) {
                     journal.interrupted = FALSE;
                     if ((erc = OutputBytesWithWrap(&journal.prev_msg[1],
                                                    journal.prev_msg[0]))
                           != ercOK)
                        exit_with_msg(erc, 0);
                  }
                  journal.prev_msg[0] = 0;
               } else if (!journal.interrupted) {
                  journal.interrupted = TRUE;
                  PutChar('\n');
               }
            } else
               journal.prev_signature = signature & 0x00FF;
            if ((erc = OutputBytesWithWrap(msg, msg_len)) != ercOK)
               exit_with_msg(erc, 0);
            if (journal.bswa != NULL) {
               for (i = 0; i < msg_len; i++)
                  if (msg[i] == '\n')
                     journal.line++;
               if (journal.line > LINES_PER_PAGE - 6)
                  if (delay_page_break)
                     force_page_break = TRUE;
                  else
                     log_page_header();
               if ((j = _find_char(msg, msg_len, 0xFF)) != msg_len)
                  for (i = j = 0; i < msg_len; i++)
                     if (msg[i] == 0xFF)	/* Strip escape sequences */
                        switch (msg[++i]) {
                           case 'C':
                           case 'I':
                              i += 2;
                              break;

                           case 'S':
                              i += 4;
                              break;

                           case 'F':
                              i += 5;
                              break;

                           default:
                              i += 1;
                        }
                     else if (i == j)
                        j++;
                     else
                        msg[j++] = msg[i];
               if ((erc = WriteBsRecord(journal.bswa, msg, j, &junk)) != ercOK)
                  exit_with_msg(erc, 0);
               if (force_page_break) {
                  force_page_break = FALSE;
                  log_page_header();
               }
               delay_page_break = signature >> 8;
            }
            video_lock = FALSE;
            return;
         }
         video_lock = FALSE;
         Delay(1);
      } while (TRUE);

}

void log_page_header(void) {

   char hdr[256];
   unsigned erc, hdr_len, junk;
   sdType *sd_msg;

   if (journal.bswa == NULL)
      return;
   journal.page++;
   sd_msg = standard_msg(NLS_PAGE_HEADER);
   if (     (erc = ExpandLocalMsg(log_parms, sizeof(log_parms),
                                  sd_msg->pb, sd_msg->cb, hdr,
                                  sizeof(hdr), &hdr_len,
                                  FALSE)) == ercOK
         && (erc = WriteBsRecord(journal.bswa, hdr,
                                 hdr_len, &junk)) == ercOK)
      journal.line = 4;
   else
      exit_with_msg(erc, 0);

}

pragma Page(1);
void vid_only_msg(unsigned nls_msg_index, unsigned signature,
                  sdType nls_parms[], unsigned nls_parms_len) {

   char msg[256];
   unsigned erc, msg_len;
   sdType *sd_msg;

   sd_msg = standard_msg(nls_msg_index);
   if (ExpandLocalMsg(nls_parms, nls_parms_len, sd_msg->pb, sd_msg->cb, msg,
                      sizeof(msg), &msg_len, FALSE) == ercOK)
      do {
         while (lockset(&video_lock) == TRUE)
            Delay(1);
         if (     journal.video_semaphore == 0
               || journal.video_semaphore == (signature & 0x00FF)) {
            if (signature >> 8) {
               memcpy(&journal.prev_msg[1], msg,
                      journal.prev_msg[0] = msg_len);
               journal.prev_signature = signature & 0x00FF;
            } else if (journal.prev_msg[0] != 0) {
               if (journal.prev_signature == signature) {
                  if (journal.interrupted) {
                     journal.interrupted = FALSE;
                     if ((erc = OutputBytesWithWrap(&journal.prev_msg[1],
                                                    journal.prev_msg[0]))
                           != ercOK)
                        exit_with_msg(erc, 0);
                  }
                  journal.prev_msg[0] = 0;
               } else if (!journal.interrupted) {
                  journal.interrupted = TRUE;
                  PutChar('\n');
               }
            } else
               journal.prev_signature = signature & 0x00FF;
            if ((erc = OutputBytesWithWrap(msg, msg_len)) != ercOK)
               exit_with_msg(erc, 0);
            video_lock = FALSE;
            return;
         }
         video_lock = FALSE;
         Delay(1);
      } while (TRUE);

}

pragma Page(1);
void lock_video(unsigned signature) {

   do {
      while (lockset(&video_lock) == TRUE)
         Delay(1);
      if (journal.video_semaphore == 0) {
         journal.video_semaphore = signature;
         video_lock = FALSE;
         return;
      }
      video_lock = FALSE;
      Delay(10);
   } while (TRUE);

}

void unlock_video(unsigned signature) {

   while (lockset(&video_lock) == TRUE)
      Delay(1);
   if (journal.video_semaphore == signature)
      journal.video_semaphore = 0;
   video_lock = FALSE;

}

void display_filename_once() {

   return;

}

pragma Page(1);
void open_video(void) {

   extern char bsVid[130];
   static char buffer[1024];
   unsigned erc;
   sdType *sd_vid;

   sd_vid = standard_msg(NLS_VID);
   if ((erc = OpenByteStream(bsVid, sd_vid->pb, sd_vid->cb, NULL, 0,
                             modeWrite, buffer, sizeof(buffer))) != ercOK)
      exit_with_msg(erc, 0);

}

pragma Page(1);
unsigned open_log_file(void) {

   unsigned long current_time;
   unsigned erc, i, j;
   sdType nls_parms[] = {         NULL, 0,
                         (void *) &sbVerRun[1], sbVerRun[0],
                         (void *) &current_time, sizeof(current_time)};
   sdType *sd_msg, *sd_vid;

   sd_vid = standard_msg(NLS_VID);	/* [Vid] specified redundantly? */
   if (     journal.filespec[0] == 0
         || (   journal.filespec[0] == sd_vid->cb
             && NlsULCmpB(NULL, &journal.filespec[1], sd_vid->pb,
                          sd_vid->cb, &i) == ercOK
             && i == 0xFFFF))
      return(ercOK);
   journal.bswa = (void *) &journal.bswa;	/* Little trick so that... */
   journal.buffer = (void *) &journal.buffer;	/* .....ExpandAreaSL works */
   journal.line = LINES_PER_PAGE + 1;	/* Insure page overflow on first use */
   if ((erc = ExpandAreaSL(BSWA_SIZE, selector_of(journal.bswa),
                           &offset_of(journal.bswa))) == ercOK) {
      if ((erc = ExpandAreaSL(BUFFER_SIZE,
                              selector_of(journal.buffer),
                              &offset_of(journal.buffer))) == ercOK) {
         if ((erc = OpenByteStream(journal.bswa, &journal.filespec[1],
                                   journal.filespec[0], &journal.password[1],
                                   journal.password[0], modeAppend,
                                   journal.buffer,
                                   BUFFER_SIZE)) == ercOK) {
            RgParam(0, 0, &nls_parms[0]);
            sd_msg = standard_msg(NLS_COMMAND_VERSION);
            memset(command_id, ' ', sizeof(command_id));
            ExpandLocalMsg(nls_parms, sizeof(nls_parms), sd_msg->pb,
                           sd_msg->cb, command_id, sizeof(command_id), &j,
                           FALSE);
            for (i = 0; i < j; i++)
               if (command_id[i] == '\n')
                  command_id[i] = ' ';
            GetDateTime(&current_time);
            sd_msg = standard_msg(NLS_DATE_TIME);
            memset(run_time, ' ', sizeof(run_time));
            ExpandLocalMsg(nls_parms, sizeof(nls_parms), sd_msg->pb,
                           sd_msg->cb, run_time, sizeof(run_time), &j, FALSE);
            return(erc);
         }
         ShrinkAreaSL(journal.buffer, sizeof(*journal.buffer));
         journal.buffer = NULL;
      }
      ShrinkAreaSL(journal.bswa, sizeof(*journal.bswa));
      journal.bswa = NULL;
   }
   return(erc);

}

pragma Page(1);
Boolean proceed(void) {

   char cmd, reply[4];
   unsigned reply_len, erc;

   vid_only_msg(NLS_PROMPT, 2, NULL, 0);
   do {
      while ((erc = ENLS_GetChar(NULL, 0, reply, &reply_len,
                         ENLS_KBD_DIRECT_MODE_0)) == ercOK){
         if ((erc == ercOK) && (ENLS_MapCharToStdValue(reply, &cmd) == ercOK)){
            switch (cmd) {
               case ENLS_GO:
                  return(TRUE);

               case ENLS_CANCEL:
                  return(FALSE);

               case ENLS_FINISH: 
                  exit_with_msg(ercOK, 0);      
            }
         Beep();
         }
      Delay(1);
      }
   } while (erc == ercReadOrPeekAlreadyReceived);
   return(FALSE);			/* Unexpected ENLS problem... */

}

pragma Page(1);
Boolean proceed_sel_arch(flag *finish_flag) {

   char cmd, reply[4];
   unsigned reply_len;

   *finish_flag = FALSE;
   vid_only_msg(NLS_PROMPT, 2, NULL, 0);
   while (ENLS_GetChar(NULL, 0, reply, &reply_len,
                       ENLS_KBD_DIRECT_MODE_0) == ercOK) {
      if (ENLS_MapCharToStdValue(reply, &cmd) == ercOK)
         switch (cmd) {
            case ENLS_GO:
               return(TRUE);

            case ENLS_CANCEL:
               return(FALSE);

            case ENLS_FINISH:
               *finish_flag = TRUE;
               return(FALSE);
               
         }
      Beep();
   }
   return(FALSE);			/* Unexpected ENLS problem... */

}

pragma Page(1);
void hex_dump(char *buffer, unsigned length, unsigned blackouts,
              unsigned blackout_floor[], unsigned blackout_ceiling[]) {

   Boolean blackout;
   char class;
   static char hex_chars[16] = "0123456789ABCDEF";
   static char msg[80] = "   NNN:  "
                         "   XX XX XX XX XX XX XX XX XX XX XX XX XX XX XX XX "
                         "   XXXXXXXXXXXXXXXX" "\n";

   unsigned i, j, junk, k, m, n;

   for (i = 0; i < length; i += 16) {
      msg[3] = i / 100 | '0';
      msg[4] = i % 100 / 10 | '0';
      msg[5] = i % 10 | '0';
      m = 12;
      n = 63;
      for (j = i; j < i + 16; j++) {
         blackout = FALSE;
         for (k = 0; k < blackouts; k++)
            if (j >= blackout_floor[k] && j <= blackout_ceiling[k]) {
               blackout = TRUE;
               break;
            }
         if (blackout) {
            msg[m++] = '0';
            msg[m++] = '0';
            m++;
            msg[n++] = '.';
         } else {
            msg[m++] = hex_chars[buffer[j] >> 4];
            msg[m++] = hex_chars[buffer[j] & 0x0F];
            m++;
            if (buffer[j] == 0xFF || NlsClass(NULL, buffer[j], &class))
               msg[n++] = '.';
            else if (class == 4)
               msg[n++] = '.';
            else
               msg[n++] = buffer[j];
         }
      }
      nPrint(msg, sizeof(msg));
      if (journal.bswa != NULL)
         WriteBsRecord(journal.bswa, msg, sizeof(msg), &junk);
   }
   PutChar('\n');
   if (journal.bswa != NULL)
      WriteByte(journal.bswa, '\n');

}

