; file: RmosSwitch_v.asm
;
; CTOS v series V86-based real mode emulator.

; 06/25/87  DR V86 RMOS prototype
; 02/21/90  DR Recreated for v series
; 03/27/90  DR InvalidOpcodeTrap
; 05/09/90  DR CoprocessorTrap
; 09/04/90  JM change rTssRgb87 to rTssrRgb87
; 02/26/91	JF	Move CoprocessorTrap into Crash_all so that it can be called
;				from PC Emulator.
; 09/12/91  DR Gratuitous cleanup in EnterRealMode
; 09/24/91  DR Relocate RealDispatcher from V86Entry_v (now obsolete).  For
;              gp faults, set up task gate in IDT and restart.  Remove
;              IDT update code in CallRealISR
; 09/28/91  DR CallBios
; 10/15/91  DR Set up bootstrap signature in SI/DI in SwitchNub
; 10/18/91  DR PC specific code in SwitchNub
; 12/11/91  DR Crashdump for PCs
; 03/18/92  SG EnterBootRom for SG2000, SG5000; clear dma ear 386i
; 04/10/92  SG reset all dma high EARs, fixes crashdump on B39 > 16M
; 05/13/92  DR new interface to Client Card for autodump/reboot
; 05/20/92  DR in crashdump, dump in 128k chunks rather than 512k - some
;              PCs have memory in 128k increments
; 10/05/92  DR in crashdump, dump in 64k chunks rather than 128k - some
;              PCs have memory in 64k increments
; 10/29/92  DR get SP0 from TSS
; 01/08/93  DR in case using VideoCard, in SwitchNub,
;              put video memory back in first MB
; 02/19/93  DR for crashdump, get disk geometry from BIOS memory
; 02/19/93  FW In EnterRealMode and SwitchNub handle PC case where we want
;              to reboot using Int19 but don't want a crashDump.  See the
;              changes to PSinterface_v.plm too.
; 02/24/93  DR in crashdump, search for a crash dump partition in the 
;              partition table - support up to 64Mb crashdump - improve
;              error message
; 03/08/93  DR Reset DMA in PC crash path so cluster will not corrupt
;              image following reboot from hard disk.  Also, moved
;              video card fix of 1/8 so that it happens for Bootstrap too
;              (this allows messages from realnub to be seen).
; 03/11/93  DR ConvertLba now uses 10 bit cylinder address; PC switchnub
;              turns on A20 wrap
; 03/17/93  DR Turn on A20 wrap AFTER crashdump
; 03/24/93  DR Do not turn on A20 wrap if dumping to cluster
; 03/30/93  FW SG-4000 SCSI Dump.  Use Int13 Fn8 to get disk geometry.
; 04/15/93  JM mask doorbell interrupt before dumping.
; 05/10/93  JA DoInt19 preserve dx value: load after i/o.
; 05/11/93  DR Support string i/o in v86 emulation
; 06/23/93  FW No more finger area; instead the 600h-67FFh area is reserved
;              parameter passing to real mode, debug file info for crash
;              dumps, and hardware configuration information.
; 06/23/93  FW Attempt to correctly ID the Weitek 5186 before we blast port
;              320h.
; 06/28/93  FW OOps, add RM_GenPointer at location 600h for the Knob.
; 07/01/93  JM Oops, RM_reserved is 78 bytes, not 78 words.
;			   Move RelocateVideoRam out of relocated part of SwitchNub.
;			   use RM_paEndMem instead of the BIOS to get memory size.
; 08/05/93  JM support dumps > 16mb.

$MOD386
%SET(ctosv,0FFh)
%SET(debug,0)
%SET(debugComm,0)

$INCLUDE(:f1:Descriptors.mdf)
$INCLUDE(:f1:FadsTypesAsm.edf)
$INCLUDE(:f1:ioaddr1.edf)

; RMOSSwitch parameter area for CTOS III.  Used to pass parameters from 
; protected to real mode prior to entering real mode to take a crash dump.
; NOTE: Also defined as RMDataType in CtosTypes.edf
;       Some fields initialized in  InitMem_p.plm.
RMdata SEGMENT AT 0
ORG 600h
RM_GenPointer   DD ?         ;Pointer to the Gen structure for the Knob
RM_Signature    DW ?         ;Set to 'FW' to id this new area*
RM_Version      DW ?         ;Version, starting at 0, for following structure*
RM_paEndMem     DD ?         ;end of used RAM*
RM_paRAMMax     DD ?         ;end of physical RAM*
                             ;  *-These values set in InitMem_p.plm
RM_reserved     DB 78 DUP (?);reserve these bytes for future crashdump params
; 96 bytes in above area

RM_idtr386Real  DW 3 DUP(?)  ;RMOSSwitch temp storage+
RM_wProcType    DW ?         ;RMOSSwitch temp storage+
RM_portProtectEnable DW ?    ;RMOSSwitch temp storage+
                             ;  +-These values set here in RMOSSwitch and
                             ;    referenced only here (i.e. local storage).

; 106 bytes currently   *** DO NOT GROW THIS TOTAL OVER 127! (600h-67Fh)
;						*** (RmosSwitch is relocated to 680h).
RMdata ENDS


public CallBios
public CallRealCommIsr
public CallRealIsr
public CallVideoRom
public EnterV86
public InvalidOpcodeTrap
public RealDispatcher
public V86Emulator

; publics in support of bootstrap
public EnterRealMode
public SaRealVector
public SaOsVector
public RaOsVector
public SwitchNub
public EndSwitchNub

; utilitiy procedures
public Shl4

; publics for debugging
public V86SystemCall
public V86ReturnFromSystemCall
public V86EnterKernel
public V86EnterOssub
public V86EnterRqInterface

extrn Crash:far
extrn ErrorExit:far
extrn Exit:far
extrn FarDealiasToSr:far
extrn FarAliasIpcSr:far
extrn FarSystemTrapHandler:far
extrn OssubEntry:far
extrn pCEntry:far
extrn RqInterface:far


; these sg literals must be the same as those in descriptors.edf
ercNotImplemented equ 7

%define(Data32) (	db 66h
)
%define(addr32) (	db 67h
)
rTssCr3          equ [28]
rTssrRgb87 equ 0B2h
rTss386SS0 equ 8
rTss386SP0 equ 4

; These fields are reused from the CTOS II RMWA
rTss386wSpSave equ 68h  ; Stores protected mode stack pointer while
rTss386wSsSave equ 6Ah  ; client is executing real mode BIOS.
rTss386saSS0   equ 72h  ; Real mode sa for BIOS stack - reused rmwa_wFLReal
sV86Frame  equ 36

NewGen equ 9
b38lcw equ 14
SG5000 equ 15
PCAT equ 16
maxRetries       EQU 5

;386i equates 
dmaEarEnableNewGen equ 820Ch
dmaEarHighBaseNewGen equ 0bc00h	 

;b38lcw equates 
b38MemIndex   equ   0DH	
SeqIndexReg   equ  03c4h
VidToHiAddr   equ  0320h 

; sgen equates
lShadowRomEnableReg equ 0C02h


saSaveArea equ 9f60h ; also in RealNub.asm
atPSysTime equ 240h

DGroup group data
data segment word public 'data'
extrn pRgoGdtLink:dword
extrn rgSgAsib:word
extrn paGlobalPageMap:dword
extrn protectedModeEnabl:word
extrn plaMemMax:dword
extrn processorType:byte
extrn oCoprocessor:word
extrn sgTssGpFault:word
extrn sgTssIntLast: word
data ends

RmosSwitch_code segment public 'code'
assume cs:RmosSwitch_code, ds:nothing, es:nothing
SwitchNub proc near

; SKULL AND CROSSBONES - SwitchNub MUST be the first procedure in this
; code segment because relative offsets from the beginning of SwitchNub
; must be the same as relative offsets from the beginning of the link
; time code segment.
;
; SwitchNub gets relocated by VBootstap, which places it at the end of the
; boot image, or EnterBootrom, which places it in low memory.  The size of
; SwitchNub is determined using the EndSwitchNub label at the end of
; the procedure.  SwitchNub contains code to reset the
; processor protect mode bit and, on a 386i, code to reenable the
; bootrom.  These actions must be done from real mode, consequently,
; this code must be in the first megabyte when it executes.  
;
; There is no stack until one is set up.

	xor ax, ax
	mov ds, ax

; restore base and limit of real mode interrupt vector table
assume ds:RMdata
	lidt RM_idtr386Real
	mov  cx, RM_wProcType
	cmp  cl,PCAT
	je   DoPc
	jmp	 NotPc
	

;*******************************************************
;     PC
;*******************************************************

saPartitionTable equ 0B0h  ; contract with LowMem_all.asm
saSegBase equ 1000h
cSectors1Mb  equ 800h
cSectors640k equ 500h
cSectors64k equ 80h
c64kPerMeg equ 16
cDwordsPer64k equ 4000h
cSectorsPerDigit equ 80h
maskfTryClusterDump equ 2

portWeitekSequencer equ 3c4h
portWeitekData equ 320h
kbdControlRegPCAT  equ 64H ;   8042
kbdDataRegPCAT     equ 60H

localSectorsPerTrack   equ [bp]
localTracksPerCylinder equ [bp+2]
localLbaCX             equ word ptr [bp+4]
localLbaDX             equ word ptr [bp+6]
localc64kExtMem        equ word ptr [bp+8]
localcSectors          equ word ptr [bp+10]
localDigitCount        equ word ptr [bp+12]
localDigit             equ word ptr [bp+14]
localDX                equ word ptr [bp+16]
localES                equ word ptr [bp+18]
cbLocals               equ 20

spInitial              equ 0f8h
bpInitial              equ spInitial-cbLocals

;--------  GDT for moving memory ------
rgGdt	db 8 dup (0) ; unused
; 8 - gdt alias
gdtr	dw 27h       ; limit
gdt_lo	dw 0
gdt_hi	db 0
		db 0
		dw 0
; 10 - source descriptor
		dw 0FFFFh    ; limit
src_lo	dw 0
src_hi	db 10h
		db 92h       ; access
		db 0CFh
src_msb db 0
; 18 - destination descriptor
		dw 0FFFFh    ; limit
trg_lo	dw 0
trg_hi	db 1
		db 92h       ; access
		dw 0CFh
; 20 - identity descriptor for real mode reentry
		dw 0FFFFh    ; limit
		dw 0
		db 0
		db 92h       ; access
		dw 0

DoPc:
	mov ax,30h       ; set up stack in BIOS stack segment
	mov ss,ax        ; ss = 30 => 30:0 = 300 => lower stack limit
	mov sp,spInitial ; ss:sp = 30:F8 = 3F8 => upper stack limit
				     ; Note: GDT pointer for Debug File is at 3F8
	mov bp,bpInitial

	or   ch, ch  ; if 0 then jump to StartOS .. bootstrap.run has loaded
	             ; a CTOS image for us to execute.  Don't need to do Int19.
	jz   StartOS

; doing a warm BIOS boot and maybe a crashdump too!

	push bx       ; ES from Crash
	push cx          ; CH = fDoCrashDump, CL = processorType

	mov ax, 0003h  ; set video mode to mode 3 (ah = 0)
	int 10h

; put cursor at row 24, column 0
	mov ax, 0200h
	xor bx, bx
	mov dx, 1800h
	int 10h

; shut down DMA, in case cluster is still going (if this is not done,
; cluster can corrupt image after boot from hard disk)
	xor  al,al
	mov  dx,0Dh     ; Reset DMA1 controller (write Master clear register)
	out  dx,al
	jmp  .+2
	mov  dx,08h     ; Clear DMA1 command register
	out  dx,al
	jmp  .+2
	mov  dx,0DAh     ; Reset DMA2 controller (write Master Clear register)
	out  dx,al
	jmp  .+2
	mov  dx,0D0h     ; Clear DMA2 command register
	out  dx,al
	jmp  .+2

	pop cx        ; CH = fDoCrashDump, CL = processorType
	shr ebx, 16
	push bx       ; DX from Crash
	mov bx, cx

; copy the osBuffer to the save area - Note: RealNub copies it back to
; a new osBuffer
	xor ax, ax
	mov ds, ax
	lds si, dword ptr [atpSysTime]
	mov ax, saSaveArea
	mov es, ax
	xor di, di
	mov cx, 12
	cld
	repnz movsw

;*******************************************************
; Attempt a crashdump
;*******************************************************

lPartitionTableSig  equ 0AA55h
lsPartitionEntry    equ 10h
loPartitionTable    equ 1BEh
loPartitionTableSig equ 1FEh
lCrashDumpSig       equ 0CDh
lrSystemCode        equ 4
lrLVAlo             equ 08h
lrLBAhi             equ 0Ah
lrPartitionSize     equ 0Ch

lrTracksPerCylinder equ 2
lrSectorsPerTrack   equ 0Eh

	cmp  bh, 1    ; if 1, then skip dump, reboot using Int19
	jz   DoReboot

; display  message
	mov si, offset rgbDumping
	call OutAsciiz
	call OutLf

; Must unmask interrupts so that ADAPTEC 7770 SCSI BIOS can run on SG4000
; Need to do this intelligently and only unmask the needed IRQ; for now
; unmask them all so we can dump.
	mov dx, 0A1h       ;
	in al, dx          ;
	jmp $+2
	and al, 0e1h       ;unmask IRQ9, 10, 11, 12 for SCSI    FW_Int13_Fn8
	out dx, al         ;
	jmp $+2
	mov dx, 21h        ;
	in al, dx          ;
	jmp $+2
	and al, 0FBh       ;enable IRQ2 cascade to second 8259         FW_Int13_Fn8
	out dx, al         ;
	sti                ;allow SCSI controller to interrupt

; read partition table
	mov ax, saPartitionTable
	mov es, ax
	mov bx, 0E30h
	push bx          ; localDigit
	xor bx, bx
	push bx          ; localDigitCount
	push bx          ; localcSectors

	mov di, maxRetries
	mov dx, 80h      ; dh=head#=0, dl=drive id=80h
	mov cx, 1      ; ch=track#=0, cl=sector#=1
_read_PartitionTable:
   	mov ax,0201h    ; al=02=sector count, ah=2=read command
	int 13h         ; BIOS read command
	jnc _GotPartitionTable
	dec di
	jnz _read_PartitionTable

	mov si, offset rgbIOFailed
	jmp DoRebootWithMessage
_BogusCrashDumpFile:
	mov si, offset rgbBogusCrashDumpFile
	jmp DoRebootWithMessage

_GotPartitionTable:  ; read happened, so check signatures
	mov ax, es:[loPartitionTableSig]
	cmp ax, lPartitionTableSig
%if (%debugComm) then (
	jmp short _BogusCrashDumpFile
)fi
	jne _BogusCrashDumpFile	
;	mov ax, es:[1B4h] - 12.3 FormatDisk no longer inserts the CT signature
;	cmp ax, 04354h
;	jne _BogusCrashDumpFile

; find the crash dump partition - this loop depends on the partition
; table signature as a sentinel

	mov si, loPartitionTable

FindCrashPartition:
	mov ax, es:[si]
	cmp ax, lPartitionTableSig
	je _BogusCrashDumpFile ; didn't find crashdump partition

	mov al, es:[si+lrSystemCode]
	cmp al, lCrashDumpSig
	je CrashPartitionFound
	add si, lsPartitionEntry
	jmp short FindCrashPartition 

CrashPartitionFound:
; verify that the partition is big enough
	xor ax, ax
	mov ds, ax
%IF (0) THEN (
	xor eax, eax  ; will be using high bits of eax
	mov ax, 8800h
	push si     ; preserve offset to partition table entry
	int 15h     ; get extended memory size in AX in k bytes - can handle
                ; up to 64 Mb
	pop si
)ELSE(
	mov eax, RM_paEndMem
	shr eax, 10		; bytes -> KBs
	sub eax, 1024	; size of extended memory = size of RAM - 1mb
)FI

	mov edx, eax
	shr edx, 6
	push dx     ; localc64kExtMem

	shl eax, 1   ; convert to sectors
	add eax, cSectors1Mb

	mov dx, saPartitionTable
	mov es, dx
	cmp es:[si+lrPartitionSize], eax ; size in sectors
	jb _BogusCrashDumpFile

; store partition address in local variables
	push word ptr es:[si+lrLBAhi] ; lba DX
	push word ptr es:[si+lrLVAlo] ; lba CX


; store disk geometry in local variables
	mov dx, 80h        ;drive id=80h                               FW_Int13_Fn8
	mov ax, 0800h      ;ah=8=Return Disk Drive Parameters command  FW_Int13_Fn8
	int 13h            ;Invoke the disk (or SCSI) BIOS             FW_Int13_Fn8
	jnc GotTheGeometry ;jif got geometry - otherwise skip the dump FW_Int13_Fn8
	                   ;else we may trash the system disk.         FW_Int13_Fn8
	mov si, offset rgbBIOSerrorInt13Fn8 ;                          FW_Int13_Fn8
	jmp DoRebootWithMessage;                                       FW_Int13_Fn8

GotTheGeometry:        ;                                           FW_Int13_Fn8
;	les bx, dword ptr [41h*4] ; pointer to disk geometry table is at int 41h
;	mov al, es:[bx+lrTracksPerCylinder] ; tracks per cylinder
	inc dh             ;convert from "last track" to tracks/cyl    FW_Int13_Fn8
	mov al, dh         ;tracks per cylinder - from Int13 call      FW_Int13_Fn8
	xor  ah, ah        ;                                           FW_Int13_Fn8
	push ax            ;ConvertLba will get this from the stack    FW_Int13_Fn8

;	mov al, es:[bx+lrSectorsPerTrack] ; sectors per track
	mov al, cl         ;sectors per track - from Int13 call        FW_Int13_Fn8
	and al,3fh         ;discard upper two bits (part of cyl info)  FW_Int13_Fn8
	push ax            ;ConvertLba will get this from the stack    FW_Int13_Fn8

; ASSERT - Local variables initialized up and addressable

;****************************************************************
; Dump first 640k
;****************************************************************
	mov cx, localLbaCX
	mov dx, localLbaDX
	mov si, cSectors640k
	xor bx,bx      ; segBase = 0		
	mov es,bx
	call WriteSpan

;****************************************************************
; Adjust lba for extended memory
;****************************************************************
	add localLbaCX, cSectors1Mb
	adc localLbaDX, 0
	inc localDigit
	mov localDigitCount, c64kPerMeg
;****************************************************************
; Dump extended memory
;****************************************************************

_DumpExtMem:

	call MoveExtMemory

	mov cx, localLbaCX
	mov dx, localLbaDX
	mov si, cSectors64k
	mov bx, saSegBase
	mov es,bx
	call WriteSpan

	dec localc64kExtMem
	jz _CrashDumpDone

	add localLbaCX, cSectors64k
	adc localLbaDX, 0
	add src_hi, 1 ; add 64k to source base
	adc src_msb, 0	; support >16mb dumps

	dec localDigitCount
	jnz _DumpExtMem	
	mov localDigitCount, c64kPerMeg
	inc localDigit
; handle wrap from 9 to A or from F to 0
	cmp localDigit, 0E3Ah ; '9' + 1
	jne _CheckLocalDigit
	mov localDigit, 0E41h ; 'A'
	jmp short _DumpExtMem
_CheckLocalDigit:
	cmp localDigit, 0E47h ; 'F' + 1
	jne _DumpExtMem
	mov localDigit, 0E30h ; '0'
	jmp short _DumpExtMem

_CrashDumpDone:
	call OutLf
	xor ax, ax  ; will be ORed with ES on stack to indicate to Client Card
	push ax     ; that a dump is not necessary 
	jmp DoReboot

;****************************************************************
; OutLf - display a CR and LineFeed
;****************************************************************
OutLf:
	pusha
	xor bx, bx      ;
 	mov ax, 0E0Dh
	int 10h
	mov ax, 0E0Ah
	int 10h
	popa
	ret
;OutLf endp

;****************************************************************
; OutAsciiz - display zero terminiated string at cs:si
;****************************************************************
OutAsciiz:
	pusha
	mov ax, cs
	mov ds, ax
OutAsciizLoop:
	lodsb
	or al, al
	jz OutAsciizExit
	mov ah, 0Eh
	push si
	int 10h
	pop si
	jmp OutAsciizLoop
OutAsciizExit:
	popa
	ret
;OutAsciiz endp


;****************************************************************
; MoveExtMemory - moves 64k of extended memory to segment at 1000:0
;
; Assume:
;   gdt source address is initialized
;****************************************************************

MoveExtMemory:
; reinit target base
;	xor eax, eax
;	mov ax, saSegBase
;	shl eax, 4
;	mov trg_lo, ax
;	shr eax, 16
;	mov trg_hi, al

; initialize gdtr base
	cli                    ;No interrupt during this since no IDT! FW_Int13_Fn8
	xor eax, eax
	mov ax, cs
	shl eax, 4
	add eax, offset rgGdt
	mov gdt_lo, ax
	shr eax, 16
	mov gdt_hi, al

	lgdt gdtr				; load the gdt base and limit
	smsw ax					; set protected mode bit
	or ax, 1
	lmsw ax 
	jmp .+2					; clear bogus pre-fetch queue

; cs and ss reference same addresses as before protected switch
	cld
	xor esi, esi
	xor edi, edi
	mov ax, 10h
	mov ds, ax
	mov ax, 18h
	mov es, ax
	mov ecx, cDwordsPer64k
	db 66h
	db 67h
	db 0F3h ; rep
	db 0A5h ; movsw

; go real
	mov ax, 20h; reload segment regs that were used
	mov es, ax
	mov ds, ax

; set PE bit to zero
	mov eax, cr0
	and  al, 0FEh
   	mov cr0, eax

	jmp .+2
	sti                    ;SCSI needs interrupt enabled           FW_Int13_Fn8
	ret

; MoveExtMemory endp


;****************************************************************
; WriteSpan - writes contiguous sectors to disk
; Assume:
;   SI = # sectors to write
;   DX:CX = lba to write to
;   ES = seg base to write from
;****************************************************************

WriteSpan:
	push cx         ; lba
	push dx
	call ConvertLba ; convert to C/H/S format 

;****************************************************************
; cSectors = sectorsPerTrack - sector # + 1
;****************************************************************
	xor ax,ax       
	mov al,localSectorsPerTrack
	sub al,cl 
	inc al          ;  cSectors

;****************************************************************
; if cSectors > size then cSectors = size
;****************************************************************
	cmp si,ax       ;  check if it is the end of the job 
	jae _JustOnce   ;  if it just partial 
	mov ax,si       ;  load the left over sectors
_JustOnce:
	push cx         ;  int 13 params
	push dx         ;
	push ax         ;  cSectors

;****************************************************************
; If span would cross 64K boundary in memory, break it up.
;
; segNext = (cSectors * 20h) + segBase
;****************************************************************
	shl ax, 5	    ;  cSectors * 20h
	mov dx,es       ;  segBase 
	add ax,dx       ;  segNext

;****************************************************************
; if high4(segBase) <> high4(segNext) then do
;     cOver = (segNext AND 0FFFh) / 20h
;     cSectors = cSectors - cOver
;     end
;****************************************************************
	push ax         ;  segnext
	shr ah,4        ;  high4(segNext)
	shr dh,4        ;  high4(segBase)
	cmp ah,dh       ;
	pop ax          ;  segNext
	je _GoWrite     ;  

	and ax,0fffh    ;  cOver = (segNext AND 0FFFh) / 20h
	shr ax,5  	    
	mov cx,ax

	pop ax
	sub ax,cx       ; cSectors = cSectors - cOver
	jmp short _OK
_GoWrite:
	pop ax          ; cSectors

_OK:
	pop dx          ; int 13 params
	pop cx
	xor bx,bx       ; buffer offset
	mov ah,03h      ; ah=2=writecommand
	mov di, maxRetries
_do_write_dumpfile:
	pusha
	int 13h         ; disk read
	popa
	jnc _write_dumpfile_OK
	dec di
	jnz _do_write_dumpfile

	call OutLf
	mov si, offset rgbIOFailed
	jmp DoRebootWithMessage

_write_dumpfile_OK:
	xor ah,ah

;****************************************************************
; Assume:
;   AX    = cSectors
;   ES:BX = buffer
;   SI    = size
;   lba on stack

;****************************************************************
; Display a digit for every cSectorsPerDigit sectors dumped
;****************************************************************
	push ax         ; cSectors

	add localcSectors, ax
	mov ax, localcSectors
	cmp ax, cSectorsPerDigit
	jb _adjust_size
	sub localcSectors, cSectorsPerDigit
	mov ax,localDigit
	pusha
	xor bx, bx
	int 10h         ; video
	popa

;****************************************************************
; size = size - cSectors
;****************************************************************
_adjust_size:
	pop ax         ; cSectors
	sub si,ax
	jz _done

;****************************************************************
; segBase = segBase + cSectors * 20h
;****************************************************************
	mov bx, ax
	shl bx,5       ; cSectors * 20h
	mov cx,es      ; segBase
	add cx,bx
	mov es,cx      ; new segBase

;****************************************************************
; Adjust lba
;****************************************************************
	pop dx          ; lba = dx:cx
	pop cx
	add cx, ax      ; lba = lba + cSectors
	adc dx, 0
	jmp WriteSpan
_done:
	pop dx
	pop cx
	ret
; WriteSpan endp

;****************************************************************
;                     ConvertLBA
; On input, DX:CX holds 24-bit Logical Block Address (LBA)
;
; On output, registers set up for INT 13
;    CH - low 8 bits of 10 bit cylinder number
;	 CL - bits 0-5: sector number; bits 6-7, high 2 bits of cyl num
;    DH - track number
;    DL - drive id (from localSaveDX)
;****************************************************************

ConvertLba proc near
	mov ax,cx
	div localSectorsPerTrack
	inc dx          ; dx contains the sector count
	mov cl,dl       ; 
	and cl,03Fh     ; six bit sector count
; assert: CL = bits 0-5: sector number
	or ax, ax
	je _nodiv
	xor dx, dx      ; dx:ax is the track count
	mov bx,localTracksPerCylinder
	div bx          ; divide by trackspercylinder
; quotient  is AX --> CH (cylinder number, low 8 bits) CL (high 2 bits)
; remainder is DX --> DH (track number)
	shl dx, 8       ; get the remainder as the cylinder number
; assert: DH = track number
	mov ch, al      ; get the quotient as the cylinder number
; assert: CH = low 8 bits of 10 bit cylinder number
	shr ax, 2
	and al, 0C0h
	or cl, al
; assert: CL =bits 0-5: sector number; bits 6-7, high 2 bits of cyl num
	jmp short _setDriveId
_nodiv:
	mov dh,ah       ; get the remainder as the track number
	mov ch,al       ; get the quotient as the cylinder number
_setDriveId:
	mov dl, 80h     ; set drive id for int 13
	ret
ConvertLba endp

;****************** END CRASHDUMP **************************

%if(%debug)then(
DisplayAX:
	pusha
	mov cx, 4
	mov dx, ax
DisplayNext:
	rol dx, 4
	mov ax,dx
	and ax, 0fh
	cmp ax, 0Ah
	jae DisplayHexDigit
	add al, 30h
	jmp short DoDisplay
DisplayHexDigit:
	add al, 37h
DoDisplay:
	push dx
	mov ah, 0Eh
	int 10h         ; video
	pop dx
	loop DisplayNext
	mov ax, 0E2fh
	int 10H
	popa
	ret
;DisplayAX endp
)fi

WaitControllerReady proc near
	mov dx, kbdControlRegPCAT
@WCR:
	in al, dx
	test al, 02h
	jnz @WCR
	ret
WaitControllerReady endp

; crash dump to hard disk failed

DoRebootWithMessage:
	call OutAsciiz
	call OutLf
	mov ax, maskfTryClusterDump ; fTryClusterDump = true (in bit 1)
	push ax    
	mov si, offset rgbAutoDump
	jmp short DoRebootAlt
DoReboot:
	xor ax, ax ; fTryClusterDump = false (in bit 1)
	push ax    
	mov si, offset rgbRebooting
DoRebootAlt:
	call OutAsciiz
	call OutLf

	mov ax, 80h
DelaySome:
	mov cx, 0ffffh
	loop $
	dec ax
	or  ax, ax
	jnz DelaySome

	pop bx           ; fTryClusterDump in bit 1
	test bx, maskfTryClusterDump
	jnz DoInt19

; Poke the 8042 keyboard controller to turn ON 1Mb wrap; Gate A20 
	call WaitControllerReady
; cmnd write output port */
	mov dx, kbdControlRegPCAT
	mov ax, 0D1h
	out dx, al
	call WaitControllerReady
; Enable A20 
	mov dx, kbdDataRegPCAT
	mov al, 1
	out dx, al
	call WaitControllerReady

DoInt19:
; unmask timer 0 interrupt - so Client Card delay code will work
; unmask keyboard output buffer full interrupt
	cli
	mov dx, 21h
	in al, dx
	jmp .+2
	and al, 0FCh
	out dx, al

; Do this after i/o above so dx preserved
	mov dx, localDX  ; wsNNN
	mov ax, localES  ; ES value for Client Card
	or ax, bx
	mov es, ax

	mov si, 'RL'
	mov di, 'DG'
	sti ; let timer generate interrupts
	int 19h  

%if(0)then(
; reset via keyboard microcontroller ------------
	mov dx, 64h
@1:	in al, dx
	test al, 2
	jnz @1

	mov al, 0D1h
	out dx, al
	
@2:	in al, dx
	test al, 2
	jnz @2

	mov dx, 60h
	mov al, 0DEh
	out dx, al
	jmp .+0
; -----------------------------------------------
)fi


NGen:
;
; reset system protected mode bit
	xor ax,ax
	mov dx, RM_portProtectEnable
	out dx, al

; on 386i, enable bootrom
	cmp  cl, NewGen
	jne StartOs
	mov dx, parityEnablePortNewGen
	out dx, al
	jmp $+2

; clear the dma ear since the 386i bootrom doesn't do it. crashdumping fails on
; 386i w > 16mb if the ear is non-zero. -JM 02/05/92
	mov dx, dmaEarEnableNewGen
	out dx, ax
	jmp $+2

; clear all high dma ears also:  0bc00h - 0bc0eh    - SG
	mov cx, 8
	mov dx, dmaEarHighBaseNewGen	
resetDmaEar:
	out dx, al				
	add dx, 2
	loop resetDmaEar			

StartOS:
	mov es, bx				; restore ES for bootrom entry
	shr ebx, 16
	mov dx, bx				; restore DX for bootrom entry
	mov di, 'JM'			; signatures recognized by realNub
	mov si, 'JF'
	mov  cx, RM_wProcType  ; restore to cx

	cmp  cl,SG5000
	jne  ByeBye
	or   ch, ch				; if fEnterBootrom=0 then jump to target code
	jz   ByeBye				;  otherwise, reset the system to enterbootrom

	mov   bx, ES 			; get parameter off stack
	test  bx, 1h			; wstype set?  
	jz    testDisk
	test  bx, 10h			; Yes, so test for nodump flag
	jnz   NoAutoDump
	jmp   AutoDump
testDisk:
	test  bx, 02h
	jz    NoAutoDump
AutoDump:
	mov   al, 0bfH			; address 3fh + bit 7 set (no nmi)
	out   RTCmosAddressReg, AL
	jmp   $+2
	mov   al, 0aah			; auto dump signature
	out   RTCmosDataReg, AL
	jmp   $+2
	
NoAutoDump:
	mov   dx, lShadowRomEnableReg
	xor   al, al			; ExRomEn#=0, RomEn#=0
	out   dx, al
	jmp   $+2

	mov   dx, protectedModeEnablSGEN;
	mov   al, 3h			; system reset
	out   dx, al			; should not return
	jmp .+0 


ByeBye:
	db 0EAh
; The os vector is fixed up by VBootstrap to be the starting address
; of the bootstrap image.
RaOsVector label far
	dw 0
SaOsVector label far
	dw 0

rgbRebooting:
	db 'Rebooting ...',0
rgbAutoDump:
	db 'Dumping to cluster...',0
rgbDumping:
	db 'Dumping to disk...',0
rgbBIOSerrorInt13fn8:
rgbIOFailed:
	db 'I/O failed',0
rgbBogusCrashDumpFile:
	db 'Crashdump not found or too small',0

%SET(oSwitchNub, $)
%IF (%oSwitchNub GT (0b00h-680h)) THEN (
; error - the OS uses the area at 0B00h to read the partition table, therefore 
; the size of the relocated portion of SwitchNub (used by the PC) must be less 
; than 0b00h-680h. 
	Error
)FI



NotPc:
	cmp  cl, b38lcw					; if b38lcw:  turn off A20,
	je   B38lcwReset				; enable video ram, bootrom
	cmp  cl,SG5000
	jne  NGen
	mov  dx, protectedModeEnablSgen	; Mask A20
	mov  al, 0
	out  dx, al
	jmp  StartOs

B38LcwReset:
	mov  al, cmdWrite82106
	out  Reg82c106Cmd, al			; turn off A20
WaitCmdTaken1:
	in   al, Reg82c106Cmd
	and  al, 2
	jne  WaitCmdTaken1				; Wait for controller to take cmd
	mov  al, GetReal
	out  Reg82c106Data, al
WaitCmdTaken2:
	in   al, Reg82c106Cmd
	and  al, 2
	jne  WaitCmdTaken2	

; enable video ram and bootrom 0a0000h - 0fffffh
	out  EnableMemCntlB38lcw, AL	; dummy write enables memory config
	jmp  $+2						;  regs
	mov  ah, b38MemIndex			; first index into shadow memory
	mov  cx, 6						; 6-64k memory chunks
RemoveShadow:
	mov  al, ah						; AAXS (0dh) - FAXS (12h)
	out  MemCntlIndexB38lcw, al		; Out index register
	jmp  $+2
	xor  al, al
	out  MemCntlRegB38lcw, al		; disable 64k
	inc  ah
	loop RemoveShadow

	out  DisableMemCntlB38lcw, al	; disable access to config regs
	jmp  $+2
	mov  dx, SeqIndexReg			; enable CT Regs on vga chip
	mov  al, 08h
	out  dx, al
	jmp  $+2
	mov  dx, VidToHiAddr			; change video memory to low mem
	xor  al, al
	out  dx, al
	jmp  StartOs



EndSwitchNub label far
SwitchNub endp


v86_errorCode equ word ptr [bp+2]
v86_eip       equ word ptr [bp+6]
v86_cs        equ word ptr [bp+10]
v86_efl       equ word ptr [bp+14]
v86_eflMsb    equ word ptr [bp+16]
v86_esp       equ word ptr [bp+18]
v86_ss        equ word ptr [bp+22]
v86_es        equ word ptr [bp+26]
v86_ds        equ word ptr [bp+30]
v86_fs        equ word ptr [bp+34]
v86_gs        equ word ptr [bp+38]

v86NoErrorCode_eip       equ word ptr [bp+2]
v86NoErrorCode_cs        equ word ptr [bp+6]
v86NoErrorCode_eflMsb    equ word ptr [bp+12]

RealDispatcher proc far
EnterV86 label far
; First code executed when V86 Rmos runs.
; Switch to level 0 stack
	str ax
	sub ax, 8h
	mov es, ax
	mov ax, word ptr es:[rTss386SS0]
	mov bx, word ptr es:[rTss386SP0]
	sub bx, sV86Frame
	mov ss, ax
	mov sp, bx
	db 66h
	iRet
RealDispatcher endp

CallVideoRom proc far
	mov ax, ercNotImplemented
	ret
CallVideoRom endp

AddressV86SsSp proc near
; compute the effective address of the V86 SS:SP and store in ebx

	xor ebx, ebx
	mov bx, v86_ss
	shl ebx, 4
	add ebx, v86_esp
	ret
AddressV86SsSp endp

AddressV86Target proc near
	and edi, 0FFFFh
	xor ebx, ebx
	mov bx, v86_es
	shl ebx, 4
	add edi, ebx
	ret

AddressV86Target endp

AddressV86Source proc near
	and esi, 0FFFFh
	xor ebx, ebx
	mov bx, v86_ds
	shl ebx, 4
	add esi, ebx
	ret
AddressV86Source endp

FetchPort proc near
	xor dx, dx
	inc ebx
	mov dl, byte ptr [ebx]
	inc v86_eip
	ret
FetchPort endp

EnterGpFault proc near

; Set up IDT for GP fault task and restart the fault.

	push ds
	mov bx, DGroup
	mov ds, bx
assume ds:DGroup
	mov cx, sgTssGpFault
	mov bx, sgIdt
	mov ds, bx
assume ds:nothing
	cli
	mov word ptr ds:[6Ah], cx
	mov byte ptr ds:[6Dh], accessTaskGate
	pop ds
	jmp LeaveEmulator
EnterGpFault endp

V86Emulator proc far

maskZeroIPL   equ 0CFFFh ; iopl=0,
mask8086Flag  equ 8000h  ; bit 15 = 1 so FProtectedMode
                         ; will think chip is a 8086
miPushF  equ 9Ch
miPopF   equ 9Dh
miInsb   equ 6Ch
miInsw   equ 6Dh
miOutsb  equ 6Eh
miOutsw  equ 6Fh
miInt    equ 0CDh
miIRet   equ 0CFh
miInBIm  equ 0E4h
miInWIm  equ 0E5h
miOutBIm equ 0E6h
miOutWIm equ 0E7h
miInB    equ 0ECh
miInW    equ 0EDh
miOutB   equ 0EEh
miOutW   equ 0EFh
miLock   equ 0F0h
miRepz   equ 0F3h
miHlt    equ 0F4h
miCli    equ 0FAh
miSti    equ 0FBh
miCSSeg  equ 03Eh ; coded after LOCK prefix in LOCK OUT DX, AL
initFlagV86Msb equ 2

	push bp
	mov bp, sp
	push ebx
	push cx
; check for non-v86 GP fault 
	test v86_eflMsb, initFlagV86Msb
	jnz V86Ok
	jmp short EnterGpFault
V86Ok:
	mov bx, v86_errorCode
	or bx, bx
	jnz EnterGpFault

	mov bx, sgZero
	mov ds, bx
	mov es, bx
assume ds:nothing, es:nothing

; address opcode
	xor ebx, ebx
	mov bx, word ptr v86_cs
	shl ebx, 4
	add ebx, v86_eip

; fetch instruction and increment ip
	mov cl, byte ptr [ebx]
	inc v86_eip

EmulateInt:
	cmp cl, miInt
	jne EmulatePushf

; ebx addreses opcode
; fetch int number and increment ip again
	push dx
	inc ebx
	mov dl, byte ptr [ebx]
	inc v86_eip

	cmp dl, 49h
	je V86RqInterfaceEntry
	cmp dl, 4Ah
	je V86OssubEntry
	cmp dl, 48h
	je V86KernelEntry
; int 4B and 4C are special returns to the emulator
	cmp dl, 4Bh
	je ReturnFromRealIsr
	cmp dl, 4Ch
	je ReturnFromBios

EmulateRealInt:
; push v86 mode flags, cs, ip on v86 mode stack
	call AddressV86SsSp

	mov cx, v86_efl
	or  cx, mask8086Flag
	dec ebx
	dec ebx
	mov word ptr [ebx], cx

	mov cx, v86_cs
	dec ebx
	dec ebx
	mov word ptr [ebx], cx

	mov cx, v86_eip
	dec ebx
	dec ebx
	mov word ptr [ebx], cx

; adjust v86 stack pointer
	sub v86_esp, 6

; fetch cs:ip for the interrupt level in cl and store
; in v86 cs:ip
	xor bx, bx
	mov bl, dl
	shl bx, 2
	mov cx, [bx]
	mov v86_eip, cx
	mov cx, [bx+2]
	mov v86_cs, cx

; turn off real mode interrupts
	and v86_efl, 0FDFFh

	jmp PopDxLeaveEmulator

V86KernelEntry:
	xor di, di
	jmp V86SystemCall
V86RqInterfaceEntry:
	mov di, 1
	jmp V86SystemCall
V86OssubEntry:
	mov di, 2
	jmp V86SystemCall

EmulatePushf:
	cmp cl, miPushf
	jne EmulatePopf

	call AddressV86SsSp
	mov cx, v86_efl
	or cx, mask8086Flag
	dec ebx
	dec ebx
	mov word ptr [ebx], cx
	sub v86_esp, 2
	jmp LeaveEmulator

EmulatePopf:
	cmp cl, miPopf
	jne EmulateIRet

	call AddressV86SsSp
	mov cx, word ptr [ebx]
	and cx, maskZeroIPL ; in case user has modified stack
	mov v86_efl, cx
	add v86_esp, 2
	jmp LeaveEmulator

EmulateIRet:
	cmp cl, miIRet
	jne EmulateInB

	call AddressV86SsSp

	mov cx, word ptr [ebx]
	mov v86_eip, cx

	inc ebx
	inc ebx
	mov cx, word ptr [ebx]
	mov v86_cs, cx

	inc ebx
	inc ebx
	mov cx, word ptr [ebx]
	and cx, maskZeroIPL ; in case user has modified stack
	mov v86_efl, cx

; adjust v86 stack pointer
	add v86_esp, 6

	jmp LeaveEmulator


EmulateInB:
	cmp cl, miInB
	jne EmulateOutB

	in al, dx
	jmp LeaveEmulator

EmulateOutB:
	cmp cl, miOutB
	jne EmulateInW

	out dx, al
	jmp LeaveEmulator

EmulateInW:
	cmp cl, miInW
	jne EmulateOutW

	in ax, dx
	jmp LeaveEmulator

EmulateOutW:
	cmp cl, miOutW
	jne EmulateRepz

	out dx, ax
	jmp LeaveEmulator

EmulateRepz:
	cmp cl, miRepz
	jne EmulateSti

	xor ecx, ecx ; restore program cx in ecx
	pop cx
	push 0       ; cx will be zero after the repz operation

	inc ebx
	inc v86_eip
EmulateStringIO:
	mov bl, byte ptr [ebx]

EmulateOutsb:
	cmp bl, miOutsb
	jne EmulateOutsw

	call AddressV86Source
	%addr32
	repz outsb
	sub esi, ebx	; restore program si
	jmp LeaveEmulator

EmulateOutsw:
	cmp bl, miOutsw
	jne EmulateInsb

	call AddressV86Source
	%addr32
	repz outsw
	sub esi, ebx   ; restore program si	
	jmp LeaveEmulator

EmulateInsb:
	cmp bl, miInsb
	jne EmulateInsw

	call AddressV86Target
	%addr32
	repz insb
	sub edi, ebx	; restore program di
	jmp LeaveEmulator

EmulateInsw:
	cmp bl, miInsw
	jne UnknownOpcode

	call AddressV86Target
	%addr32
	repz insw
	sub edi, ebx	; restore program di
	jmp LeaveEmulator

UnknownOpcode:
	dec V86_cs	
	jmp EnterGpFault

EmulateSti:
	cmp cl, miSti
	jne EmulateCli

; turn on real mode interrupts
	or v86_efl, 200h
	jmp LeaveEmulator

EmulateCli:
	cmp cl, miCli
	jne EmulateLock

	and v86_efl, 0FDFFh
	jmp LeaveEmulator

EmulateLock:
	cmp cl, miLock
	jne EmulateCSSeg
; do nothing, LOCK is a no operation
	jmp LeaveEmulator

EmulateCSSeg:
	cmp cl, miCSSeg ; this is a prefix that gets coded after the
			        ; LOCK prefix
	jne EmulateInBIm
	jmp LeaveEmulator

EmulateInBIm:
	push dx
;
; Note: following emulations must end via PopDxLeaveEmulator
;
	cmp cl, miInBIm
	jne EmulateOutBIm

	call FetchPort
	in al, dx

	jmp PopDxLeaveEmulator

EmulateOutBIm:
	cmp cl, miOutBIm
	jne EmulateInWIm

	call FetchPort
	out dx, al

	jmp PopDxLeaveEmulator

EmulateInWIm:
	cmp cl, miInWIm
	jne EmulateOutWIm

	call FetchPort
	in ax, dx

	jmp PopDxLeaveEmulator

EmulateOutWIm:
	cmp cl, miOutWIm
	jne EmulateHlt

	call FetchPort
	out dx,ax

	jmp PopDxLeaveEmulator



EmulateHlt:
	cmp cl, miHlt
	jne TryStringIO
; just terminate program if HLT encountered
	call Exit

TryStringIO:
	pop dx
	mov cx, 1   ; will use REPZ prefix in emulation, so just do it once
	jmp EmulateStringIO

PopDxLeaveEmulator:
	pop dx
LeaveEmulator:
	pop cx
	pop ebx
	pop bp
	add sp, 4 ; discard error code
	%data32 iRet
V86Emulator endp

V86SystemCall proc near


; DI indicates the type of system call
;
; 0 = kernel call
; 1 = rqInterface
; 2 = ossub 

; V86 work area is in the TSS

V86wa_wSpReal      equ word ptr [68h+0]		; unused
V86wa_wSSReal      equ word ptr [68h+2]		; V86 sa for interrupt handler stack
V86wa_sgSS         equ          [68h+4]     ; alias for real mode stack V86wa_ear          equ word ptr [68h+6]		; used by 8087 emulator
V86wa_wDSReal      equ word ptr [68h+8]		; used by 8087 emulator
V86wa_wFlReal      equ word ptr [68h+0ah]	; unused
V86wa_wCSReal      equ word ptr [68h+0ch]	; unused
V86wa_wIPReal      equ word ptr [68h+0eh]	; unused
V86wa_rgSgAlias    equ word ptr [68h+10h]	; v86wa_rgSgAlias is 16 words

	pop ax ; discard saved dx

	str ax
	sub ax, 8
	mov ds, ax
assume ds:nothing

	mov bx, v86_ss
; let fs address the level 0 stack
	push ss
	pop  fs

; ds is data alias for TSS, bx is ssReal
;
; Establish protected mode stack - this must be done on each
; entry since program may change SS.
;
; After a real mode program has create a process, the v86wa_sgSS is
; zero.  On the first system call, the comparison to rgGdtLink(0).sa will
; fail and a new stack selector will be created.
;
; bx = ssReal

	mov ax, DGroup
	mov es, ax
assume es:DGroup
	mov ax, word ptr [pRgoGdtLink+2]
	mov es, ax
assume es:nothing
	mov ax, v86wa_sgSS
	and ax, maskSelector ; optimized iSg*sGdtLink
	xchg ax,bx        ; bx = .rgGdtLink(iSgSS), ax = ssReal

	mov dx, es:[bx+rGdtLinkSa] ; dx = link.sa
	cmp dx, ax ; if rgGdtLink(iSgSS).sa = ssReal then done
	jne V86NewStack
	mov bx, V86wa_sgSS
	jmp short V86EstablishSS

V86NewStack:
; user is switching stacks, need a new selector
	push ds
	mov bx, DGroup
	mov ds, bx
assume ds:dGroup
	call FarAliasIpcSr
	mov bx, ax
; set reference count to 0FFh
	mov si, ax        ; si = .rgGdtLink(iSgSS)
	and si, maskSelector ; optimized iSg*sGdtLink

	mov byte ptr es:[si+rGdtLinkCRef], 0FFh
	pop ds  ; data alias for TSS
assume ds:nothing
	mov v86wa_sgSS, bx

V86EstablishSS:
	mov ss, bx	

; put cs:ip hack into registers
	mov bx, fs:v86_eip ; BX = encoded IP
	mov si, fs:v86_cs  ;SI = encoded CS

; let ss:sp addresses V86 stack
	mov sp, fs:v86_esp

; put v86 return address in level 0 stack
	pop fs:v86_eip
	pop fs:v86_cs
; put return address on V86 stack (ip done later)
	push cs

; dispatch to system call
	mov	 cx, di
	jcxz V86EnterKernel
	dec cx
	jcxz V86EnterRqInterface
	dec cx
	jcxz V86EnterOssub
	mov  ax, 3
	push ax
	call Crash

V86EnterOssub:
	push offset V86OssubReturn
	push fs:v86_efl
	push si ; cs hack
	push bx ; ip hack

; cx:di = @V86wa_rgSgAlias, needed by OssubEntry
	mov cx, ds
	lea di, v86wa_rgSgAlias
; dx = v86wa_sgSS, needed by OssubEntry
	mov dx, v86wa_sgSS
; make DS=SS for medium model
	push ss
	pop  ds
; cx:di is pV86wa_rgSgAlias
	jmp far ptr OssubEntry

V86EnterRqInterface:
	push offset V86ReturnFromSystemCall
	push fs:v86_efl
	push si ; cs hack
	push bx ; ip hack
	jmp far ptr RqInterface

V86EnterKernel:
; enter the kernel, with stack looking the same as a normal
; real mode kernel entry EXCEPT that encoded CS:IP are already
; in registers.
	push offset V86ReturnFromSystemCall
	push fs:v86_efl

	jmp pCEntry

V86OssubReturn:
	push ax ; ercRet
	str  bx
	sub  bx, 8
	mov  ds, bx
	mov  es, bx ; prevents GP fault from dealiasing contents of ES

	lea di, v86wa_rgSgAlias
V86DealiasLoop:
	mov ax, word ptr [di]
	or ax, ax
	jz V86DealiasDone
	call FarDealiasToSr
	add di, 2
	jmp short V86DealiasLoop

V86DealiasDone:
	pop ax ; ercRet
	jmp short V86SwitchOut

V86ReturnFromSystemCall:
; address TSS
	str bx
	sub bx, 8
	mov ds, bx

V86SwitchOut:
	mov cx, sp ; v86 sp

; reestablish level 0 stack
	mov bx, word ptr [rTss386Ss0]
	mov ss, bx
	mov sp, bp

	mov v86_esp, cx ; establish V86 sp
	pop bp
	add sp, 4 ; discard error code

	%data32 iret

V86SystemCall endp
;
; Coprocessor State Save and Restore
;



;
;  Utility procedures
;
Shl4 proc far
	push bp
	mov bp, sp
	mov eax, [bp+6]
	shl eax, 4
	mov dx, ax
	shr eax, 16
	xchg ax, dx
	pop bp
	ret 4
Shl4 endp

;
;  System traps
;
csFault equ word ptr ss:[bp+6]
ipFault equ word ptr ss:[bp+2]
maskOffRpl equ 0FFFCh
slLdt   equ 0ch
opCodeLock equ 0F0h
opCodeNop  equ 090h
maskLdt equ 4
Desc386_base      equ word ptr [si+2]
Desc386_baseHi    equ byte ptr [si+4]
Desc386_baseMsb   equ byte ptr [si+7]
ercInvalidOpcode    equ 28

InvalidOpcodeTrap proc far
	push bp
	mov  bp, sp
	push ebx
	push ds
	test v86NoErrorCode_eflMsb, initFlagV86Msb
	jz PInvalidOpcodeTrap

; illegal opcode from V86 environment
	mov bx, sgZero
	mov ds, bx
assume ds:nothing

; address opcode
	xor ebx, ebx
	mov bx, word ptr v86NoErrorCode_cs
	shl ebx, 4
	add ebx, v86NoErrorCode_eip

	jmp short TestForLock

PInvalidOpcodeTrap:
	push cx
	push si

	mov si, csFault
	and si, maskOffRpl

	mov cx, sgGdt
	test si, maskLdt     ; is the cs LDT or GDT based?
	jz  GetGla
	mov cx, slLdt
	and si, maskSelector

GetGla:
	mov ds, cx
	mov bh, desc386_baseMsb
	mov bl, desc386_baseHi
	shl ebx, 16
	mov bx, desc386_base
	add ebx, ipFault 

	mov cx, sgZero
	mov ds, cx
	pop si
	pop cx

TestForLock:
	cmp byte ptr [ebx], opcodeLock
	jne InvalidOpcodeError
	mov byte ptr [ebx], opcodeNop

	pop ds
	pop ebx
	pop bp
	db 66h
	iret

InvalidOpcodeError:
	pop ds
	pop ebx
	pop bp
	push ercInvalidOpcode
	jmp  FarSystemTrapHandler
InvalidOpcodeTrap endp

;
;  Real mode pseudo-interrupt support
;
CallRealCommIsr proc far
; On entry,
;   SS:SP are an interrupt stack, return address on stack
;	DX = i/o port
;	BX = ISR BX
;	CX = ISR DS
;	DS = caller's DS (preserved)
;	DI = opProc
;   AX = usernum
; On exit
;   AX, BX from isr, CX preserved

	push cx
	push ax               ; usernum
	push word ptr [di]    ; ip
	push word ptr [di+2]  ; cs
	push cx               ; ds
	call far ptr CallRealIsr
	pop cx
	ret	

CallRealCommIsr endp

CallRealIsr proc far
;	procedure(userNum, ip, cs, ds)

;
; Preserve BX and DX, as these are passed through to comm interrupt handler
;
	push bp
	mov bp, sp
	push ds
	mov ax, DGroup
	mov ds, ax
assume ds:dgroup

; get the target user's cr3
	mov si, word ptr [bp+12]
	shl si, 1
	mov ax, rgSgAsib[si]
	mov es, ax
	mov ecx, es:[oAsibCr3]

	push ds  ; OS DGroup

; real mode runs on this stack so push a hack return address
	push 20h ;  230h -> INT 4Bh
	push 30h

; clear nested task flag
	pushf
	pop ax
	and ax, 0BFFFh
	push ax
	popf

; switch to level 0 stack and build a v86 stack frame
	mov ax, ss
	mov fs, ax
	mov edi, esp

	str ax
	sub ax, 8
	mov ds, ax ; tss data alias
assume ds:nothing
	sub ax, 8
	mov ss, ax ; address level 0 stack
	mov	sp, [rTSS386SP0]	
	xor		eax, eax
	push	eax ; GS
	push	eax ; FS

	push	ax
	push	word ptr fs:[bp+6] ; DS

	push	eax ; ES

	push	ax
	push	V86wa_wSSReal; SS

	push	edi ; SP

	push	initFlagV86Msb ; FL
	pushf

	push	ax
	push	word ptr fs:[bp+8] ; CS

	push	ax
	push	word ptr fs:[bp+10] ; IP


; load up user's CR3
	
	mov rTssCr3, ecx
	mov cr3, ecx

; enter V86 mode

	db 66h
	iret

ReturnFromRealISR:
; preserve AX (return parameter in comm interrupts)
; restore SS AND DS
	pop dx
	pop ebx ; restore isr bx
	mov esp, v86_esp

	str dx
	sub dx, 8
	mov fs, dx
	mov dx, fs:v86wa_sgSS
	mov ss, dx
	pop ds ; OS DGroup
assume ds:dGroup
; restore global CR3 and set nested task
	mov ecx, paGlobalPageMap
	mov fs:rTssCr3, ecx
	mov cr3, ecx

	pushf
	pop dx
	or dx, 4000h
	push dx
	popf

	pop ds
	pop	bp
	ret 8
CallRealIsr endp


CallBios proc far
;	procedure(iLevel)

; no registers clobbered - on return callers DS and EBP preserved
; high 16 bits of eax is real mode ES
; high 16 bits of ecx is real mode DS

paramILevel equ word ptr [bp+10]

	push ds
	push ebp

	ror ebx, 16 ; using bx so save in high 16 bits of ebx
	ror edx, 16 ; using dx so save in high 16 bits of edx
	ror ebp, 16 ; using bp so save in high 16 bits of ebp

	mov bp, sp

; push V86 stack frame on current stack
	xor		bx, bx
	push	ebx 				; GS
	push	ebx 				; FS
; high 16 bits of ecx is real mode DS
	ror		ecx, 16
	push	ecx 				; DS
	ror		ecx, 16
; high 16 bits of eax is real mode ES
	ror		eax, 16
	push	eax 				; ES
	ror		eax, 16

; address TSS with DS
	str		dx		
	sub		dx, 8
	mov		ds, dx

; save return stack pointer
	mov word ptr [rTss386wSpSave], bp
	mov dx, ss
	mov word ptr [rTss386wSsSave], dx

	mov		dx, word ptr [rTss386saSS0]
	or		dx, dx
	jz		NoCanDo

	push	bx
	push	dx			; SS

; let BIOS stack pointer be 128 bytes below top of level 0 stack pointer
	push	bx
	mov		bx, [rTSS386SP0]	;
	sub		bx, 86h				; 3 words are being pushed
	push	bx					; ESP

; address real mode stack with DS
	mov		dx, word ptr [rTss386SS0]
	mov		ds, dx

; initialize real mode stack with flags and a hack return address
	pushf
	pop		dx
	mov word ptr [bx+4], dx
	mov word ptr [bx+2], 20h
	mov word ptr [bx],   32h ;  232h -> INT 4Ch

	push	initFlagV86Msb
	pushf						; EFL

; fetch CS:IP from real mode interrupt vector
	xor		dx, dx
	mov		bx, sgZero
	mov		ds, bx
	mov		bx, paramIlevel
	shl		bx, 2				; vector address = iLevel * 4
	push	dx
	push	word ptr [bx+2]		; CS

	push	dx
	push	word ptr [bx]		; IP

; clear nested task flag - only necessary when called from int handler
	pushf
	pop dx
	and dx, 0BFFFh
	push dx
	popf


; restore registers and enter V86 mode
	ror ebx, 16
	ror edx, 16
	ror ebp, 16
	db 66h
	iret

ReturnFromBIOS:

; no registers destroyed except bp, which is restored
; BIOS ES put in high 16 bits of EAX
; BIOS DS put in high 16 bits of ECX

	pop dx  ; from V86Emulator
	pop ebx
	pop cx

	ror eax, 16
	mov ax, v86_es ; ES from BIOS
	ror eax, 16

	ror ecx, 16
	mov cx, v86_ds ; DS from BIOS
	ror ecx, 16

; switch stacks, using high 16 of edx for temp storage for dx
	ror edx, 16

	mov dx, DGroup
	mov ds, dx
assume ds:DGroup
	str dx
	cmp dx, word ptr [sgTssIntLast]
assume ds:Nothing
	ja NestedTaskOk

; if interrupt task, restore nested task flag
	pushf
	pop dx
	or dx, 4000h
	push dx
	popf
	str dx

NestedTaskOk:
	sub dx, 8
	mov ds, dx
; don't worry about non atomic ss:sp update since interrupts use task gates
	mov dx, word ptr [rTss386wSpSave]
	mov sp, dx
	mov dx, word ptr [rTss386wSsSave]
	mov ss, dx
	ror edx, 16	

Adios:	
	pop ebp
	pop ds
	ret 2

NoCanDo:
	mov ax, ercNotImplemented
	mov sp, bp
	jmp short Adios
CallBios endp

;
;  Bootstrap and ExtCrashDump support
;

EnterRealMode proc far
;	procedure (bEnterMode, laSource, laDestination, qbImage, rES, rDX)
assume ds:nothing

bEnterMode    equ [bp+22]
laSource      equ [bp+18]
laDestination equ [bp+14]
qbImage       equ [bp+10]
rES           equ [bp+8]
rDX           equ [bp+6]
NewGen equ 9

	push bp
	mov bp, sp
assume ds:dGroup

; relocate video ram to 1mb so the BIOS can use it.
	call RelocateVideoRam

; plaMemMax is destination for first copy and source for second copy.
;
; Why two copies? The bootstrap image lives in virtual memory.  The first copy 
; composes a contiguous image in memory that is not used my the page service.
; The second copy relocates the image to low memory. This gets around the 
; problem of overwriting a frame before it is read.

	mov edi, plaMemMax
	shl edi, 4
	mov edx, edi

; will need processorType and protectedModeEnabl in SwitchNub
	xor ax, ax
	mov al, processorType
	mov ah, bEnterMode
	shl eax, 16
	mov ax, protectedModeEnabl

	mov bx,  rDX     ; preserve across mode switch
	shl ebx, 16
	mov bx,  rES     ; preserve across mode switch

	mov cx, sgZero
	mov es, cx
	mov ds, cx

; relocate the bootstrap image and/or SwitchNub 
; LowMem (OS DGroup) is about to be clobbered so cannot take interrupts
	cli
	mov esi, laSource
; from this point on, there can be no use of stack or BP
	mov esp, laDestination
	mov ecx, qbImage
	mov ebp, ecx

; first copy - source image overwrites plaMemMax
	db 67h
	repnz movsb

; second copy - image overwrites target
	mov ecx, ebp
	mov esi, edx
	mov edi, esp
	db 67h
	repnz movsb

; initialize RMdata area for real mode switch
assume es:RMdata
	mov RM_portProtectEnable, ax
	shr eax, 16
	mov RM_wProcType, ax
	mov dx, 3FFh
	mov RM_idtr386Real, dx ; this is the limit, base assummed zero
assume es:nothing

; clear paging bit in CR0
	xor eax, eax
	smsw dx
	xchg ax, dx
   	mov cr0, eax
	xchg ax, dx
;clear page cache
	mov cr3, eax
	Jmp far ptr PagingOff

PagingOff:
; NOTE !!! WE ARE ABLE TO CONTINUE WITH PAGING DISABLED BECAUSE
; THIS CODE RESIDES IN IDENTITY SPACE
	mov ax, sgIdentityLowMem	
	mov es, ax
	mov ss, ax
	mov ds, ax
	mov fs, ax
	mov gs, ax

; set PE bit to zero
	mov eax, cr0
	and  al, 0FEh
   	mov cr0, eax
;	jmp pRealVector
	db 0EAh
	dw 0
; saRealVector is fixed up by VBootstrap to address the relocated SwitchNub.
saRealVector label far
	dw 0

EnterRealMode endp

public ResetKbdForBios
ResetKbdForBios proc far
		ret					; for now, does not work on sg3000

%if (0) then (
		in 	al, 60h			; purge out buff
		out	0EDh, al
		mov	al, 0AAh		; reset 8042 on mo bo
		out	64h, al			;
		out	0EDh, al

		call	wait_out_full

		in 	al, 60h			; purge out buff
		out	0EDh, al
		call	wait_in_empty
		mov	al, 0FFh		; reset the kybd
		out	60h, al
		out	0EDh, al

		call	wait_in_empty

		call	wait_out_full
		in 	al, 60h			; purge out buff
		out	0EDh, al

		call	wait_out_full

		in 	al, 60h			; purge out buff
		out	0EDh, al

		call	wait_in_empty
		mov	al, 60h
		out	64h, al
		call	wait_in_empty
		mov	al, 65h
		out	60h, al
		ret
)fi
ResetKbdForBios endp

%if (0) then (
wait_out_full	proc	near
obf_mask	equ	01h

		push	dx
		push	ax

try_again:
		in	al, 64h
		out	0EDh, al
		test	al, obf_mask
		jz	try_again


		pop	ax
		pop	dx
		ret
wait_out_full	endp

wait_in_empty	proc	near
		push	dx
		push	ax
		mov	dx, 064h		; status info port
wait_in_more:
ibf_mask	equ	02h
		in	al, dx			; 
		out	0EDh, al
		test	al, ibf_mask
		jnz	wait_in_more

		pop	ax
		pop	dx
		ret
wait_in_empty	endp
)fi

;
; this code moved out of the relocated portion os SwitchNub because it made
; SwitchNub too large. 
;
RelocateVideoRam proc near
	enter 0, 0
; in case using VideoCard, put video memory back in first megabyte
	mov dx, portWeitekSequencer   ;3c4h
	push dx                       ;
	mov al, 7                     ;read 5186 id register
	out dx, al                    ;
	jmp .+2                       ;
	inc dx                        ;3c5h
	xor al,al                     ;
	out dx, al                    ;try to clear the register
	jmp .+2                       ;
	in al,dx                      ;if 5186, id bits still there
	shr al,5                      ;
	cmp al,1                      ;
	pop dx                        ;3C4h
	jne notWeitek                 ;jif not 5185
	mov al, 8                     ;enable 5186 port 320h
	out dx, al                    ;
	push dx                       ;
	jmp .+2                       ;
	mov dx, portWeitekData        ;320h
	xor ax, ax                    ;clear the memory map register
	out dx, al                    ;
	jmp .+2                       ;
	pop dx                        ;3c4h
	out dx, al                    ;clear index register
notWeitek:
	leave
	ret
RelocateVideoRam endp

RmosSwitch_code ends

end
