;**********************************************************************
; Title: Int_all.asm 
; This module handles interrupts for protected mode applications.
; Update Log:
; 10/31/90 JM created.
; 01/24/91 JF implemented
; 02/26/91 JF Stacks only get mediated from 386 to 286.  They never get 
;			mediated from 286 to 386 because all trap gates are 386
;			on 386 processors.
; 03/28/91 JF if interrupt due to hardware, enable interrupts asap if
; 			handler runs with interrupts enabled.
; 04/04/91 JF NopIsr only gets called for 386, hence we always IRET long.
; 05/07/91 JF Forgot IRET for 286.
; 05/13/91 AT Early EOI for some device interrupts.
; 08/09/91 JA slJumpTable=7Ch.
; 05/13/92 JM handle errorcode generating exceptions in 286 gate emulation
; 01/05/93 JM preserve high(EIP & ESP) when emulating a gate from a tss.
; 05/13/93 JF Fix real mode traps
;**********************************************************************
%set(DEBUG,0)
%set(LOGGING,0)

; PUBLIC ENTRY PROCEDURES 

PUBLIC MediateTrap
PUBLIC FilterDeviceInterrupt
PUBLIC NopIsr
PUBLIC ResumeInterruptFilter

; EXTERNAL PROCEDURES

EXTRN ErrorExit: FAR
EXTRN Crash: FAR
EXTRN CheckErcBreak: FAR

$MOD386
%IF(%DEBUG) THEN(
)FI
DATA SEGMENT PUBLIC 'Data'
PUBLIC sgIntLog, sbIntLog, pIntLog
pIntLog DD 0
sgIntLog EQU WORD PTR pIntLog+2
EXTRN cascadeOCW2_8259: WORD
EXTRN OCW2_8259: WORD
EXTRN Ext8259_OCW1: WORD
EXTRN Ext8259_OCW2: WORD
EXTRN mAHighest: BYTE
EXTRN mACascade: BYTE
EXTRN mpTyDevMask: WORD
extrn vf:byte
%IF(%MF)then(%' MF+CTOSP
EXTRN cascadeCascadeOCW2_8259: WORD
)FI
%IF(%DEBUG) THEN(
PUBLIC int_ISR, int_TSS, int_fDevice, int_backlink, int_level, int_LDT
PUBLIC int_access, int_fSoftware, int_fl, int_ip, int_cs
int_ISR DW 0
int_TSS DW 0
int_fDevice DW 0
int_fSoftware DW 0
int_backlink DW 0
int_level DW 0
int_LDT DW 0
int_access DW 0
int_fl DW 0
int_ip DW 0
int_cs DW 0
)FI

sLog EQU 0	;will be zero if no logging
%IF(%Logging) THEN(
;	To look at the log:
;		pIntLog points to the last entry
;		each previous entry is further down in memory.
;		We start at the end of the log and go backwards to make it
;		easier to look at the log in the debugger. 
;	Each entry is as follows:
;		FFFF	New Filter entry (6 words)
;			int level (word)
;			ISR register (word) 
;			IP  of handler invoked
;			CS  of handler invoked
;			FL  of handler invoked
;		AAAA	New Test program Software Interrupt entry (3 words)
;			int level (word)
;			ISR register (word) 
sbFilterLogEntry	EQU	14	;size of filter log entry in bytes
swFilterLogEntry	EQU	6	;size of filter log entry in words
sbIntLogEntry		EQU	6	;size of interrupt log entry in bytes
swIntLogEntry		EQU	3	;size of interrupt log entry in words
cLogEntries 		EQU	100
sLog				EQU	cLogEntries*sbFilterLogEntry
oIntLog				EQU	WORD PTR pIntLog
log_id				EQU WORD PTR GS:[DI+0]
log_level			EQU WORD PTR GS:[DI+2]
log_ISR				EQU WORD PTR GS:[DI+4]
log_IP				EQU WORD PTR GS:[DI+6]
log_CS				EQU WORD PTR GS:[DI+8]
log_FL				EQU WORD PTR GS:[DI+10]
log_DS				EQU WORD PTR GS:[DI+12]
log_filterId		EQU 0FFFFh
log_softIntId		EQU 0AAAAh
)FI
sbIntLog			DW sLog	;zero if no logging
DATA ENDS
DGroup GROUP Data

mask386	EQU	8
slJumpTable EQU 7Ch
sJumpTableEntry EQU 8 ; INT CD + 4 bytes for pointer + access + unused
shJumpTableEntry EQU 3	;shift factor for JumpTableEntry
sgGdt   EQU 8
sgIdt	EQU 10h
sgZero	EQU 48h

$Include(:f1:Kernel.mdf)
$include(:f1:vfequ.idf)

ercBadVector EQU 13

; tss offsets
tss_backLink EQU 0

tss386_iInt EQU 0C0h
tss386_ldt EQU 60h
tss386_SP EQU 38h
tss386_SS EQU 50h
tss386_CS			EQU 04Ch
tss386_IP			EQU 020h
tss386_FL			EQU 24h
tss386_wDsIntFilter EQU 0C2h
tss386_wFlIntFilter EQU 0C4h
tss386_wIpIntFilter EQU 0C6h
tss386_wCsIntFilter EQU 0C8h
tss386_tyDev		EQU 09Ah
tss386_SS0			EQU 8
tss386_SP0			EQU 4

tss286_iInt EQU 68h
tss286_ldt EQU 2Ah
tss286_SP EQU 1Ah
tss286_SS EQU 26h
tss286_CS			EQU 024h
tss286_IP			EQU 0E0h
tss286_FL			EQU 10h
tss286_wDsIntFilter EQU 6Ah
tss286_wFlIntFilter EQU 6Ch
tss286_wIpIntFilter EQU 6Eh
tss286_wCsIntFilter EQU 70h
tss286_tyDev		EQU 5Eh

; jumpTable equates:

bNull EQU 0
l286TrapGate	EQU 7
desc_bAccess 	EQU 5	; offset of code descriptor access	
jte_access		EQU 6	; offset in jumptable entry
jte_sa			EQU 4	; offset in jumptable entry
jte_ra			EQU 2	; offset in jumptable entry

IntMediation SEGMENT PUBLIC 'Code'
ASSUME CS:IntMediation

MediateTrap PROC FAR
; Jump here from RqInterface when INT CD occurs and CS = slJumpTable
; No registers have been changed.
; We come here when the trap needs to be mediated:  There are two cases:
; (1) A real mode application has set a trap handler, and a trap occurs.
; (2) The interrupt gate is 386, and we need to set up the stack for
; a 286 trap.  We never need to mediate a 286 trap to 386 format.  On a
; 386 processor, all trap gates are 386 gates.  On a 286, it is not possible
; to set a 386 trap.
; ss:sp ->	ipHack	(iInt*8)		<- ss:bp+8
;			csHack	(slJumpTable)	   ss:bp+10
;			flHack					   ss:bp+12
;			(e)ipTrap
;			( )csTrap
;			(e)flTrap

ipHack			EQU WORD PTR [BP+2]
csHack			EQU WORD PTR [BP+4]
flHack			EQU WORD PTR [BP+6]
	PUSH BP
	MOV  BP, SP
; save registers we use
	PUSH SI
	PUSH ES
	PUSH AX
	MOV  SI, ipHack		; (assumes sJumpTable = 8)
; ES:SI = address of trap code descriptor+2
	MOV  AX,slJumpTable
	MOV  ES,AX
; ES:SI = address of jump table entry+2
; SI = (iInt*8) + 2
; Check to see if we have an error code on the stack.
; Interrupts 8-14 and 17 have error codes, and the rest do not.
	CMP SI, 17*8+2
	JA  NoErrorCode
	JE  ErrorCode
	CMP SI, 8*8+2
	JB  NoErrorCode
	CMP SI, 14*8+2
	JA  NoErrorCode

ErrorCode:
; The stack DOES have an error code.  It looks like this:
ipHack			EQU  WORD PTR [BP+2]		;becomes raTrapHandler to ret to
csHack			EQU  WORD PTR [BP+4]		;becomes saTrapHandler to ret to
flHack			EQU  WORD PTR [BP+6]		;discarded by RET
errorCodeTrap	EQU DWORD PTR [BP+8]		;discarded by RET
ipLoTrap		EQU  WORD PTR [BP+12]		;discarded by RET
ipHiTrap		EQU  WORD PTR [BP+14]		;discarded by RET
csTrap			EQU  WORD PTR [BP+16]		;becomes error code
fillTrap		EQU  WORD PTR [BP+18]		;becomes IP
flLoTrap		EQU  WORD PTR [BP+20]		;becomes CS
flHiTrap		EQU  WORD PTR [BP+22]		;becomes FL
	CMP ipHiTrap, 0	
	JNZ  IpWayBig
; See if a privilege level change occurred.
	TEST flHiTrap,2
	JNZ  ErrorCodeLevelChange
; shuffle cs, ip, fl, and errorcode into 286 format
	MOV  AX, csTrap
	XCHG flLoTrap, AX	
	MOV  flHiTrap,AX
	MOV  AX, ipLoTrap
	MOV  fillTrap, AX
	MOV  AX, errorCodeTrap
	MOV  csTrap, AX
; put user's pTrapHandler on stack where we can ret to it
	MOV  AX, ES:[SI + jte_ra-2]	; ip of handler
	MOV  ipHack, AX
	MOV  AX, ES:[SI + jte_sa-2]	; cs of handler
	MOV  csHack, AX
	POP  AX
	POP  ES
	POP  SI
	POP  BP
	RET  10

ErrorCodeLevelChange:
; A real mode trap occurred, and there is an error code.
; The stack looks like this:
ipHack			EQU  WORD PTR [BP+2]		;Discarded before IRET
csHack			EQU  WORD PTR [BP+4]		;Discarded before IRET
flHack			EQU  WORD PTR [BP+6]		;Discarded before IRET
errorCodeTrap	EQU DWORD PTR [BP+8]		;Discarded before IRET
ipLoTrap		EQU  WORD PTR [BP+12]		;becomes IP of trap handler
ipHiTrap		EQU  WORD PTR [BP+14]		;must be zero
csTrap			EQU  WORD PTR [BP+16]		;becomes CS of trap handler
fillTrap		EQU  WORD PTR [BP+18]		;
flLoTrap		EQU  WORD PTR [BP+20]		;
flHiTrap		EQU  WORD PTR [BP+22]		;
oldSpLowTrap	EQU  WORD PTR [BP+24]		;ra of stack before trap
oldSpHiTrap		EQU  WORD PTR [BP+26]		;must be zero
oldSsTrap		EQU  WORD PTR [BP+28]		;sg of stack before trap
	PUSH EBX
	PUSH FS
; Get linear address specified by old SS:SP
	XOR  EBX,EBX
	MOV  BX,oldSsTrap
	SHL  EBX,4
	ADD  EBX,DWORD PTR oldSpLowTrap
	MOV  AX,sgZero
	MOV  FS,AX
; Now FS:EBX points to old SS:SP location.
; Set up old stack to ret to instruction that caused trap.
; A trap handler should do an IRET, so push the flags on the stack first.
	SUB  EBX,2
	MOV  AX,flLoTrap
	MOV  FS:[EBX],AX
	SUB  EBX,2
	MOV  AX,csTrap
	MOV  FS:[EBX],AX
	SUB  EBX,2
	MOV  AX,ipLoTrap
	MOV  FS:[EBX],AX
	SUB  oldSpLowTrap,6
; Set up CS:IP on current stack to return to client trap handler.
	MOV  AX, ES:[SI + jte_ra-2]	; ip of handler
	MOV  ipLoTrap, AX
	MOV  AX, ES:[SI + jte_sa-2]	; cs of handler
	MOV  csTrap, AX
	POP  FS
	POP  EBX
	POP  AX
	POP  ES
	POP  SI
	POP  BP
	ADD  SP,10					;Discard ipHack, csHack, flHack, error code
;SS:SP now points to ipLoTrap
	DB		66h
	IRET


NoErrorCode:	
; The stack does NOT have an error code, and it looks like this:
ipHack			EQU WORD PTR [BP+2]		;becomes raTrapHandler to ret to
csHack			EQU WORD PTR [BP+4]		;becomes saTrapHandler to ret to
flHack			EQU WORD PTR [BP+6]		;discarded by RET
ipLoTrap		EQU WORD PTR [BP+8]		;discarded by RET
ipHiTrap		EQU WORD PTR [BP+10]	;discarded by RET
csTrap			EQU WORD PTR [BP+12]	;discarded by RET
fillTrap		EQU WORD PTR [BP+14]	;becomes IP
flLoTrap		EQU WORD PTR [BP+16]	;becomes CS
flHiTrap		EQU WORD PTR [BP+18]	;becomes FL
	CMP ipHiTrap, 0	
	JNZ  IpWayBig
; See if a privilege level change occurred.
	TEST flHiTrap,2
	JNZ  NoErrorCodeLevelChange
; shuffle cs, ip, and fl into 286 format
	MOV  AX, csTrap
	XCHG flLoTrap, AX	
	MOV  flHiTrap,AX
	MOV  AX, ipLoTrap
	MOV  fillTrap, AX
; put user's pTrapHandler on stack where we can ret to it
	MOV  AX, ES:[SI + jte_ra-2]	; ip of handler
	MOV  ipHack, AX
	MOV  AX, ES:[SI + jte_sa-2]	; cs of handler
	MOV  csHack, AX
	POP  AX
	POP  ES
	POP  SI
	POP  BP
	RET  8

NoErrorCodeLevelChange:
; A real mode trap occurred, and there is NO error code.
ipHack			EQU  WORD PTR [BP+2]		;Discarded before IRET
csHack			EQU  WORD PTR [BP+4]		;Discarded before IRET
flHack			EQU  WORD PTR [BP+6]		;Discarded before IRET
ipLoTrap		EQU  WORD PTR [BP+8]		;becomes IP of trap handler
ipHiTrap		EQU  WORD PTR [BP+10]		;must be zero
csTrap			EQU  WORD PTR [BP+12]		;becomes CS of trap handler
fillTrap		EQU  WORD PTR [BP+14]		;
flLoTrap		EQU  WORD PTR [BP+16]		;
flHiTrap		EQU  WORD PTR [BP+18]		;
oldSpLowTrap	EQU  WORD PTR [BP+20]		;ra of stack before trap
oldSpHiTrap		EQU  WORD PTR [BP+22]		;must be zero
oldSsTrap		EQU  WORD PTR [BP+24]		;sg of stack before trap
	PUSH EBX
	PUSH FS
; Get linear address specified by old SS:SP
	XOR  EBX,EBX
	MOV  BX,oldSsTrap
	SHL  EBX,4
	ADD  EBX,DWORD PTR oldSpLowTrap
	MOV  AX,sgZero
	MOV  FS,AX
; Now FS:EBX points to old SS:SP location.
; Set up old stack to ret to instruction that caused trap.
; A trap handler should do an IRET, so push the flags on the stack first.
	SUB  EBX,2
	MOV  AX,flLoTrap
	MOV  FS:[EBX],AX
	SUB  EBX,2
	MOV  AX,csTrap
	MOV  FS:[EBX],AX
	SUB  EBX,2
	MOV  AX,ipLoTrap
	MOV  FS:[EBX],AX
	SUB  oldSpLowTrap,6
; Set up CS:IP on current stack to return to client trap handler.
	MOV  AX, ES:[SI + jte_ra-2]	; ip of handler
	MOV  ipLoTrap, AX
	MOV  AX, ES:[SI + jte_sa-2]	; cs of handler
	MOV  csTrap, AX
	POP  FS
	POP  EBX
	POP  AX
	POP  ES
	POP  SI
	POP  BP
	ADD  SP,6					;Discard ipHack, csHack, flHack
;SS:SP now points to ipLoTrap
	DB		66h
	IRET


IpWayBig:
; high(eip) <> 0 - can't emulate 286 gate
	PUSH DS		;Save DS
	MOV  AX,DGroup
	MOV  DS,AX
	PUSH ercBadVector
	CALL CheckErcBreak
	POP  DS		;Restore DS
	PUSH ercBadVector
	CALL ErrorExit

MediateTrap ENDP

	


;**********************************************************************
; Define macro first so we can call it below for 286 or 386 TSS.
; We have an interrupt for a 286 TSS or a 386 TSS.
; Determine whether it's a device interrupt or a software interrupt.
; ES:BX points to TSS of interrupt task.
; emulate 286 or 386 trap gate if trap has been installed by interrupted task.
; must reside in GDT.
;**********************************************************************
%*DEFINE(CreateTssMediator(Name)) (
ConvertTss%Name LABEL FAR
PUBLIC ConvertTss%Name
ASSUME DS: Dgroup, ES: Nothing

%IF(%Debug) THEN (
	MOV		FS:int_TSS,BX
	MOV		FS:int_fDevice,0
	MOV		FS:int_fSoftware,0
)FI
	SUB		BX, 8 ; make data alias
	MOV		ES, BX
; ES addresses TSS of interrupt task as data
%IF(%Logging) THEN (
	LGS		DI,FS:pIntLog
; GS:DI points to log.  This memory was allocated by InitTraps1 if
; sbIntLog <> 0.  Memory is not in OS DGroup.
	SUB		DI,sbFilterLogEntry
	JNS		Log1%Name
; no room left in log, must restart at the bottom.
	MOV		DI,(cLogEntries-1)*sbFilterLogEntry
Log1%Name:
	MOV		FS:oIntLog,DI		;Update ra of pIntLog
	MOV		log_id,log_filterId
	MOV		AX,ES:[%intTss_iInt]
	MOV		log_level,AX
	MOV		log_ISR,0
)FI
	MOV		CX, ES:[%intTss_LDT]	;Save value of LDT in CX
	CMP		WORD PTR ES:[%intTss_iInt],32
	JAE		ReadInService%Name
; This interrupt is for one of the Intel-reserved levels.  This means
; there is no In-Service bit associated with the interrupt.
; We will look to see if the user that is back-linked to the interrupt TSS
; has a trap defined for this interrupt.  If so, we will jump to his handler.
; If the user does not have a trap defined, we will jump to the system
; interrupt handler.  We set DX non-zero to remember to jump to the
; system handler if the user does not have a trap defined.
	MOV		DX,1
	JMP		SoftwareInt%Name
ReadInService%Name:
; Read the In-Service register to see if the interrupt was caused by hardware.
	%READ_ISR_AXDX		;Uses AX, DX, BX
	OR		DX,AX
%IF(%Debug) THEN (
	MOV		FS:int_ISR,DX
)FI
%IF(%Logging) THEN (
	MOV		log_ISR,DX
)FI
	JNZ		JmpHandler%Name
	JMP		SoftwareInt%Name
JmpHandler%Name:
; The interrupt was caused by a device.
; Test mpTyDevMask(tyDev).f to see if we want to do an early EOI
; Mask structure is (port, wEnableAnd, wDisableOr, oMem, iInt, f, oCount)
devPort			EQU 0
devEnableAnd	EQU 2
devDisableOr	EQU 4
devoMem			EQU 6
deviInt			EQU 8
devF			EQU 9
devoCount		EQU 10
lfEarlyEoi		EQU 2
; Assumption: tyDev is set in the TSS for all device interrupts.
; This is true for all SysGen'd interrupts, and we don't allow setting a 
; device handler for interrupts not in SysGen.  If this changes we'll still be 
; ok because tyDev=0 maps to a controller entry in mpTyDevMask, and lfEarlyEoi 
; is never set for controllers.
	MOV		BX,ES:[%intTss_tyDev]
	MOV		AX,12
	MUL		BL
	MOV		BX,AX
	TEST	BYTE PTR mpTyDevMask[BX+devF],lfEarlyEoi
	JNZ		DoMask%Name
	JMP		NoEoi%Name
DoMask%Name:
	PUSH	DI	;Only need to save DI for this one piece, not entire Filter
; bOpDisable code cloned from ControlInterrupt in Kernel_all.asm
	MOV		DX,mptyDevMask[BX+devPort]		;Read current interrupt mask value
	XOR		AH,AH
	IN		AL,DX							;Byte register
	MOV		DI,mptyDevMask[BX+devoCount]	;Offset of mask count
	INC		WORD PTR [DI]
	OR		AX,mptyDevMask[BX+devDisableOr]
	OUT		DX,AL							;Byte register
	MOV		DI,mptyDevMask[BX+devoMem]		;Find mask memory copy
	OR		DI,DI
	JZ		DoEoi%Name
	MOV		[DI],AX
DoEoi%Name:
	POP		DI
	%DO_EOI
NoEoi%Name:
; Get flags, CS:IP from int area of TSS so we can pop them from the stack
%IF(%Debug) THEN (
	MOV		FS:int_fDevice,0FFh
)FI
	MOV		AX,ES:[%intTss_wFlIntFilter]
; If the handler can run with interrupts enabled, we can enable here.
; If the interrupt was a software interrupt, we must leave interrupts disabled
; because we could get another interrupt from the device and cause a fault
; becuase our TSS would be busy.
	TEST	AX,200h
	JZ		SetupFlags%Name
	STI
SetupFlags%Name:
	MOV		flHandler,AX
%IF(%Debug) THEN (
	MOV		FS:int_fl,AX
)FI
%IF(%Logging) THEN (
	MOV		log_FL,AX
)FI
	MOV		AX,ES:[%intTss_wIpIntFilter]
	MOV		ipHandler,AX
%IF(%Debug) THEN (
	MOV		FS:int_ip,AX
)FI
%IF(%Logging) THEN (
	MOV		log_IP,AX
)FI
	MOV		AX,ES:[%intTss_wCsIntFilter]
	MOV		csHandler,AX
%IF(%Debug) THEN (
	MOV		FS:int_cs,AX
)FI
%IF(%Logging) THEN (
	MOV		log_CS,AX
)FI
	LLDT	CX			;Restore LDT
	MOV		ES:[%intTss_LDT],CX
; Get DS for handler from int area of TSS
	MOV		DS, ES:[%intTss_wDsIntFilter]
%IF(%Logging) THEN (
	MOV		log_DS,DS
)FI
; Restore registers we used.
%IF(%Logging) THEN (
	POP		GS
	POP		DI
)FI	
%IF(%Debug) THEN (
	POP		FS
)FI	
	POP		CX
	POP		DX
	POP		AX
	POP		ES
	POP		BX
	POP		BP
	PUSH	0
	POPF				;Turn off Nested Task bit
; Set flags as they were specified for the interrupt handler, and jump
; to the interrupt handler.  We are running with the Nested Task bit set to
; zero so that, when we IRET, we do not do a task switch.
	IRET

PUBLIC SoftwareInt%Name
SoftwareInt%Name:
; Int in-service bit is not on, so the interrupt was not caused by hardware.
; We assume the interrupt was caused by a software INT instruction.
; extract int level from our tss
; Get the backlink and check the type of TSS that was executing
	MOV		BX, ES:WORD PTR [tss_backLink]
%IF(%Debug) THEN (
	MOV		FS:int_fSoftware,0FFh
	MOV		int_backlink,BX
)FI
	MOV		AX,sgGdt
	MOV		DS,AX
; DS:BX points to descriptor for TSS we are backlinked to.  
%if(not %ctosv)then(
; Test the access byte to see if this is a 286 or a 386 TSS.
; If ctosv, then all TSS are 386.
	TEST	BYTE PTR DS:[BX+5],mask386
	JZ		Backlink286Tss%Name
	JMP		Backlink386Tss%Name
Backlink286Tss%Name:	
; We are backlinked to a 286 TSS
%SET(userTss_LDT,tss286_LDT)
%SET(userTss_SP,tss286_SP)
%SET(userTss_SS,tss286_SS)
%SET(userTss_FL,tss286_FL)
%SET(userTss_CS,tss286_CS)
%SET(userTss_IP,tss286_IP)
	MOV		AL, ES:BYTE PTR [%intTss_iInt]	
; AL = iInt

; check jump table of interrupted task to see if we must emulate. 
	SUB		BX, 8			; make data alias
	MOV		DS, BX
; DS points to TSS of task that caused the software interrupt
ASSUME DS: Nothing
	MOV		BX, DS:WORD PTR [%userTss_LDT]
; BX = LDT of interrupt task
	OR		BX, BX		 	; no LDT?
	JNZ		Tss286LdtNotNull%Name
	JMP		NoUserTrap%Name		;User has no LDT
Tss286LdtNotNull%Name:
; Load LDT of the interrupted task.
; The LDT register points to the LDT of the interrupted task.  This will allow
; us to get selector(4) from the LDT table to access the trap jump table.
; Given iInt in AL, get index of JumpTableEntry
	LLDT	BX
; Store LDT in our TSS -- otherwise we can't debug this.
; CX has old value of LDT from TSS.
	MOV		ES: WORD PTR[%intTss_LDT],BX
	MOV		BX,AX
	XOR		BH,BH
	SHL		BX,shJumpTableEntry			;Size of JumpTableEntry = 8
; BX = offset into jump table for the interrupted task.
	MOV		AX, slJumpTable
	MOV		ES, AX
; ES points to the trap jump table for the interrupted task.
	MOV		AL, ES:BYTE PTR [BX+jte_access]
	AND		AL,0Fh
	JNZ		Tss286EmulateTrap%Name
	JMP		NoUserTrap%Name			;user has no trap defined

Tss286EmulateTrap%Name:
; build the right kind of stack for the trap
	CMP		AL, l286TrapGate
	JNE		Tss286Emulate386Gate%Name

Tss286Emulate286Gate%Name:
	MOV		AX, ES:WORD PTR [BX+jte_sa]
	MOV		DX, ES:WORD PTR [BX+jte_ra]
	MOV		ES, DS:WORD PTR [%userTss_SS]
; AX:DX = user pTrapHandler
; ES = user SS

	MOV		BX, DS:WORD PTR [%userTss_SP]
; ES:BX = user SS:SP
; Set the backlinked TSS CS:IP to point to the interrupt routine,
; and save the old CS in the stack so that the int routine will 
; return to the instruction following the INT instruction.
	XCHG	AX,[%userTss_CS]
    MOV		ES:WORD PTR [BX-4], AX	; CS
	XCHG	DX,[%userTss_IP]
    MOV		ES:WORD PTR [BX-6], DX	; IP
; Get the flags word of the backlinked TSS and store it in the stack
; so the int routine can IRET.
	MOV		AX, DS:WORD PTR [%userTss_FL]
    MOV		ES:WORD PTR [BX-2], AX	; FL
	SUB		DS:WORD PTR [%userTss_SP], 6
	JMP		TssToTrapMediateRet%Name
	
Tss286Emulate386Gate%Name:
	MOV		AX, ES:WORD PTR [BX+jte_sa]
	MOV		DX, ES:WORD PTR [BX+jte_ra]
	MOV		ES, DS:WORD PTR [%userTss_SS]
; AX:DX = user pTrapHandler
; ES = user SS

	MOV		BX, DS:WORD PTR [%userTss_SP]
; ES:BX = user SS:SP
; Set the backlinked TSS CS:IP to point to the interrupt routine,
; and save the old CS in the stack so that the int routine will 
; return to the instruction following the INT instruction.
	XCHG	AX,[%userTss_CS]
    MOV		ES:WORD PTR [BX-8], AX		; CS
	XCHG	DX,[%userTss_IP]
    MOV		ES:WORD PTR [BX-12], DX		; LOW(IP)
	MOV		ES:WORD PTR [BX-10],0 		; HIGH(EIP)
; Get the flags word of the backlinked TSS and store it in the stack
; so the int routine can IRET.
	MOV		AX, DS:WORD PTR [%userTss_FL]
    MOV		ES:WORD PTR [BX-4], AX		; LOW(EFL)
	MOV		AX, DS:WORD PTR [%userTss_FL+2]
    MOV		ES:WORD PTR [BX-2], AX		; HIGH(EFL)
	SUB		DS:WORD PTR [%userTss_SP], 12
	JMP		TssToTrapMediateRet%Name
)FI
; end of not ctosv

Backlink386Tss%Name:
; We are backlinked to a 386 TSS
%SET(userTss_LDT,tss386_LDT)
%SET(userTss_SP,tss386_SP)
%SET(userTss_SS,tss386_SS)
%SET(userTss_FL,tss386_FL)
%SET(userTss_CS,tss386_CS)
%SET(userTss_IP,tss386_IP)
%SET(userTss_SS0,tss386_SS0)
%SET(userTss_SP0,tss386_SP0)
	MOV		AL, ES:BYTE PTR [%intTss_iInt]	
; AL = iInt

%IF(%Debug) THEN (
	MOV		BYTE PTR FS:int_level,AL
)FI
; check jump table of interrupted task to see if we must emulate. 
	SUB		BX, 8			; make data alias
	MOV		DS, BX
; DS points to TSS of task that caused the software interrupt
ASSUME DS: Nothing
	MOV		BX, DS:WORD PTR [%userTss_LDT]
; BX = LDT of interrupt task
%IF(%Debug) THEN (
	MOV		FS:int_LDT,BX
)FI
	OR		BX, BX		 	; no LDT?
	JNZ		Tss386LdtNotNull%Name
	JMP		NoUserTrap%Name		;User has no LDT
Tss386LdtNotNull%Name:
; Load LDT of the interrupted task.
; The LDT register points to the LDT of the interrupted task.  This will allow
; us to get selector(4) from the LDT table to access the trap jump table.
; Given iInt in AL, get index of JumpTableEntry
	LLDT	BX
; Store LDT in our TSS -- otherwise we can't debug this
; CX has old value of LDT from TSS.
	MOV		ES: WORD PTR[%intTss_LDT],BX
	MOV		BX,AX
	XOR		BH,BH
	SHL		BX,shJumpTableEntry			;Size of JumpTableEntry = 8
; BX = offset into jump table for the interrupted task.
	MOV		AX, slJumpTable
	MOV		ES, AX
; ES points to the trap jump table for the interrupted task.
	MOV		AL, ES:BYTE PTR [BX+jte_access]
%IF(%Debug) THEN (
	MOV		BYTE PTR FS:int_access,AL
)FI
	AND		AX,0Fh
	JNZ		Tss386EmulateTrap%Name
	JMP		NoUserTrap%Name		;User has no trap defined

Tss386EmulateTrap%Name:
; build the right kind of stack for the trap
	CMP		AL, l286TrapGate
	JNE		Tss386Emulate386Gate%Name

Tss386Emulate286Gate%Name:
	MOV		AX, ES:WORD PTR [BX+jte_sa]
	MOV		DX, ES:WORD PTR [BX+jte_ra]
	MOV		ES, DS:WORD PTR [%userTss_SS]
; AX:DX = user pTrapHandler
; ES = user SS

	MOV		BX, DS:WORD PTR [%userTss_SP]
; ES:BX = user SS:SP
; AX:DX has CS:IP of interrupt routine.
; Set the backlinked TSS CS:IP to point to the interrupt routine,
; and save the old CS in the stack so that the int routine will 
; return to the instruction following the INT instruction.
	XCHG	AX,[%userTss_CS]
; AX has CS of instruction that caused the software interrupt.
; Check the privilege level of the code segment.  If the level is
; not zero, then there will be a privilege level transition.
	TEST	AX,3
	JZ		Tss386Emulate286Level0%Name
; There is a privilege level transition.
; This means we must store SS and ESP on the stack we are building.
; ES:[BX]   ->
;	 [BX-2] ->	SS of user routine
;	 [BX-4] ->	SP of user routine
;	 [BX-6] ->	flags of user routine
;	 [BX-8] ->	CS of user routine that caused interrupt
;	 [BX-10]->	IP of user routine that caused interrupt
	MOV		ES:WORD PTR [BX-2], ES	; SS
	MOV		ES:WORD PTR [BX-4], BX	; SP
	SUB		BX,4
	SUB		DS:WORD PTR [%userTss_SP], 4
Tss386Emulate286Level0%Name:
; ES:[BX]   ->
;	 [BX-2] ->	flags of user routine
;	 [BX-4] ->	CS of user routine that caused interrupt
;	 [BX-6]->	IP of user routine that caused interrupt
    MOV		ES:WORD PTR [BX-4], AX	; CS
	XCHG	DX,[%userTss_IP]
    MOV		ES:WORD PTR [BX-6], DX	; IP
; Get the flags word of the backlinked TSS and store it in the stack
; so the int routine can IRET.
	MOV		AX, DS:WORD PTR [%userTss_FL]
    MOV		ES:WORD PTR [BX-2], AX	; FL
	SUB		DS:WORD PTR [%userTss_SP], 6
	JMP		TssToTrapMediateRet%Name
	
Tss386Emulate386Gate%Name:
	MOV		AX, ES:WORD PTR [BX+jte_sa]
$MOD386
	DB 		66h
	MOVZX	EDX, ES:WORD PTR [BX+jte_ra]
; AX:DX = user pTrapHandler

; Set the backlinked TSS CS:IP to point to the interrupt routine,
; and save the old CS in the stack so that the int routine will 
; return to the instruction following the INT instruction.
	XCHG	AX,[%userTss_CS]
; AX has CS of instruction that caused the software interrupt.
; Check the privilege level of the code segment.  If the level is
; not zero, then there will be a privilege level transition.
	TEST	AX,3
	JZ		Tss386Emulate386Level0%Name
; There is a privilege level transition.
; This means we must store SS and ESP on the stack we are building.
; We must first get the pointer to the SS:SP for the correct privilege level.
	MOV		BX,[%userTss_CS]
	AND		BX,3
	SHL		BX,3
	MOV		ES, [%userTss_SS0][BX]
	MOV		BX, [%userTss_SP0][BX]
; Subtract five words from the stack pointer of the privilege level.
	SUB		BX,20
; Now set the stack as follows:
; ES:[BX]->	EIP of user routine that caused interrupt
;	 [BX+4] ->	CS of user routine that caused interrupt
;	 [BX+8] ->	EFlags of user routine
;	 [BX+12] ->ESP of user routine
;	 [BX+16] ->SS of user routine
    MOV		ES:[BX+4], AX			; CS
	MOV		AX, [%userTss_SS]
	MOV		ES:[BX+16], AX			; SS
$MOD386
	MOV		EAX, [%userTss_SP]
	MOV		ES:[BX+12], EAX			; ESP
; Set up SS:SP in the TSS to have the SS:SP for the privilege level
; of the interrupt routine.
	MOV		[%userTss_SS],ES
	MOV		[%userTss_SP],BX
	JMP		SHORT Tss386Emulate386StoreIP%Name
Tss386Emulate386Level0%Name:
; Subtract three words from user stack pointer.
	SUB		WORD PTR [%userTss_SP], 12
	MOV		ES, [%userTss_SS]
	MOV		BX, [%userTss_SP]
; Now set the stack as follows:
; ES:[BX]->	EIP of user routine that caused interrupt
;	 [BX+4] ->	CS of user routine that caused interrupt
;	 [BX+8] ->	EFlags of user routine
    MOV		ES:[BX+4], AX	; CS
Tss386Emulate386StoreIP%Name:	
$MOD386
	XCHG	EDX,DWORD PTR [%userTss_IP]
    MOV		ES:DWORD PTR [BX], EDX	; EIP 

; Get the flags word of the backlinked TSS and store it in the stack
; so the int routine can IRET.
	MOV		AX, [%userTss_FL]
    MOV		ES:[BX+8], AX	; EFL (low)
	MOV		AX, [%userTss_FL+2]
    MOV		ES:[BX+10], AX	; EFL (high)
	JMP		TssToTrapMediateRet%Name

NoUserTrap%Name:
; Check to see if this was an interrupt in the range 0-31.
; If so, jump to the system interrupt handler.
	OR		DX,DX
	JZ		TssToTrapMediateRet%Name	;NOT in range 0-31
; Interrupt is in the range 0-31.
; Set ES to point to the interrupt TSS.
	STR		AX
	SUB		AX,8
	MOV		ES,AX
	JMP		JmpHandler%Name
TssToTrapMediateRet%Name:
	STR		AX
	SUB		AX,8
	MOV		ES,AX
	LLDT	CX					;Restore LDT
	MOV		ES:[%intTss_LDT],CX
%IF(%Logging) THEN (
	POP		GS
	POP		DI
)FI	
%IF(%Debug) THEN (
	POP		FS
)FI	
	POP		CX
	POP		DX					;Restore DX
	POP		AX					;Restore AX
	POP		ES					;Restore ES
	POP		BX					;Restore BX
	POP		BP					;Restore BP
	ADD		SP,6				;Discard, flHandler, ipHandler, csHandler
)

;**********************************************************************
; Come here for all device interrupts.  Determine whether the interrupt 
; is really a device interrupt or a software interrupt.
; Always come here with interrupts disabled.
;**********************************************************************
FilterDeviceInterrupt PROC FAR
ASSUME DS: Dgroup
Filter:
; Reserve space on the stack for flags and CS:IP of interrupt handler
; so we can do a RET to it.
	SUB		SP,6
	PUSH	BP
	MOV		BP,SP
ipHandler	EQU WORD PTR [BP+2]
csHandler	EQU WORD PTR [BP+4]
flHandler	EQU WORD PTR [BP+6]
; Save registers we use.  Note that we don't save DS.  
; We set it up for the TSS being filtered from the TSS
; before we invoke the interrupt handler.
	PUSH	BX
	PUSH	ES
	PUSH	AX
	PUSH	DX
	PUSH	CX
; We must set up DS with our DGroup.  We set it back to the value required
; by the handler when we exit.
	MOV		AX,DGroup
	MOV		DS,AX
%IF(%Debug) THEN (
	PUSH	FS
	MOV		FS,AX
)FI	
%IF(%Logging) THEN (
	PUSH	DI
	PUSH	GS
	MOV		AX,DGroup
	MOV		FS,AX
)FI	
	STR		BX
%if(not %ctosv)then(
; Determine which type of TSS we are dealing with: 286 or 386
; If ctosv, then all TSS are 386 because ctosv only runs on 386.
	MOV		AX,sgGdt
	MOV		ES,AX
; ES:BX points to descriptor for interrupt task TSS.  
; Test the access byte to see if this is a 286 or 386 TSS.
	TEST	BYTE PTR ES:[BX+5],mask386
	JZ		Int286Tss
	JMP		Int386TSS
; The interrupt task is a 286 TSS
Int286Tss:
%SET(intTss_iInt, tss286_iInt)
%SET(intTss_wDsIntFilter,tss286_wDsIntFilter)
%SET(intTss_wFlIntFilter,tss286_wFlIntFilter)
%SET(intTss_wIpIntFilter,tss286_wIpIntFilter)
%SET(intTss_wCsIntFilter,tss286_wCsIntFilter)
%SET(intTss_tyDev,tss286_tyDev)
%SET(intTss_LDT,tss286_LDT)
%SET(intTss_FL,tss286_FL)
%CreateTssMediator(286)
; When task is reentered later, the code that follows IRET is executed.
	IRET
	JMP		ResumeInterruptFilter
)FI

Int386TSS:
%SET(intTss_iInt, tss386_iInt)
%SET(intTss_wDsIntFilter,tss386_wDsIntFilter)
%SET(intTss_wFlIntFilter,tss386_wFlIntFilter)
%SET(intTss_wIpIntFilter,tss386_wIpIntFilter)
%SET(intTss_wCsIntFilter,tss386_wCsIntFilter)
%SET(intTss_tyDev,tss386_tyDev)
%SET(intTss_LDT,tss386_LDT)
%SET(intTss_FL,tss386_FL)
%CreateTssMediator(386)
; When task is reentered later, the code that follows IRET is executed.
	IRET
; This label is used by RemoveInterruptFilter.  When the CS:IP of a TSS
; points here, we can change it to point to the entry point of the handler
; when the filter is removed.
ResumeInterruptFilter:
	JMP		Filter

	
FilterDeviceInterrupt ENDP


NopIsr PROC FAR
; We get here if a software interrupt has occurred, and the user does
; not have a trap defined for the interrupt level, and we are running
; on a 386.  If we are running on a 286, we go to DmyIsr, which does a
; simple IRET.
	DB		66h	;operand size override
	IRET
NopIsr ENDP


IntMediation ENDS

END	

