/*---------------------------------------------------------------------
   gensync.c     11 MAR 85     Congruent Software, Inc.

   This manuscript provides the procedural interface from MS-DOS
   application program(s) to the General Synchronous Communications
   Support system service running alongside the hosted MS-DOS.  See
   each of the routine headers for the parameters needed in calling
   from the application.					     */

#include "gensync.h"

#define OPEN_RQ		0X8003
#define	CLOSE_RQ	0X8004
#define	READ_RQ		0X8005
#define	WRITE_RQ	0X8006
#define DEINSTALL_RQ	0X8007
#define	STATUS_RQ	0X8008
#define	CONTROL_RQ	0X8009

#define RQ_MISMATCH	248

typedef struct {
    unsigned *r;		/* address offset portion */
    unsigned *s;		/* address segment portion */
} ADS;

typedef struct {
    ADS address;		/* 4-byte address of parameter */
    unsigned count;		/* size of parameter pointed to */
} PB_CB;

typedef struct {
    unsigned sCntInfo;		/* size of control information */
    BYTE nReqPbCb;		/* number of request parameters */
    BYTE nRespPbCb;		/* number of response parameters */
    unsigned userNum;		/* user number */
    unsigned exchResp;		/* exchange for response */
    unsigned ercRet;		/* error code returned */
    unsigned rqCode;		/* request code */
    unsigned control[3];	/* control information... */
    PB_CB param[4];		/* parameter information... */
    unsigned dataRet;		/* for overlapped io */
} RQ_BLOCK;
/*---------------------------------------------------------------------
   Assembler language kluges to access CTOS facilities while MS-DOS is
   running.  These are required for three reasons: 1) DeSmet C pushes
   function arguments in the reverse order of CTOS languages, 2) the
   object modules RqLabl.Obj and GenSyncRqLabl.Obj are MOSTLY in Intel
   linker format---they just don't include the checksum---and thus can
   not be directly linked using the DeSmet BIND program or MS-DOS LINK
   and 3) DeSmet C doesn't use LCALLs for external procedures and DeSmet
   assembler doesn't allow the LCALL to be coded (hence the DB 09Ah,
   etc. rigmarole to represent an LCALL.  WARNING: If the locations of
   these CTOS functions change (e.g in release 10.0), this manuscript
   must also be changed.					     */


unsigned GetUserNumber(userNumberRet)
unsigned *userNumberRet;

{
#asm
    PUSH	DS
    MOV		AX,WORD [BP+4]
    PUSH	AX
    DB		09Ah
    DW		0053Ch
    DW		0FFCFh
#
}

unsigned QueryDefaultRespExch(exchRet)
unsigned *exchRet;

{
#asm
    PUSH	DS
    MOV		AX,WORD [BP+4]
    PUSH	AX
    DB		09Ah
    DW		00230h
    DW		0FFFFh
#
}

unsigned Request(rq)
RQ_BLOCK *rq;
{
#asm
    PUSH	DS
    MOV		AX,WORD [BP+4]
    PUSH	AX
    DB		09Ah
    DW		00310h
    DW		0FFEFh
#
}
unsigned Wait(exchange, pMsgRet)
unsigned exchange;
ADS *pMsgRet;

{
#asm
    MOV		AX,WORD [BP+4]
    PUSH	AX
    PUSH	DS
    MOV		AX,WORD [BP+6]
    PUSH	AX
    DB		09Ah
    DW		00214h
    DW		0FFFFh
#
}

/*---------------------------------------------------------------------
   Common function to get user number and default response exchange
   from CTOS.  Used by all the procedural interfaces that follow.    */

unsigned user_info(rq)
RQ_BLOCK *rq;

{
    unsigned erc;

    if ((erc = GetUserNumber(&rq->userNum)) == 0) /* need user ID...  */
	erc = QueryDefaultRespExch(&rq->exchResp); /* ...and exchange */
    return(erc);
}

/*---------------------------------------------------------------------
   Common Request and Wait function to reduce code size.  Used by all
   the procedural interfaces that follow.			     */

unsigned request_and_wait(rq)
RQ_BLOCK *rq;

{
    unsigned erc;
    ADS pMsgRet;

    if ((erc = Request(rq)) == 0)	/* dispatch request...     */
	if ((erc = Wait(rq->exchResp, &pMsgRet)) == 0)	/* ...wait */
	    erc = rq.ercRet;		/* return erc from gensync */
    return(erc);
}
/*---------------------------------------------------------------------
   GenSyncOpen:  Longest and dreariest of all the calls.	     */

unsigned gsc_open(gspb, device_id, terminators, char_set, modem_ctrl,
		  clocking, comm_handle)
GSPB_PTR gspb;
char *device_id, *terminators;
unsigned char_set, modem_ctrl, clocking;
unsigned *comm_handle;

{
    unsigned data_segment;	/* contents of DS from _showds() */
    unsigned erc;		/* for Request and Wait primitives */
    RQ_BLOCK rq;		/* request block for all calls */

    data_segment = _showds();	/* need segmented addresses for CTOS */
    rq.sCntInfo = 6;		/* constant among all calls */
    rq.nReqPbCb = 3;		/* gspb, device_id and terminators */
    rq.nRespPbCb = 1;
    if ((erc = user_info(&rq)) != 0)
	return(erc);		/* failure from the outset! */
    rq.ercRet = 0;		/* clean erc at start */
    rq.rqCode = OPEN_RQ;	/* insert request code */
    rq.control[0] = char_set;	/* ASCII vs. EBCDIC */
    rq.control[1] = modem_ctrl;	/* DTR, RTS, etc. asserted or not */
    rq.control[2] = clocking;	/* external vs. internal */
    rq.param[0].address.r = gspb;	/* pointer to gspb */
    rq.param[0].address.s = data_segment;
    rq.param[0].count = sizeof(GSPB);
    rq.param[1].address.r = device_id;	/* pointer to device_id */
    rq.param[1].address.s = data_segment;
    rq.param[1].count = strlen(device_id);
    rq.param[2].address.r = terminators;	/* ptr to terminators */
    rq.param[2].address.s = data_segment;
    rq.param[2].count = strlen(terminators);
    rq.param[3].address.r = comm_handle;
    rq.param[3].address.s = data_segment;
    rq.param[3].count = sizeof(unsigned);
    return(request_and_wait(&rq));
}
/*---------------------------------------------------------------------
   CloseGenSync:  Pass on the comm_handle, only, and we're done.	     */

unsigned gsc_close(comm_handle)
unsigned comm_handle;

{
    unsigned data_segment;	/* contents of DS from _showds() */
    unsigned erc;		/* for Request and Wait primitives */
    RQ_BLOCK rq;		/* request block for all calls */

    data_segment = _showds();	/* need segmented addresses for CTOS */
    rq.sCntInfo = 6;		/* constant among all calls */
    rq.nReqPbCb = 0;
    rq.nRespPbCb = 0;
    if ((erc = user_info(&rq)) != 0)
	return(erc);		/* failure from the outset! */
    rq.ercRet = 0;		/* clean erc at start */
    rq.rqCode = CLOSE_RQ;	/* request code to close */
    rq.control[0] = comm_handle;
    rq.control[1] = 0;
    rq.control[2] = 0;
    return(request_and_wait(&rq));
}
/*---------------------------------------------------------------------
   ReadGenSync :  Pass in the comm handle, the address and the size of
   the buffer, the mode and the timeout and then wait.		     */

unsigned gsc_read(comm_handle, buffer, buffer_size, data_ret, mode,
		  timeout)
unsigned comm_handle, *buffer;
unsigned buffer_size;
unsigned *data_ret;
unsigned mode, timeout;

{
    unsigned data_segment;	/* contents of DS from _showds() */
    unsigned erc;		/* for Request and Wait primitives */
    RQ_BLOCK rq;		/* request block for all calls */

    data_segment = _showds();	/* need segmented addresses for CTOS */
    rq.sCntInfo = 6;		/* constant among all calls */
    rq.nReqPbCb = 0;
    rq.nRespPbCb = 2;		/* buffer and data count returned */
    if ((erc = user_info(&rq)) != 0)
	return(erc);		/* failure from the outset! */
    rq.ercRet = 0;		/* clean erc at start */
    rq.rqCode = READ_RQ;	/* read a block from the comm line */
    rq.control[0] = comm_handle;
    rq.control[1] = mode;	/* mode: transparent or not */
    rq.control[2] = timeout;	/* 100s of milliseconds to timeout */
    rq.param[0].address.r = buffer;	/* pointer to buffer */
    rq.param[0].address.s = data_segment;
    rq.param[0].count = buffer_size;
    rq.param[1].address.r = data_ret;	/* pointer to data count returned */
    rq.param[1].address.s = data_segment;
    rq.param[1].count = sizeof(unsigned);
    return(request_and_wait(&rq));
}

/*---------------------------------------------------------------------
   WriteGenSync :  Pass in the comm handle, the address and the size
   of the buffer, the mode and the timeout and then wait.	     */

unsigned gsc_write(comm_handle, buffer, buffer_size, data_ret, mode,
		   timeout)
unsigned comm_handle, *buffer;
unsigned buffer_size;
unsigned *data_ret;
unsigned mode, timeout;

{
    unsigned data_segment;	/* contents of DS from _showds() */
    unsigned erc;		/* for Request and Wait primitives */
    RQ_BLOCK rq;		/* request block for all calls */

    data_segment = _showds();	/* need segmented addresses for CTOS */
    rq.sCntInfo = 6;		/* constant among all calls */
    rq.nReqPbCb = 1;		/* buffer */
    rq.nRespPbCb = 1;		/* data count returned */
    if ((erc = user_info(&rq)) != 0)
	return(erc);		/* failure from the outset! */
    rq.ercRet = 0;		/* clean erc at start */
    rq.rqCode = WRITE_RQ;	/* write a block out to the comm line */
    rq.control[0] = comm_handle;
    rq.control[1] = mode;	/* mode: transparent or not */
    rq.control[2] = timeout;	/* 100s of milliseconds to timeout */
    rq.param[0].address.r = buffer;	/* pointer to buffer */
    rq.param[0].address.s = data_segment;
    rq.param[0].count = buffer_size;
    rq.param[1].address.r = data_ret;	/* ptr to data count returned */
    rq.param[1].address.s = data_segment;
    rq.param[1].count = sizeof(unsigned);
    return(request_and_wait(&rq));
}
/*---------------------------------------------------------------------
   ReadGenSyncOvl:  Pass in the comm handle, the address and the size
   of the buffer, the mode and the timeout and then return.	     */

unsigned gsc_read_ovl(comm_handle, buffer, buffer_size, mode, timeout,
		      rq, exch_reply)
unsigned comm_handle, *buffer;
unsigned buffer_size;
unsigned mode, timeout;
RQ_BLOCK *rq;
unsigned exch_reply;

{
    unsigned data_segment;	/* contents of DS from _showds() */
    unsigned erc;		/* for Request and Wait primitives */

    data_segment = _showds();	/* need segmented addresses for CTOS */
    rq->sCntInfo = 6;		/* constant among all calls */
    rq->nReqPbCb = 0;
    rq->nRespPbCb = 2;		/* buffer and data count returned */
    if ((erc = GetUserNumber(&rq->userNum)) != 0)
	return(erc);		/* failure from the outset! */
    rq->exchResp = exch_reply;	/* user provided exchange for reply */
    rq->ercRet = 0;		/* clean erc at start */
    rq->rqCode = READ_RQ;	/* read a block from the comm line */
    rq->control[0] = comm_handle;
    rq->control[1] = mode;	/* mode: transparent or not */
    rq->control[2] = timeout;	/* 100s of milliseconds to timeout */
    rq->param[0].address.r = buffer;	/* pointer to buffer */
    rq->param[0].address.s = data_segment;
    rq->param[0].count = buffer_size;
    rq->param[1].address.r = &rq->dataRet;	/* pointer to data count */
    rq->param[1].address.s = data_segment;
    rq->param[1].count = sizeof(unsigned);
    rq->dataRet = 0;
    return(Request(rq));
}
/*---------------------------------------------------------------------
   WriteGenSyncOvl:  Pass in the comm handle, the address and the size
   of the buffer, the mode and the timeout and then return.	     */

unsigned gsc_write_ovl(comm_handle, buffer, buffer_size, mode, timeout,
		       rq, exch_reply)
unsigned comm_handle, *buffer;
unsigned buffer_size;
unsigned mode, timeout;
RQ_BLOCK *rq;
unsigned exch_reply;

{
    unsigned data_segment;	/* contents of DS from _showds() */
    unsigned erc;		/* for Request and Wait primitives */

    data_segment = _showds();	/* need segmented addresses for CTOS */
    rq->sCntInfo = 6;		/* constant among all calls */
    rq->nReqPbCb = 1;		/* buffer */
    rq->nRespPbCb = 1;		/* data count returned */
    if ((erc = GetUserNumber(&rq->userNum)) != 0)
	return(erc);		/* failure from the outset! */
    rq->exchResp = exch_reply;	/* user provided exchange for reply */
    rq->ercRet = 0;		/* clean erc at start */
    rq->rqCode = WRITE_RQ;	/* write a block out to the comm line */
    rq->control[0] = comm_handle;
    rq->control[1] = mode;	/* mode: transparent or not */
    rq->control[2] = timeout;	/* 100s of milliseconds to timeout */
    rq->param[0].address.r = buffer;	/* pointer to buffer */
    rq->param[0].address.s = data_segment;
    rq->param[0].count = buffer_size;
    rq->param[1].address.r = &rq->dataRet;	/* ptr to data count */
    rq->param[1].address.s = data_segment;
    rq->param[1].count = sizeof(unsigned);
    rq->dataRet = 0;
    return(Request(rq));
}
/*---------------------------------------------------------------------
   CheckReadGenSyncOvl and CheckWriteGenSyncOvl:  Wait on the
   previously issued request.  When it is done, copy the byte count
   from the intermediate request block back to the user.	     */

unsigned gsc_check_read_ovl(rq, data_ret)
RQ_BLOCK *rq;
unsigned *data_ret;

{
    unsigned erc;
    ADS pMsgRet;

    if ((erc = Wait(rq->exchResp, &pMsgRet)) != 0)
	return(erc);
    *data_ret = rq->dataRet;
    if (! ((rq & 0X000F) == (pMsgRet.r & 0X000F)))
	return(RQ_MISMATCH);
    else if (! ((_showds() + (rq >> 4)) == (pMsgRet.s +
	    (pMsgRet.r >> 4))))
	return(RQ_MISMATCH);
    else
	return(rq->ercRet);
}

unsigned gsc_check_write_ovl(rq, data_ret)
RQ_BLOCK *rq;
unsigned *data_ret;

{
    unsigned erc;
    ADS pMsgRet;

    if ((erc = Wait(rq->exchResp, &pMsgRet)) != 0)
	return(erc);
    *data_ret = rq->dataRet;
    if (! ((rq & 0X000F) == (pMsgRet.r & 0X000F)))
	return(RQ_MISMATCH);
    else if (! ((_showds() + (rq >> 4)) == (pMsgRet.s +
	    (pMsgRet.r >> 4))))
	return(RQ_MISMATCH);
    else
	return(rq->ercRet);
}
/*---------------------------------------------------------------------
   DeinstallGenSync:  Ask the system service to remove itself, if
   possible.							     */

unsigned gsc_deinstall()

{
    unsigned erc;		/* for Request and Wait primitives */
    RQ_BLOCK rq;		/* request block for all calls */

    rq.sCntInfo = 0;
    rq.nReqPbCb = 0;
    rq.nRespPbCb = 0;
    if ((erc = user_info(&rq)) != 0)
	return(erc);		/* failure from the outset! */
    rq.ercRet = 0;		/* clean erc at start */
    rq.rqCode = DEINSTALL_RQ;	/* ask for deinstallation */
    return(request_and_wait(&rq));
}
/*---------------------------------------------------------------------
   StatusGenSync and ControlGenSync:  Ask for return or modem status or
   assert modem control leads.					     */

unsigned gsc_status(comm_handle, status)
unsigned comm_handle, *status;

{
    unsigned data_segment;	/* contents of DS from _showds() */
    unsigned erc;		/* for Request and Wait primitives */
    RQ_BLOCK rq;		/* request block for all calls */

    data_segment = _showds();	/* need segmented addresses for CTOS */
    rq.sCntInfo = 6;		/* constant among all calls */
    rq.nReqPbCb = 0;
    rq.nRespPbCb = 1;		/* status returned */
    if ((erc = user_info(&rq)) != 0)
	return(erc);		/* failure from the outset! */
    rq.ercRet = 0;		/* clean erc at start */
    rq.rqCode = STATUS_RQ;	/* ask for modem status */
    rq.control[0] = comm_handle;
    rq.control[1] = 0;
    rq.control[2] = 0;
    rq.param[0].address.r = status;	/* pointer to status */
    rq.param[0].address.s = data_segment;
    rq.param[0].count = sizeof(unsigned);
    return(request_and_wait(&rq));
}

unsigned gsc_control(comm_handle, modem_ctrl)
unsigned comm_handle;
unsigned modem_ctrl;

{
    unsigned data_segment;	/* contents of DS from _showds() */
    unsigned erc;		/* for Request and Wait primitives */
    RQ_BLOCK rq;		/* request block for all calls */

    data_segment = _showds();	/* need segmented addresses for CTOS */
    rq.sCntInfo = 6;		/* constant among all calls */
    rq.nReqPbCb = 0;
    rq.nRespPbCb = 0;
    if ((erc = user_info(&rq)) != 0)
	return(erc);		/* failure from the outset! */
    rq.ercRet = 0;		/* clean erc at start */
    rq.rqCode = CONTROL_RQ;	/* assert modem control leads */
    rq.control[0] = comm_handle;
    rq.control[1] = modem_ctrl;	/* modem control word */
    rq.control[2] = 0;
    return(request_and_wait(&rq));
}
