;		******** PROPRIETARY PROGRAM MATERIAL ********
;		*********** (C) 1994 ADAPTEC, INC. ***********
;
; All rights reserved.  This software contains the valuable trade secrets of
; Adaptec.  The software is protected under copyright laws as an unpublished
; work of Adaptec.  Notice is for informational purposes only and does not
; imply publication.  The user of this software may make copies of the software
; for use with parts manufactured by Adaptec or under license from Adaptec and
; for no other use.
;
; Module:		DRVR6X60.ASM
;
; Author(s):		Peter Johansson (Congruent Software, Inc.)
;
; Abstract:		AIC-6X60 SCSI host adapter driver for ROM BIOS
;
; Base Code:		XXXXXX.00
;
; PVCS:
;
; Revisions:		17FEB94 PGJ	Initial release.

			PAGE	60, 132
			TITLE	SCSI Host Adapter Driver for AIC-6X60 BIOS
			.386P
			.MODEL	SMALL, FARSTACK

Code			SEGMENT	PUBLIC READONLY USE16 'CODE'
Code			ENDS


			ASSUME	CS:Code
			ASSUME	DS:NOTHING
			ASSUME	ES:ERROR
			ASSUME	FS:ERROR
			ASSUME	GS:ERROR
			ASSUME	SS:NOTHING

			INCLUDE	AIC6X60.INC
			INCLUDE	SCB6X60.INC
			INCLUDE	SCSI.INC

; External procedures used by the BIOS 6X60 device driver.

Code			SEGMENT

			EXTERN	In_PortA:NEAR
			EXTERN	In_PortB:NEAR

Code			ENDS

; External variables (constants, actually) specified by the customer in the
; Customer Options module, B636OPT.ASM.

			EXTERN	BasePort:ABS

Code			SEGMENT

			EXTERN	DBusOff:BYTE
			EXTERN	DBusOn:BYTE
			EXTERN	SlTime:BYTE

Code			ENDS

			PAGE

; If the 'ChkHAPresent' routine has earlier reported to its client that a 6X60
; was detected at the specified address, the procedure below is called to
; initialize the adapter.  Set the 6X60 registers to a predetermined baseline
; state.  Then assert RST for 25us.  Poll the interrupt service rouutine so
; that the RST is detected and the remainder of initialization takes place.

Code			SEGMENT

			PUBLIC	BIOS_Init

BIOS_Init	PROC	NEAR

			PUSHA
			MOV	DH, HIGH BasePort
			MOV	DL, LOW SCSISEQ
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SCSIRATE
			OUT	DX, AL
			MOV	DL, LOW SCSIDAT
			OUT	DX, AL
			CALL	In_PortA
			MOV	AH, AL		;Save copy for stack init
			MOV	DL, LOW SCSIID
			AND	AL, 07h
			SHL	AL, 4
			OUT	DX, AL
			MOV	DL, LOW CLRSINT0
			MOV	AL, 0FFh AND NOT SETSDONE
			OUT	DX, AL
			MOV	DL, LOW CLRSINT1
			MOV	AL, 0FFh
			OUT	DX, AL
			MOV	DL, LOW CLRSERR
			MOV	AL, CLRSYNCERR OR CLRFWERR OR CLRFRERR
			OUT	DX, AL
			MOV	DL, LOW SIMODE0
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SIMODE1
			OUT	DX, AL
			MOV	DL, LOW DMACNTRL0
			MOV	AL, RSTFIFO
			OUT	DX, AL
			MOV	DL, LOW BRSTCNTRL
			MOV	AL, DBusOn
			SHL	AL, 4
			OR	AL, DBusOff
			OUT	DX, AL
			MOV	DL, LOW DMACNTRL1	;Set up stack...
			MOV     AL, ENSTK32
			OUT	DX, AL
			MOV	DL, LOW STACK
			MOV	AL, 52h			;6360 BIOS signature
			OUT	DX, AL
			MOV	AL, AH
			OUT	DX, AL
			CALL	In_PortB
			MOV	DL, LOW STACK
			OUT	DX, AL
                        MOV     AH, 01h                 ;Signal 32-bit PIO
                        TEST    AL, 10h                 ;Fast SCSI enabled?
                         JZ     @F
                        OR      AH, 08h                 ;Also indicate 10 Mbs
@@:			XOR     AL, AL
                        MOV     CX, 0Eh
@@:			OUT	DX, AL
			 LOOP	@B
			MOV	AL, AH
			OUT	DX, AL
			XOR	AL, AL
			MOV	CX, 0Eh
@@:			OUT	DX, AL
			 LOOP	@B
			MOV	DL, LOW SIMODE1
			MOV	AL, ENSCSIRST
			OUT	DX, AL
			CALL	BIOS_ResetSCSI
			POPA
			CLC
			RET

BIOS_Init	ENDP

			PAGE

; This procedure returns the IRQ and the DMA channel to be used for the
; AIC-6X60 in AH and AL, respectively.  Each value is encoded by the bit
; position that is on:  the IRQ is relative to 9 (i.e. 01h represents IRQ 9 and
; 08h represents IRQ 12) while the DMA channel is relative to zero (i.e. 01h
; represents DMA channel zero while 20h represents DMA channel 5).

			PUBLIC	HA_Return_Config_Cmd

HA_Return_Config_Cmd PROC	NEAR

			PUSH	CX
			CALL	In_PortA
			MOV	AH, AL
			AND	AL, 18h
			SHR	AL, 3
			MOV	CL, AL
			MOV	AL, 01h
			SHL	AL, CL
			XCHG	AH, AL
			AND	AL, 060h
			SHR	AL, 5
			 JZ	@F
			ADD	AL, 04h
@@:			MOV	CL, AL
			MOV	AL, 1
			SHL	AL, CL
			POP	CX
			RET

HA_Return_Config_Cmd ENDP

			PAGE

; This is the single-threaded SCSI I/O procedure for the BIOS.  Upon entry,
; [BP] points to a SCSI Control Block (SCB) built on the stack by the caller.
; The SCB is initiated and when it is complete (or has failed because of an
; error), return is made to the caller with the SCB completion status and the
; target status updated.  Note that no automatic REQUEST SENSE is performed by
; the BIOS driver.  Also note that when the target disconnects, we perform an
; INT 15h to let other tasks have access to the CPU.  If a 6X60 interrupt
; occurs, this procedure is reactivated.

			PUBLIC	BIOS_Driver

BIOS_Driver		PROC	NEAR

			PUSH	DS
			MOV	DH, HIGH BasePort	;Preserved hereafter!
			MOV	(SCB PTR [BP]).ScbStatus, SCB_PENDING
			MOV	(SCB PTR [BP]).ScsiPhase, DISCONNECTED
			MOV	DL, LOW DMACNTRL1
			XOR	AL, AL
			OUT	DX, AL
			CALL	In_PortA
			MOV	AH, AL			;Need jumpers later
			SHL	AL, 1
			SBB	AL, AL
			AND	AL, 100h - ENSPCHK
			ADD	AL, ENSPCHK OR ENSTIMER
			OR	AL, SlTime
			MOV	DL, LOW SXFRCTL1
			OUT	DX, AL
			MOV	DL, LOW SIMODE0
			MOV	AL, ENSELDO OR ENSELDI OR ENSELINGO
			OUT	DX, AL
			MOV	DL, LOW SIMODE1
			MOV	AL, ENSELTIMO OR ENSCSIRST
			OUT	DX, AL
			MOV	DL, LOW SCSIID
			AND	AH, 07h			;Port A has SCSI ID
			SHL	AH, 4
			MOV	AL, (SCB PTR [BP]).TargetID
			SHR	AL, 5
			OR	AL, AH
			OUT	DX, AL
			MOV	DL, LOW SCSISEQ
			MOV	AL, ENSELO OR ENRESELI OR ENAUTOATNO
			OUT	DX, AL
MaskInterrupt:		MOV	DL, LOW DMACNTRL0
			IN	AL, DX
			AND	AL, NOT INTEN
			OUT	DX, AL
PollInterrupt:		CALL	ISR
			TEST	(SCB PTR [BP]).CurrentState, TARGET_CONNECTED
			 JNZ	PollInterrupt
			CMP	(SCB PTR [BP]).ScbStatus, SCB_PENDING
			 JE	UnmaskInterrupt
			POP	DS
			RET				;SCB done, return

UnmaskInterrupt:	MOV	DL, LOW DMACNTRL0	;May be a long wait...
			IN	AL, DX
			OR	AL, INTEN		;So permit interrupts
			OUT	DX, AL
PerhapsToDream:		MOV	DL, LOW DMASTAT
			IN	AL, DX
			TEST	AL, INTSTAT		;Interrupt yet?
			 JNZ	MaskInterrupt		;Yes, resume loop
			MOV	AX, 9000h		;No, snooze a bit
			INT	15h
			JMP	PerhapsToDream

BIOS_Driver		ENDP

			PAGE

; This is an entry point provided to the OS specific device driver for its use
; if it ever wishes to assert RST on the SCSI bus.  Note that a bogus SCB
; (referenced by [BP]) must be created so that the ISR and the procedures it
; calls do not trash the stack.

			PUBLIC	BIOS_ResetSCSI

BIOS_ResetSCSI		PROC	NEAR

			ENTER	SIZE SCB, 0
			CALL	ResetScsiBus
@@:			CALL	ISR
			MOV	DL, LOW DMASTAT
			IN	AL, DX
			TEST	AL, INTSTAT
			 JNZ	@B
			LEAVE
			RET

BIOS_ResetSCSI		ENDP

; This is the procedure that actually performs the SCSI bus RST and then polls
; the interrupt service routine to clear the RST interrupt.  It is also jumped
; to from a number of locations in the device driver code when extreme error
; conditions occur.

ResetScsiBus		PROC	NEAR

			MOV	DL, LOW DMACNTRL1
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SCSISEQ
			MOV	AL, SCSIRSTO
			OUT	DX, AL
			MOV	CX, 100h
			LOOP	$
			XOR	AL, AL
			OUT	DX, AL
			RET

ResetScsiBus		ENDP

			PAGE

; The SCSI bus is free, either because the 6X60 has dropped SEL after the
; selection timeout expired, the active SCSI target has dropped BSY or because
; RST was detected and everyone got off the bus.  First, reprogram the 6X60 to
; a known state.  This is important because all SCSI transactions, in error or
; successful, are expected to pass by this way.  The "idle" state of the 6X60
; is to watch for either RST or a target reselection.  Last, sort out the
; impact of the bus free on whatever IO was in progress.  If it was an
; attempted selection, it has timed out.  If a RST was detected, any connected
; SCSI IO is terminated.  Or, if it simply bus free, sort out whether it was
; expected or unexpected.  Before returning, clean up the SCB state variables.

ScsiBusFree		PROC	NEAR

			MOV	DL, LOW SCSISEQ
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SXFRCTL0
			MOV	AL, CHEN OR CLRSTCNT OR CLRCH
			OUT	DX, AL
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SCSIRATE
			OUT	DX, AL
			MOV	DL, LOW SCSISIG
			OUT	DX, AL
			MOV	DL, LOW SCSIDAT
			OUT	DX, AL
			MOV	DL, LOW DMACNTRL0
			MOV	AL, RSTFIFO
			OUT	DX, AL
			MOV	DL, LOW CLRSINT0
			MOV	AL, NOT (SETSDONE OR CLRSELDI)
			OUT	DX, AL
			MOV	DL, LOW CLRSINT1
			MOV	AL, 0FFh
			OUT	DX, AL
			MOV	DL, LOW SIMODE0
			MOV	AL, ENSELDI
			OUT	DX, AL
			MOV	DL, LOW SIMODE1
			MOV	AL, ENSCSIRST
			OUT	DX, AL
			MOV	DL, LOW SCSISEQ
			MOV	AL, ENRESELI
			OUT	DX, AL
			MOV	DL, LOW PORTA
			MOV	AL, LED_OFF
			OUT	DX, AL
			TEST	BH, SCSIRSTI
			 JNZ	ScsiBusReset
			TEST	BH, SELTO
			 JNZ	SelectionTimeout
			TEST	(SCB PTR [BP]).CurrentState, DISCONNECT_OK
			 JZ	UnexpectedBusFree
			CMP	(SCB PTR [BP]).ScbStatus, SCB_PENDING
			 JNE	PowerDown
			JMP	ClearNexus

ScsiBusReset:		MOV	AL, SCSI_RATES
			MOV	DL, LOW DMACNTRL1
			OUT	DX, AL
			CALL	In_PortB
			AND	AL, 08h
			XOR	AL, 08h
			SHL	AL, 4
			MOV	DL, LOW STACK
			MOV	CX, 7
@@:			OUT	DX, AL
			 LOOP	@B
			MOV	AL, SCB_SCSI_BUS_RESET
			JMP	ScbFinished

UnexpectedBusFree:	MOV	AL, SCB_BUS_FREE
			JMP	ScbFinished

SelectionTimeout:	MOV	AL, SCB_SELECTION_TIMEOUT
ScbFinished:		CMP	(SCB PTR [BP]).ScbStatus, SCB_PENDING
			 JNE	PowerDown
			MOV	(SCB PTR [BP]).ScbStatus, AL
PowerDown:		MOV	DL, LOW DMACNTRL1
			MOV	AL, PWRDWN
			OUT	DX, AL
ClearNexus:		MOV	(SCB PTR [BP]).CurrentState, 0
			MOV	(SCB PTR [BP]).ScsiPhase, DISCONNECTED
			RET

ScsiBusFree		ENDP

			PAGE

; ENSELDI has been asserted by the 6X60 to indicate either a selection or a
; reselection by another SCSI device.  Since this driver does not implement
; target mode, we can safely assume it is a reselection.  First, take a look
; at the SCSI data bus for the bit pattern to identify the selecting device.
; Only two bits should be present, our own ID and that of the other device.
; Anything else is an error and the only recourse is RST on the SCSI bus.  For
; kosher reselections, reprogram the 6X60 to be on the lookout for the first
; target REQ (expected) or a SCSI bus reset or bus free (unexpected).  Also
; set up for any synchronous data phases that may follow.

Reselection		PROC	NEAR

			CALL	In_PortA
			AND	AL, 07h
			MOV	CL, AL
			MOV	AH, 1
			SHL	AH, CL
			MOV	DL, LOW SELID
			IN	AL, DX
			XOR	AL, AH
			 JE	ResetScsiBus	;No targetID bit present!
			MOV	AH, 0FFh
CalculateTargetID:	INC	AH
			SHR	AL, 1
			 JNC	CalculateTargetID
			 JNZ	ResetScsiBus
			MOV	AL, (SCB PTR [BP]).TargetID
			SHR	AL, 5
			CMP	AH, AL
			 JNE	ResetScsiBus
			OR	(SCB PTR [BP]).CurrentState, TARGET_CONNECTED
			MOV	DL, LOW PORTA		;Show bus occupied
			MOV	AL, LED_ON
			OUT	DX, AL
			MOV	DL, LOW SCSISEQ
			MOV	AL, ENAUTOATNP
			OUT	DX, AL
			MOV	DL, LOW CLRSINT0
			MOV	AL, CLRSELDI
			OUT	DX, AL
			MOV	DL, LOW CLRSINT1
			MOV	AL, CLRBUSFREE		;OK to monitor BSY now
			OUT	DX, AL
			MOV	DL, LOW SIMODE0
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SIMODE1
			MOV	AL, ENSCSIRST OR ENBUSFREE OR ENREQINIT
			OUT	DX, AL
			CALL	GetSDTR
			MOV	DL, LOW SCSIRATE
			AND	AL, NOT 80h
			OUT	DX, AL
			RET

Reselection		ENDP

			PAGE

; Oh frabjous day! SELDO has signalled that the selection commenced by
; 'BIOS_Driver' has been acknowledged by the target.  Note that because of the
; self-clearing nature of the SELINGO interrupt bit, we may or may not have
; "seen" this interrupt prior to SELDO.  Doesn't matter, SELDO cannot occur
; without it so make sure the same housekeeping is done in either case.  Most
; of the processing here is in preparation for the MESSAGE OUT phase expected
; from the target in response to our ATN signal.  Build an IDENITFY message
; according to the information in the SCB.  Also, this is as good a time as
; any to load the current pointers into the SCB from the SCB extension.  Since
; the target might switch to a synchronous data phase after MESSAGE OUT,
; program the 6X60 for this contingency.

Selection		PROC	NEAR

			MOV	DL, LOW PORTA		;Indicate bus BSY
			MOV	AL, LED_ON
			OUT	DX, AL
			OR	(SCB PTR [BP]).CurrentState, TARGET_CONNECTED
			MOV	EAX, (SCB PTR [BP]).DataPointer
			ROR	EAX, 16
			MOV	(SCB PTR [BP]).CurrentDataPointer, EAX
			MOVZX	AX, (SCB PTR [BP]).DataLengthMSB
			SHL	EAX, 16
			MOV	AX, (SCB PTR [BP]).DataLengthLSW
			MOV	(SCB PTR [BP]).CurrentDataLength, EAX
			MOV	(SCB PTR [BP]).SegmentLength, 0
			CALL	In_PortB
			SHR	AL, 3			;Disconnect in carry
                        CMC
			SBB	CL, CL
			AND	CL, 100h - PERMIT_DISCONNECT
			ADD	CL, IDENTIFY_MSG OR PERMIT_DISCONNECT
			MOV	AL, (SCB PTR [BP]).TargetID
			MOV	AH, AL
			AND	AL, 07h			;Keep only the LUN
			OR	AL, CL
			MOV	(SCB PTR [BP]).MsgOut[0], AL
			MOV	(SCB PTR [BP]).MsgOutLen, 1
			SHR	AH, 5			;Get rid of the LUN
			CALL	GetSDTR			;Synchronous parameters
			MOV	AH, AL			;Save it in AH
			TEST	AL, 80h			;Already negotiated?
			 JZ	CreateSDTR		;No, prepare message
MonitorBUSFREE:		MOV	DL, LOW CLRSINT1
			MOV	AL, CLRBUSFREE		;OK to monitor bus free
			OUT	DX, AL
			MOV	DL, LOW SCSISEQ
			MOV	AL, ENAUTOATNP
			OUT	DX, AL
			MOV	DL, LOW SIMODE0
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SIMODE1
			MOV	AL, ENSCSIRST OR ENBUSFREE OR ENREQINIT
			OUT	DX, AL
			MOV	DL, LOW SCSIRATE	;In case of DATA phase
			MOV	AL, AH
			AND	AL, NOT 80h
			OUT	DX, AL
			RET

CreateSDTR:		ADD	(SCB PTR [BP]).MsgOutLen, 5
			MOV	(SCB PTR [BP]).MsgOut[1], EXTENDED_MSG
			MOV	(SCB PTR [BP]).MsgOut[2], SDTR_MSG_LEN
			MOV	(SCB PTR [BP]).MsgOut[3], SDTR_MSG
			MOV	(SCB PTR [BP]).MsgOut[4], MIN_XFER_PERIOD_5MB SHR 2
			MOV	(SCB PTR [BP]).MsgOut[5], MAX_REQ_ACK_OFFSET
			CALL	In_PortB		;Fast SCSI enabled?
			TEST	AL, 10h
			 JZ	MonitorBUSFREE
			MOV	(SCB PTR [BP]).MsgOut[4], MIN_XFER_PERIOD_10MB SHR 2
			JMP	MonitorBUSFREE

Selection		ENDP

			PAGE

; The interrupt service procedure below handles one interrupt condition at a
; time, ranked according to their priority.  The fact that some 6X60 interrupts
; are mutually exclusive has been taken into account in the logic design.
; Because actions taken by the interrupt service may in turn generate a new
; interrupt condition, this procedure is called from 'BIOS_Driver' as long as
; INTSTAT is asserted.
;
; In the order of priority in which the interrupt causes are examined:
;
;  * SCSI Errors:  These are not really expected (they are the fault of
;    erroneous software or else a 6X60 hardware malfunction), so take the
;    easy, drastic way out and assert RST on the SCSI bus.
;
;  * BUSFREE, SCSIRSTI or SELTO:  In all three cases, the common element is
;    that the bus is now free and whatever was underway has come to an end.
;    Note that the BUSFREE interrupt is the normal conclusion to most SCSI
;    commands.
;
;  * SELINGO:  The significance of this interrupt is that the 6X60 has
;    successfully arbitrated for the SCSI bus and has commenced selection.
;    This means the bus is no longer free, so it is OK to enable the BUSFREE
;    interrupt.  This bit is cleared by the 6X60 as soon as selection is
;    complete, so even though an IRQ has been generated we may not find
;    SELINGO asserted.
;
;  * SELDI:  A SCSI target has reselected the 6X60.
;
;  * SELDO:  A selection begun by 'BIOS_Driver' has successfully selected the
;    SCSI target.
;
;  * DMADONE:  The preceding data transfer has "gone about as fur as it can
;    go."  PHASEMIS may or may not be asserted.  It is necessary to figure
;    out how much data was transferred so that the current pointers in the
;    SCB can be adjusted.
;
;  * PHASEMIS:  The SCSI target has REQuested a new, different information
;    transfer phase.  Even if DMA was in use, DMADONE may or may not be
;    asserted.  Before servicing the target, adjust the current pointers in
;    the SCB if the preceding phase was a data transfer phase.
;
; Note that in the case of the last two interrupts, the state machine that is
; responsible for responding to the target REQuests for information transfer is
; driven ONLY by a REQ asserted on the SCSI bus.  If the expected REQ is not
; yet present, enable the REQINIT interrupt so we may exit the interrupt
; service and wait for it.

ISR			PROC	NEAR

			MOV	DL, LOW SSTAT0
			IN	AL, DX
			MOV	BL, AL
			MOV	DL, LOW SIMODE0
			IN	AL, DX
			AND	BL, AL		;Masked SSTAT0 kept in BL
			MOV	DL, LOW SSTAT1
			IN	AL, DX
			MOV	BH, AL
			MOV	DL, LOW SIMODE1
			IN	AL, DX
			AND	BH, AL		;Masked SSTAT1 kept in BL
			MOV	DL, LOW SSTAT4
			IN	AL, DX
			OR	AL, AL
			 JNZ	FatalError
			TEST	BH, SELTO OR SCSIRSTI OR BUSFREE
			 JNZ	ScsiBusFree
			TEST	BL, SELINGO
			 JNZ	SelectionInProgress
			TEST	BL, SELDI
			 JNZ	Reselection
			TEST	BL, SELDO
			 JNZ	Selection
			MOV	DL, LOW SSTAT1
			IN	AL, DX
			TEST	AL, REQINIT
			 JZ	EnableREQINIT
			TEST	(SCB PTR [BP]).ScsiPhase, NON_DATA_PHASES
			 JNZ	TargetREQuest
			JMP	UpdateDataPointer

FatalError:		MOV	DL, LOW CLRSERR
			MOV	AL, CLRSYNCERR OR CLRFWERR OR CLRFRERR
			OUT	DX, AL
			MOV	(SCB PTR [BP]).ScbStatus, SCB_PROTOCOL_ERROR
			JMP	ResetScsiBus

SelectionInProgress:	MOV	DL, LOW CLRSINT1
			MOV	AL, CLRBUSFREE
			OUT	DX, AL
			MOV	DL, LOW SIMODE0
			MOV	AL, ENSELDO OR ENSELDI
			OUT	DX, AL
			MOV	DL, LOW SIMODE1
			MOV	AL, ENSELTIMO OR ENSCSIRST OR ENBUSFREE
			OUT	DX, AL
			RET

EnableREQINIT:		MOV	DL, LOW SIMODE1
			IN	AL, DX
			OR	AL, ENREQINIT
			OUT	DX, AL
			RET

ISR			ENDP

			PAGE

; The SCSI data pointer is implemented as a current buffer location and a
; remaining transfer length in the SCB.  When a 6X60 interrupt has terminated a
; DATA OUT or DATA IN phase, it is time to update these data pointers.  If we
; have been in bitbucket mode, there must be a new phase requested; no updates
; are necessary, just nuke the bitbucket mode. Otherwise, programmed IO was
; used for the phase just finished, so we may need to either retrieve some data
; remaining in the 6X60 FIFO's (DATA IN phase) or adjust the data pointer
; backwards by data left over in the FIFO's, already accounted for in the data
; pointer but not accepted by the target (DATA OUT phase).  Before we leave, if
; there is a new phase requested, clear all the FIFO's of (now) unecessary
; data.

UpdateDataPointer	PROC	NEAR

			TEST	(SCB PTR [BP]).CurrentState, BITBUCKET
			 JNZ	CleanUpBitBucket
			TEST	BH, PHASEMIS
			 JZ	CheckFifoResidual
			CMP	(SCB PTR [BP]).ScsiPhase, DATA_IN_PHASE
			 JNE	ClearScsiChannel
			MOV	DL, LOW FIFOSTAT
			IN	AL, DX
			OR	AL, AL
			 JZ	ClearScsiChannel
			CALL	DataInPhase
ClearScsiChannel:	MOV	DL, LOW SXFRCTL0
			MOV	AL, CHEN OR CLRSTCNT OR CLRCH
			OUT	DX, AL
			MOV	AL, CHEN
			OUT	DX, AL
			MOV	DL, LOW DMACNTRL0
			MOV	AL, RSTFIFO
			OUT	DX, AL
			JMP	TargetREQuest

CheckFifoResidual:	MOV	DL, LOW FIFOSTAT
			XOR	EAX, EAX
			IN	AL, DX
			MOV	ECX, EAX
			MOV	DL, LOW SSTAT2
			IN	AL, DX
			AND	AL, SFCNT
			ADD	ECX, EAX
			 JZ	ClearScsiChannel
			SUB	(SCB PTR [BP]).CurrentDataPointer, ECX
			ADD	(SCB PTR [BP]).CurrentDataLength, ECX
			ADD	(SCB PTR [BP]).SegmentLength, CX
			JMP	ClearScsiChannel

CleanUpBitBucket:	AND	(SCB PTR [BP]).CurrentState, NOT BITBUCKET
			MOV	DL, LOW SXFRCTL1
			IN	AL, DX
			AND	AL, NOT BITBUCKET
			OUT	DX, AL
			JMP	ClearScsiChannel

UpdateDataPointer	ENDP

			PAGE

; You may think it's the SCSI host adapter in charge of things, but it's really
; the target cracking the whip and REQuesting bytes of information.  The
; procedure below is just a state machine that attempts to keep the target
; happy no matter what information transfer phase transitions it leads us
; through.

TargetREQuest		PROC	NEAR

			AND	(SCB PTR [BP]).CurrentState, NOT SPIORDY
			MOV	DL, LOW SSTAT0
			IN	AL, DX
			AND	AL, SPIORDY
			OR	(SCB PTR [BP]).CurrentState, AL
			MOV	DL, LOW SIMODE0
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SIMODE1
			MOV	AL, ENSCSIRST OR ENPHASEMIS OR ENBUSFREE
			OUT	DX, AL
			MOV	DL, LOW SXFRCTL0
			MOV	AL, CHEN OR CLRSTCNT
			OUT	DX, AL
			MOV	DL, LOW SCSISIG
			IN	AL, DX
			AND	AL, CD OR IO OR MSG OR ATN
			OUT	DX, AL
			AND	AL, SCSI_PHASE_MASK
			MOV	(SCB PTR [BP]).ScsiPhase, AL
			SHR	AL, 4
			MOVZX	BX, AL
			MOV	DL, LOW CLRSINT1
			MOV	AL, CLRPHASECHG
			OUT	DX, AL
			JMP	PhaseDispatch [BX]

PhaseDispatch		DW	OFFSET DataOutPhase
			DW	OFFSET ResetScsiBus
			DW	OFFSET DataInPhase
			DW	OFFSET ResetScsiBus
			DW	OFFSET CommandPhase
			DW	OFFSET MessageOutPhase
			DW	OFFSET StatusPhase
			DW	OFFSET MessageInPhase

TargetREQuest		ENDP

			PAGE

; Programmed IO data transfer during a DATA OUT phase makes use of block IO
; instructions to transfer 16 bits of data for each IO register reference into
; the 6X60 host FIFO until all the available data is exhausted or an interrupt
; condition occurs.  Note that the data pointer in the SCB is advanced as we
; go along.  This means that we may have to adjust it back down if there is
; data left in the FIFO's at the end of the data transfer phase (see
; 'updateDataPointer').

DataOutPhase		PROC	NEAR

			CMP	(SCB PTR [BP]).CurrentDataLength, 0
			 JZ	DataOverrun
			MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_16 OR FIFO_PIO OR DWORDPIO OR FIFO_WRITE
			OUT	DX, AL
			MOV	BL, AL			;Reference copy
			MOV	DL, LOW SXFRCTL0
			MOV	AL, SCSIEN OR DMAEN OR CHEN
			OUT	DX, AL
DataOutLoop:		CMP	(SCB PTR [BP]).CurrentDataLength, 0
			 JE	DataOutComplete
			LDS	SI, (SCB PTR [BP]).CurrentDataPointer
			CMP	(SCB PTR [BP]).SegmentLength, 0
			 JNE	WaitForFifoEmpty
			MOVZX	EAX, SI
			NEG	AX
                         JNZ    @F
                        MOV     AX, 8000h               ;Arbitrary half segment
@@:			MOV	ECX, (SCB PTR [BP]).CurrentDataLength
			CMP	EAX, ECX
			 JBE	@F
			MOV	AX, CX
@@:			MOV	(SCB PTR [BP]).SegmentLength, AX
WaitForFifoEmpty:	MOV	DL, LOW DMASTAT
			IN	AL, DX
			TEST	AL, INTSTAT
			 JNZ	DataOutExit
			TEST	AL, DFIFOEMP
			 JZ	WaitForFifoEmpty
			MOV	ECX, FIFO_SIZE
			CMP	CX, (SCB PTR [BP]).SegmentLength
			 JBE	@F
			MOV	CX, (SCB PTR [BP]).SegmentLength
@@:			ADD	(SCB PTR [BP]).CurrentDataOffset, CX
			 JNC	@F
			ADD	(SCB PTR [BP]).CurrentDataSelector, 1000h
@@:			SUB	(SCB PTR [BP]).CurrentDataLength, ECX
			SUB	(SCB PTR [BP]).SegmentLength, CX
			CLD
DataOut32:		TEST	CL, 03h
			 JNZ	DataOut16
			CMP	BL, ENDMA OR FIFO_16 OR FIFO_PIO OR DWORDPIO OR FIFO_WRITE
			 JE	@F
			MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_16 OR FIFO_PIO OR DWORDPIO OR FIFO_WRITE
			OUT	DX, AL
			MOV	BL, AL
@@:			MOV	DL, LOW DMADATA32
			SHR	CX, 2
			REP OUTSD
			JMP	DataOutLoop

DataOut16:		TEST	CL, 01h
			 JNZ	DataOut8
			CMP	BL, ENDMA OR FIFO_16 OR FIFO_PIO OR FIFO_WRITE
			 JE	@F
			MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_16 OR FIFO_PIO OR FIFO_WRITE
			OUT	DX, AL
			MOV	BL, AL
@@:			MOV	DL, LOW DMADATA
			SHR	CX, 1
			REP OUTSW
			JMP	DataOutLoop

DataOut8:		CMP	BL, ENDMA OR FIFO_8 OR FIFO_PIO OR FIFO_WRITE
			 JE	@F
			MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_8 OR FIFO_PIO OR FIFO_WRITE
			OUT	DX, AL
			MOV	BL, AL
@@:			MOV	DL, LOW DMADATA
			REP OUTSB
			JMP	DataOutLoop

DataOutComplete:	MOV	DL, LOW DMASTAT
			IN	AL, DX
			TEST	AL, INTSTAT
			 JNZ	DataOutExit
			TEST	AL, DFIFOEMP
			 JZ	DataOutComplete
			MOV	DL, LOW SSTAT2
			IN	AL, DX
			TEST	AL, SEMPTY
			 JZ	DataOutComplete
			MOV	DL, LOW SIMODE1
			MOV	AL, ENSCSIRST OR ENPHASEMIS OR ENBUSFREE OR ENREQINIT
			OUT	DX, AL
DataOutExit:		RET

DataOutPhase		ENDP

			PAGE

; Programmed IO data transfer during a DATA IN phase is essentially the same as
; during data out, except for the direction.  See the comments for the
; preceding routine.  A significant difference, though, is that we cannot
; overshoot the amount of data transferred (as in a DATA OUT phase) so there is
; never any need for retroactive data pointer adjustment when the phase
; completes.

DataInPhase		PROC	NEAR

			CMP	(SCB PTR [BP]).CurrentDataLength, 0
			 JZ	DataOverrun
			MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_16 OR FIFO_PIO OR DWORDPIO OR FIFO_READ
			OUT	DX, AL
			MOV	BL, AL			;Reference copy
			MOV	DL, LOW SXFRCTL0
			MOV	AL, SCSIEN OR DMAEN OR CHEN
			OUT	DX, AL
DataInLoop:		CMP	(SCB PTR [BP]).CurrentDataLength, 0
			 JE	DataInComplete
			LES	DI, (SCB PTR [BP]).CurrentDataPointer
			CMP	(SCB PTR [BP]).SegmentLength, 0
			 JNE	WaitForFifoFull
			MOVZX	EAX, DI
			NEG	AX
                         JNZ    @F
                        MOV     AX, 8000h               ;Arbitrary half segment
@@:			MOV	ECX, (SCB PTR [BP]).CurrentDataLength
			CMP	EAX, ECX
			 JBE	@F
			MOV	AX, CX
@@:			MOV	(SCB PTR [BP]).SegmentLength, AX
WaitForFifoFull:	MOV	DL, LOW DMASTAT
			IN	AL, DX
			TEST	AL, INTSTAT OR DFIFOFULL
			 JZ	WaitForFifoFull
			MOV	ECX, FIFO_SIZE
			TEST	AL, DFIFOFULL
			 JNZ	@F
			MOV	DL, LOW FIFOSTAT
			IN	AL, DX
			OR	AL, AL
			 JZ	DataInExit
			MOVZX	ECX, AL
@@:			CMP	CX, (SCB PTR [BP]).SegmentLength
			 JBE	@F
			MOV	CX, (SCB PTR [BP]).SegmentLength
@@:			ADD	(SCB PTR [BP]).CurrentDataOffset, CX
			 JNC	@F
			ADD	(SCB PTR [BP]).CurrentDataSelector, 1000h
@@:			SUB	(SCB PTR [BP]).CurrentDataLength, ECX
			SUB	(SCB PTR [BP]).SegmentLength, CX
			CLD
DataIn32:		TEST	CL, 03h
			 JNZ	DataIn16
			CMP	BL, ENDMA OR FIFO_16 OR FIFO_PIO OR DWORDPIO OR FIFO_READ
			 JE	@F
			MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_16 OR FIFO_PIO OR DWORDPIO OR FIFO_READ
			OUT	DX, AL
			MOV	BL, AL
@@:			MOV	DL, LOW DMADATA32
			SHR	CX, 2
			REP INSD
			JMP	DataInLoop

DataIn16:		TEST	CL, 01h
			 JNZ	DataIn8
			CMP	BL, ENDMA OR FIFO_16 OR FIFO_PIO OR FIFO_READ
			 JE	@F
			MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_16 OR FIFO_PIO OR FIFO_READ
			OUT	DX, AL
			MOV	BL, AL
@@:			MOV	DL, LOW DMADATA
			SHR	CX, 1
			REP INSW
			JMP	DataInLoop

DataIn8:		CMP	BL, ENDMA OR FIFO_8 OR FIFO_PIO OR FIFO_READ
			 JE	@F
			MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_8 OR FIFO_PIO OR FIFO_READ
			OUT	DX, AL
			MOV	BL, AL
@@:			MOV	DL, LOW DMADATA
			REP INSB
			JMP	DataInLoop

DataInComplete:		MOV	DL, LOW SIMODE1
			MOV	AL, ENSCSIRST OR ENPHASEMIS OR ENBUSFREE OR ENREQINIT
			OUT	DX, AL
DataInExit:		RET

DataInPhase		ENDP

			PAGE

; Use the FIFO for the command phase for efficiency.  Configure the 6X60 for
; "SCSI normal" transfer mode and then copy all the CDB bytes into the FIFO.
; If there are any errors (such as the target refuses to take all the command
; bytes), these will surface naturally later and don't require any special
; error checking here.

CommandPhase            PROC    NEAR

                        MOV	DL, LOW DMACNTRL0
			MOV	AL, ENDMA OR FIFO_8 OR FIFO_PIO OR FIFO_WRITE OR RSTFIFO
			OUT	DX, AL
			MOV	DL, LOW SXFRCTL0
			MOV	AL, CHEN OR SCSIEN OR DMAEN
			OUT	DX, AL
			MOV	DL, LOW DMADATA
			MOV	AX, SS
			MOV	DS, AX
			LEA	SI, (SCB PTR [BP]).Cdb
			MOVZX	CX, (SCB PTR [BP]).CdbLength
			CLD
			REP OUTSB
			RET

CommandPhase            ENDP

; The status phase is, of course, supposed to be only one byte long, but stay
; in this loop as long as the target remains in STATUS phase and has bytes to
; transfer.  If a parity error occurs, send an INITIATOR DETECTED ERROR
; message and hope that the target firmware is clever enough to retry the
; status byte.  If it is, we'll pick it up the second time around and clear
; the  parity error.

StatusPhase		PROC	NEAR

			CALL	ConfigureScsiPIO
StatusLoop:		CALL	SamePhaseREQuest
			 JNZ	@F
			CALL	CheckParityError
			 JNZ	StatusParityError
			MOV	DL, LOW SCSIDAT
			IN	AL, DX
			MOV	(SCB PTR [BP]).TargetStatus, AL
			JMP	StatusLoop

StatusParityError:	MOV	AL, INITIATOR_DETECTED_ERROR_MSG
			CALL	PrepareMsgOut
			MOV	DL, LOW SCSIDAT
			IN	AL, DX
			OR	(SCB PTR [BP]).CurrentState, PARITY_ERROR
			JMP	StatusLoop

@@:			RET

StatusPhase		ENDP

			PAGE

; When the target REQuests a MESSAGE OUT phase, normally a message has been
; already built in the SCB.  There are two cases when this is not true: a)
; a parity error was detected in the preceding DATA IN phase ant the 6X60
; automatically asserted ATN to request the current MESSAGE OUT phase or b)
; the target has requested an unexpected MESSAGE OUT phase.  In the first
; case, prepare an INITIATOR DETECTED ERROR message to let the target know
; about the parity error and in the second prepare an ABORT message.  In any
; case, once a message is ready transmit all of its bytes until done or the
; SCSI phase changes.  Once the message has been sent, we may have to examine
; it for special cases.  Note that the message out area in the SCB is used
; to retain the SDTR message so that me may later compare it with an inbound
; SDTR message sent by the target in reply (see 'NegotiateSDTR').

MessageOutPhase 	PROC	NEAR

			AND	(SCB PTR [BP]).CurrentState, NOT MSG_REFUSED
			CLD
			CALL	CheckParityError
			 JZ	CheckMsgOutLength
			OR	(SCB PTR [BP]).CurrentState, PARITY_ERROR
			MOV	AL, INITIATOR_DETECTED_ERROR_MSG
			CALL	PrepareMsgOut
CheckMsgOutLength:	CMP	(SCB PTR [BP]).MsgOutLen, 0
			 JNE	@F
			MOV	AL, ABORT_MSG
			CALL	PrepareMsgOut
@@:			CALL	ConfigureScsiPIO
			MOVZX	CX, (SCB PTR [BP]).MsgOutLen
			MOV	AX, SS
			MOV	DS, AX
			LEA	SI, (SCB PTR [BP]).MsgOut
MsgOutLoop:		CALL	SamePhaseREQuest
			 JNZ	MsgOutComplete
			CMP	CX, 1			;Last message byte?
			 JNE	@F
			MOV	DL, LOW CLRSINT1	;Yes, clear ATN
			MOV	AL, CLRATNO
			OUT	DX, AL
@@:			MOV	DL, LOW SCSIDAT
			OUTSB				;Assert the byte
			 LOOP	MsgOutLoop
MsgOutComplete:		OR	CX, CX			;Entire message sent?
			 JZ	@F
			MOV	DL, LOW CLRSINT1	;No, lower ATN anyway
			MOV	AL, CLRATNO
			OUT	DX, AL
			OR	(SCB PTR [BP]).CurrentState, MSG_REFUSED
@@:			MOVZX	CX, (SCB PTR [BP]).MsgOutLen
			MOV	(SCB PTR [BP]).MsgOutLen, 0
			LEA	SI, (SCB PTR [BP]).MsgOut
			DEC	CX			;Single byte message?
			 JNZ	NormalizeMsgOut	 	;No, adjust it
			TEST	BYTE PTR [SI], IDENTIFY_MSG	;IDENTIFY?
			 JZ	ParseMsgOut		;No, check it out
                        OR      (SCB PTR [BP]).CurrentState, NEXUS_ESTABLISHED
MsgOutExit:		RET

NormalizeMsgOut:	OR      (SCB PTR [BP]).CurrentState, NEXUS_ESTABLISHED
                        MOV	DI, SI
			INC	SI
			MOV	AX, DS
			MOV	ES, AX
			REP MOVSB
			LEA	SI, (SCB PTR [BP]).MsgOut
ParseMsgOut:		LODSB
			CMP	AL, ABORT_MSG
			 JE	DisconnectOK
			CMP	AL, INITIATOR_DETECTED_ERROR_MSG
			 JE	ClearParityError
			CMP	AL, MESSAGE_PARITY_ERROR_MSG
			 JE	ClearMsgParityError
			CMP	AL, EXTENDED_MSG	;Potential SDTR?
			 JNE	MsgOutExit
			LODSB
			CMP	AL, SDTR_MSG_LEN
			 JNE	MsgOutExit
			LODSB
			CMP	AL, SDTR_MSG
			 JNE	MsgOutExit
			TEST	(SCB PTR [BP]).CurrentState, MSG_REFUSED
			 JZ	MsgOutExit
			XOR	CL, CL			;Revert to asynchronous
			JMP	UpdateSDTR

DisconnectOK:		OR	(SCB PTR [BP]).CurrentState, DISCONNECT_OK
			RET

ClearParityError:	AND	(SCB PTR [BP]).CurrentState, NOT PARITY_ERROR
			RET

ClearMsgParityError:	AND	(SCB PTR [BP]).CurrentState, NOT MSG_PARITY_ERROR
			RET

MessageOutPhase		ENDP

			PAGE

; We stay in MESSAGE IN processing until the target requests a new phase.  A
; message buffer in the SCB is used to accumulate multiple byte, extended
; messages (e.g. SDTR); the other single byte messages are interpreted and
; discarded from AL on the fly.  Note that we must read the message byte from
; unlatched SCSI bus data in order not to assert ACK (we may wish to raise ATN
; before ACK is asserted).  At the end of processing each message byte, we read
; it a second time from latched SCSI data to generate the ACK.  Any parity
; errors detected when a message byte are read force ATN so we may send a
; MESSAGE PARITY ERROR message to the target.  Any message bytes accumulated
; thus far are discarded.

MessageInPhase		PROC	NEAR

			LEA	DI, (SCB PTR [BP]).MsgIn
			MOV	(SCB PTR [BP]).MsgInLen, 0
			CALL	ConfigureScsiPIO
MsgInLoop:		CALL	SamePhaseREQuest
			 JNZ	MsgInExit
			CALL	CheckParityError
			 JNZ	MsgInParityError
			TEST	(SCB PTR [BP]).CurrentState, MSG_PARITY_ERROR
			 JNZ	ACKnowledgeMsgByte
			MOV	DL, LOW SCSIBUS
			IN	AL, DX
			TEST	AL, IDENTIFY_MSG
			 JZ	VerifyNexus
			TEST	(SCB PTR [BP]).CurrentState, NEXUS_ESTABLISHED
			 JNZ	UnexpectedReconnect
			XOR	AL, (SCB PTR [BP]).TargetID
			AND	AL, 07h
			 JNZ	UnexpectedLUN
			OR	(SCB PTR [BP]).CurrentState, NEXUS_ESTABLISHED
RestorePointersMsg:	MOV	EAX, (SCB PTR [BP]).DataPointer
			ROR	EAX, 16
			MOV	(SCB PTR [BP]).CurrentDataPointer, EAX
			MOVZX	AX, (SCB PTR [BP]).DataLengthMSB
			SHL	EAX, 16
			MOV	AX, (SCB PTR [BP]).DataLengthLSW
			MOV	(SCB PTR [BP]).CurrentDataLength, EAX
			MOV	(SCB PTR [BP]).SegmentLength, 0
			JMP	ACKnowledgeMsgByte

MsgInExit:		RET

MsgInParityError:	OR	(SCB PTR [BP]).CurrentState, MSG_PARITY_ERROR
			MOV	AL, MESSAGE_PARITY_ERROR_MSG
			JMP	RequestMsgOut

UnexpectedReconnect:	MOV	AL, ABORT_MSG
			JMP	RequestMsgOut

UnexpectedLUN:
NoNexusEstablished:	MOV	AL, BUS_DEVICE_RESET_MSG
			JMP	RequestMsgOut

VerifyNexus:		TEST	(SCB PTR [BP]).CurrentState, NEXUS_ESTABLISHED
			 JZ	NoNexusEstablished
                        CMP     (SCB PTR [BP]).MsgInLen, 0
                         JNE    ExtendedMsg
			CMP	AL, MESSAGE_REJECT_MSG	;Message OUT OF BOUNDS?
			 JA	UnsupportedMsg
			MOVZX	BX, AL
			SHL	BL, 1
			JMP	MessageDispatch [BX]

MessageDispatch		DW	OFFSET CommandCompleteMsg
			DW	OFFSET ExtendedMsg
			DW	OFFSET SaveDataPointerMsg
			DW	OFFSET RestorePointersMsg
			DW	OFFSET DisconnectMsg
			DW	OFFSET UnsupportedMsg
			DW	OFFSET UnsupportedMsg
			DW	OFFSET MessageRejectedMsg

CommandCompleteMsg:	MOV	(SCB PTR [BP]).ScbStatus, SCB_COMPLETED_OK
DisconnectMsg:		OR	(SCB PTR [BP]).CurrentState, DISCONNECT_OK
			JMP	ACKnowledgeMsgByte

SaveDataPointerMsg:	MOV	EAX, (SCB PTR [BP]).CurrentDataPointer
			ROR	EAX, 16
			MOV	(SCB PTR [BP]).DataPointer, EAX
			MOV	EAX, (SCB PTR [BP]).CurrentDataLength
			MOV	(SCB PTR [BP]).DataLengthLSW, AX
			SHR	EAX, 16
			MOV	(SCB PTR [BP]).DataLengthMSB, AL
			JMP	ACKnowledgeMsgByte

ExtendedMsg:		STOSB				;Accumulate the message
			INC	(SCB PTR [BP]).MsgInLen
			CMP	(SCB PTR [BP]).MsgInLen, 1
			 JE	ACKnowledgeMsgByte
			MOV	AL, (SCB PTR [BP]).MsgIn[1]
			ADD	AL, 2
			CMP	AL, (SCB PTR [BP]).MsgInLen
			 JA	ACKnowledgeMsgByte
			LEA	DI, (SCB PTR [BP]).MsgIn
			MOV	(SCB PTR [BP]).MsgInLen, 0
			CMP	(SCB PTR [BP]).MsgIn[2], SDTR_MSG
			 JNE	UnsupportedMsg
			LEA	DI, (SCB PTR [BP]).MsgIn
			MOV	(SCB PTR [BP]).MsgInLen, 0
			CALL	NegotiateSDTR
			JMP	ACKnowledgeMsgByte

MessageRejectedMsg:	TEST	(SCB PTR [BP]).MsgOut, IDENTIFY_MSG
			 JZ	CheckRejectedSDTR
			MOV	(SCB PTR [BP]).ScbStatus, SCB_INVALID_LUN
			JMP	ACKnowledgeMsgByte

CheckRejectedSDTR:	CMP	(SCB PTR [BP]).MsgOut[0], EXTENDED_MSG
			 JNE	MsgRejected
			CMP	(SCB PTR [BP]).MsgOut[1], SDTR_MSG_LEN
			 JNE	MsgRejected
			CMP	(SCB PTR [BP]).MsgOut[2], SDTR_MSG
			 JNE	MsgRejected
			XOR	CL, CL
			CALL	UpdateSDTR
			JMP	ACKnowledgeMsgByte

MsgRejected:		MOV	(SCB PTR [BP]).ScbStatus, SCB_MESSAGE_REJECTED
			MOV	(SCB PTR [BP]).MsgOutLen, 1
			MOV	(SCB PTR [BP]).MsgOut, NOP_MSG
			JMP	ACKnowledgeMsgByte

UnsupportedMsg:		MOV	AL, MESSAGE_REJECT_MSG
RequestMsgOut:		CALL	PrepareMsgOut
ACKnowledgeMsgByte:	MOV	DL, LOW SCSIDAT
			IN	AL, DX
			JMP	MsgInLoop

MessageInPhase		ENDP

			PAGE

; For short transfers of information in 6X60 manual programmed IO mode (only
; MESSAGE IN, MESSAGE and STATUS bytes) there is less overhead to just wait
; on the SPIORDY bit than there is to enable an interrupt and leave the
; interrupt service.  If any other interrupt condition arises, we must bail out
; as well.  This does make the simplifying assumption that all targets are
; going to be reasonably prompt with these sorts of information transfers.

SamePhaseREQuest	PROC	NEAR

			TEST	(SCB PTR [BP]).CurrentState, SPIORDY
			MOV	DL, LOW DMASTAT
			IN	AL, DX
			 JNZ	SamePhaseExit
			TEST	AL, INTSTAT
			 JNZ	SamePhaseExit
			MOV	DL, LOW SSTAT0
			IN	AL, DX
			TEST	AL, SPIORDY
			 JZ	SamePhaseREQuest
			MOV	DL, LOW DMASTAT
			IN	AL, DX
SamePhaseExit:		AND	(SCB PTR [BP]).CurrentState, NOT SPIORDY
			TEST	AL, INTSTAT
			RET

SamePhaseREQuest	ENDP

; Because the SCSI target is actually in control of all the information
; transfer phases, sometimes we have to humor it and either accept (and
; discard) ANY data it sends us or generate pad data to send it.  The 6X60 does
; this automatically, in either direction, if we assert BITBUCKET.  It's also
; important to assert ATN before we start eventually the target will tire of
; data transfer and should transition to a MESSAGE phase.  If and when that
; happens, the target will be properly quashed with an ABORT message.

DataOverrun		PROC	NEAR

			MOV	(SCB PTR [BP]).ScbStatus, SCB_DATA_OVERRUN
			OR	(SCB PTR [BP]).CurrentState, BITBUCKET
			MOV	AL, ABORT_MSG
			CALL	PrepareMsgOut
			MOV	DL, LOW SXFRCTL1
			IN	AL, DX
			OR	AL, BITBUCKET
			OUT	DX, AL
			RET

DataOverrun		ENDP

			PAGE

; Synchronous data transfer negotiations should be able to complete with the
; exchange of a message from the initiator and one from the target, but this
; procedure will accommodate redundant targets that insist on echoing any
; changed values until both messages match each other exactly.  If the target
; proposes a synchronous agreement with an SDTR, the 6X60 accepts it by
; responding with an SDTR whose minimum transfer period is greater than or
; equal to that suggested by the target and whose REQ/ACK offset field is less
; than or equal to that suggested by the target.  The only case in which
; synchronous negotiations are refused is when the target wishes to transfer
; data at too slow a rate.  The latter is a 6X60 limitation:  the largest
; transfer period programmable in the SCSIRATE register is 9 clock periods
; (450ns with a 20 MHz clock).  When the 6X60 initiates SDTR negotiations
; (a user configurable option), this procedure checks for a) the return of an
; SDTR message with a non-zero REQ/ACK offset (synchronous is OK), b) the
; return of an SDTR message with zero in the REQ/ACK offset (use asynchronous)
; or c) the return of a MESSAGE REJECT message (use asynchronous mode).  Note
; that although described in the narration here, some of these events actually
; take place in the higher level 'InterpretMessageIn' procedure that calls
; this one.

NegotiateSDTR		PROC	NEAR

			CMP	(SCB PTR [BP]).MsgIn[4], 0
			 JE	Asynchronous
			MOVZX	AX, (SCB PTR [BP]).MsgIn[3]
			SHL	AX, 2
			ADD	AX, CLOCK_PERIOD - 1
			MOV	CL, CLOCK_PERIOD
			DIV	CL
			CMP	AL, 9
			 JA	RejectSDTR
			MOV	AH, AL
			CALL	In_PortA
			SHR	AL, 4
			MOV	CL, AL
			MOV	AL, MIN_XFER_CLOCKS_5MB
			SHR	AL, CL
			CMP	AL, AH
			 JAE	@F
			MOV	AL, AH
@@:			MOV	CL, (SCB PTR [BP]).MsgIn[4]
			CMP	CL, MAX_REQ_ACK_OFFSET
			 JAE	@F
			MOV	CL, MAX_REQ_ACK_OFFSET
@@:			SUB	AL, 2
			SHL	AL, 4
			OR	CL, AL		;Composite SCSIRATE value
			CMP	(SCB PTR [BP]).MsgIn[0], EXTENDED_MSG
			 JNE	BuildSDTR
			CMP	(SCB PTR [BP]).MsgIn[1], SDTR_MSG_LEN
			 JNE	BuildSDTR
			CMP	(SCB PTR [BP]).MsgIn[2], SDTR_MSG
			 JE	UpdateSDTR
BuildSDTR:		MOV	(SCB PTR [BP]).MsgOutLen, 5
			MOV	(SCB PTR [BP]).MsgOut[0], EXTENDED_MSG
			MOV	(SCB PTR [BP]).MsgOut[1], SDTR_MSG_LEN
			MOV	(SCB PTR [BP]).MsgOut[2], SDTR_MSG
			MOV	AL, CL
			SHR	AL, 4
			MOV	DL, CLOCK_PERIOD
			MUL	DL		;AX = period in nanoseconds
			SHR	AL, 2
			MOV	(SCB PTR [BP]).MsgOut[3], AL
			MOV	AL, CL
			AND	AL, 0Fh		;Maximum REQ/ACK offset
			MOV	(SCB PTR [BP]).MsgOut[4], AL
			MOV	DL, LOW SCSISIG
			IN	AL, DX
			AND	AL, SCSI_PHASE_MASK
			OR	AL, ATN
			OUT	DX, AL
@@:			JMP	UpdateSDTR

RejectSDTR:		MOV	AL, MESSAGE_REJECT_MSG
			CALL	PrepareMsgOut
Asynchronous:		XOR	CL, CL
			JMP	UpdateSDTR

NegotiateSDTR		ENDP

			PAGE

; Enter with the target ID in AH and use it to obtain the current SDTR
; agreement from the AIC-6X60 stack.  The agreement is stored as an image of
; the SCSIRATE register, except that the most significant bit is clear if no
; negotiations have taken place yet.

GetSDTR			PROC	NEAR

			CALL	In_PortA
			SUB	AL, AH
			AND	AL, 07h
			ADD	AL, SCSI_RATES - 1
			MOV	DL, LOW DMACNTRL1
			OUT	DX, AL
			MOV	DL, LOW STACK
			IN	AL, DX
			RET

GetSDTR			ENDP

; When SDTR negotiations are completed (either because a pair of SDTR messages
; have been exchanged, a MESSAGE REJECT message has been received or the target
; refused the MESSAGE phase all together), it is necessary to record the
; results in the 6X60 stack and to program the SCSIRATE register.  The stack
; is used to save synchronous agreements for later use by the BIOS and also to
; communicate these agreements to other software components (e.g. OS drivers)
; that inherit from the BIOS.

UpdateSDTR		PROC	NEAR

			CALL	In_PortA
			MOV	AH, (SCB PTR [BP]).TargetID
			SHR	AH, 5
			SUB	AL, AH
			AND	AL, 07h
			ADD	AL, SCSI_RATES - 1
			MOV	DL, LOW DMACNTRL1
			OUT	DX, AL
			MOV	DL, LOW STACK
			MOV	AL, CL
			OR	AL, 80h
			OUT	DX, AL
			MOV	DL, LOW SCSIRATE
			MOV	AL, CL
			OUT	DX, AL
			RET

UpdateSDTR		ENDP

			PAGE

; This procedure is used to check for the occurence of a SCSI parity error
; during a MESSAGE IN or STATUS phase (indicates that the byte available on
; the SCSI BUS has a parity error) and also at the start of a MESSAGE OUT
; phase (indicates a legacy parity error from the preceding DATA IN phase).
; Note that the 6260 has a bug with respect to parity error checking:
; SCSIPERR does NOT clear until a byte with good parity is read from the SCSI
; bus.  To work around this flaw, disable parity checking until the next time
; an inbound data byte is read.

CheckParityError	PROC	NEAR

			MOV	DL, LOW SSTAT1
			IN	AL, DX
			TEST	AL, SCSIPERR
			 JZ	@F
			MOV	DL, LOW CLRSINT1
			MOV	AL, CLRSCSIPERR
			OUT	DX, AL
			OR	AL, AL
@@:			RET

CheckParityError	ENDP

; Check the occupancy of the SCSI bus and set the ZERO flag according to
; whether or not the SCSI bus is free.

			PUBLIC	Check_Bus

Check_Bus	PROC	NEAR

			PUSH	DX
			MOV	DH, HIGH BasePort
			MOV	DL, LOW SCSISIG
			IN	AL, DX
			MOV	AH, AL
			MOV	DL, LOW SCSIBUS
			IN	AL, DX
			OR	AH, AL
			MOV	DL, LOW SCSISIG
			IN	AL, DX
			OR	AH, AL
			MOV	DL, LOW SCSIBUS
			IN	AL, DX
			OR	AH, AL
			POP	DX
			RET

Check_Bus	ENDP

; Several of the information transfer phases (MESSAGE OUT, STATUS and MESSAGE
; IN) used SCSI automatic PIO.  This procedure configures the 6X60
; appropriately.  It's a space-saving ploy and may be moved in-line if another
; implementor decides in favor of saving execution time.

ConfigureScsiPIO	PROC	NEAR

			MOV	DL, LOW DMACNTRL0
			XOR	AL, AL
			OUT	DX, AL
			MOV	DL, LOW SXFRCTL0
			MOV	AL, CHEN OR SPIOEN
			OUT	DX, AL
			RET

ConfigureScsiPIO	ENDP

; These simple little steps to build a one-byte message in the SCB message out
; control structures and to assert ATN to attract the target's notice were
; needed so often it was getting tiresome.

PrepareMsgOut		PROC	NEAR

			MOV	(SCB PTR [BP]).MsgOutLen, 1
			MOV	(SCB PTR [BP]).MsgOut, AL
			MOV	DL, LOW SCSISIG
			IN	AL, DX
			AND	AL, SCSI_PHASE_MASK
			OR	AL, ATN
			OUT	DX, AL
			RET

PrepareMsgOut		ENDP

Code			ENDS

			END
